//
//  QGBrokenLineView.m
//  VirtualPayment
//
//  Created by 张博 on 2020/3/13.
//  Copyright © 2020 bo.zhang. All rights reserved.
//

#import "QGBrokenLineView.h"
#import "QGBasicUITool.h"
#import "QGCalendarManager.h"
#import "QGBrokenLineBubbleView.h"
#import "QGDefaultView.h"
@interface QGBrokenLineView () <UICollectionViewDelegate, UICollectionViewDataSource>

/// 折线图基础容器
@property (nonatomic, strong) UICollectionView * collectionView;

/// 小气泡
@property (nonatomic, strong) QGBrokenLineBubbleView * bubbleView;

/// 没有数据View
@property (nonatomic, strong) QGDefaultView * defaultView;

/// 折线图图层
@property (nonatomic, strong) CAShapeLayer * linePathLayer;

/// 折线图月模型
@property (nonatomic, strong) QGBrokenLineMonthModel * monthModel;

/// 折线图路线
@property (nonatomic, strong) UIBezierPath * brokenLineBezierPath;

/// 选中的天
@property (nonatomic, strong) QGBrokenLineDayModel * selectDayModel;

/// 默认的小圆点视图数组
@property (nonatomic, strong) NSMutableArray <UIImageView *>* pointImgViewArray;

@end

@implementation QGBrokenLineView

- (instancetype)initWithFrame:(CGRect)frame
{
    if (self = [super initWithFrame:frame]) {
        
        UICollectionViewFlowLayout * layout = [[UICollectionViewFlowLayout alloc] init];
        layout.sectionInset = UIEdgeInsetsMake(0, BrokenLineSubViewWidth, 0, BrokenLineSubViewWidth);
        layout.itemSize = CGSizeMake(BrokenLineSubViewWidth, BrokenLineViewHeigh);
        layout.minimumLineSpacing = 0;
        layout.minimumInteritemSpacing = 0;
        layout.scrollDirection = UICollectionViewScrollDirectionHorizontal;
        self.collectionView = [[UICollectionView alloc] initWithFrame:self.bounds collectionViewLayout:layout];
        self.collectionView.delegate = self;
        self.collectionView.dataSource = self;
        self.collectionView.backgroundColor = [UIColor clearColor];
        self.collectionView.showsVerticalScrollIndicator = NO;
        self.collectionView.showsHorizontalScrollIndicator = NO;
        [self addSubview:self.collectionView];
        
        [self.collectionView registerClass:[QGBrokenLineSubView class] forCellWithReuseIdentifier:NSStringFromClass([QGBrokenLineSubView class])];
        
        //折线图
        self.linePathLayer = [CAShapeLayer layer];
        self.linePathLayer.strokeColor = [UIColor yyMainThemeColor].CGColor;
        self.linePathLayer.fillColor = nil;
        self.linePathLayer.lineWidth = 1;
        self.linePathLayer.lineJoin = kCALineJoinBevel;
        self.linePathLayer.lineCap = kCALineCapRound;
        [self.collectionView.layer addSublayer:self.linePathLayer];
        
        self.pointImgViewArray = [NSMutableArray arrayWithCapacity:0];
        
        //没有数据的视图
        self.defaultView = [QGDefaultView defaultViewShowWithSuperView:self inset:UIEdgeInsetsMake(60, 0, 0, 0) reloadBlock:nil];
        self.defaultView.reloadButton.hidden = YES;
        [self.defaultView setIconImageWithImageName:@"cashback_bubble_noInfo" messageTitle:@"暂无返现，快去赚钱吧~" buttonTitle:nil];
    }
    return self;
}




/// 设置选择的月份、数据源、今天的日期
/// @param selectMonth 当前选择的月份
/// @param returnCashList 折线图数据源
/// @param today 今天的日期
- (void)setSelectMonth:(NSDate *)selectMonth returnCashList:(NSArray *)returnCashList today:(NSDate *)today
{
    //如果没数据，展示空页面
    if (returnCashList.count == 0) {
        self.defaultView.hidden = NO;
        self.collectionView.hidden = YES;
        return;
    }
    self.defaultView.hidden = YES;
    self.collectionView.hidden = NO;
    
    [self resetDataSourceWithSelectMonth:selectMonth returnCashList:returnCashList today:today];
    //设置折线图路径
    self.linePathLayer.path = self.brokenLineBezierPath.CGPath;
    
    //从视图上移除，然后清理数组
    [self.pointImgViewArray makeObjectsPerformSelector:@selector(removeFromSuperview)];
    [self.pointImgViewArray removeAllObjects];
    
    //添加默认的点
    for (QGBrokenLineDayModel * dayModel in self.monthModel.dayArray) {
        UIImageView * pointImg = [UIImageView creatImageViewWithSuperView:self.collectionView imageName:@"cashback_point_normal"];
        pointImg.frame = CGRectMake(0, 0, 6, 6);
        pointImg.center = dayModel.point;
        [self.pointImgViewArray addObject:pointImg];
        //绘制完当天的就结束
        if (dayModel.isToday) {
            break;
        }
    }
    //气泡需要放在最上面
    [self.collectionView bringSubviewToFront:self.bubbleView];
    //根据选中的最后一天滑动collectionView并且弹出气泡
    CGFloat contentOffsetX = self.selectDayModel.point.x - 4.5 * BrokenLineSubViewWidth;
    contentOffsetX = MAX(0, contentOffsetX);
    [self.collectionView setContentOffset:CGPointMake(contentOffsetX, 0)];    
}

/// 重置数据源
- (void)resetDataSourceWithSelectMonth:(NSDate *)selectMonth returnCashList:(NSArray *)returnCashList today:(NSDate *)today
{
    //排序获取最大金额的对象
    //最大返现金额对象
    QGCashbackBrokenLineModel * maxAmountModel;
    for (QGCashbackBrokenLineModel * brokenLineModel in returnCashList) {
        //因为数据是从前往后排练的，需要取最后一个最大的，所以是大于等于
        if (brokenLineModel.returnCashAmount.floatValue >= maxAmountModel.returnCashAmount.floatValue) {
            maxAmountModel = brokenLineModel;
        }
    }
    
    //获取当前月份所有的天数
    QGCalendarManager * calendar = [QGCalendarManager shareInstance];
    QGBrokenLineMonthModel * monthModel = [calendar getBrokenLineMonthDataSourceWithDate:selectMonth today:today];
    
    //折线图的贝塞尔曲线
    self.brokenLineBezierPath = [UIBezierPath bezierPath];
    
    //选中最后一个被匹配的模型
    QGBrokenLineDayModel * lastMatcheDayModel;
    //记录是否是未来的日子，如果日，后面则不再添加点数据
    BOOL isFutureTime = NO;
    //遍历当月的天数，初始化数据
    for (NSInteger i = 0; i < monthModel.dayArray.count; i++) {
        QGBrokenLineDayModel * dayModel = monthModel.dayArray[i];
    
        //匹配数据
        NSPredicate *predicate = [NSPredicate predicateWithFormat:[NSString stringWithFormat:@"monthAndDayTime MATCHES '%@'",dayModel.monthAndDay]];
        NSArray *resultArray = [returnCashList filteredArrayUsingPredicate:predicate];
        
        CGFloat pointY = BrokenLineContentHeight;
        if (resultArray.count > 0) {
            //数据匹配成功，开始计算点的位置
            QGCashbackBrokenLineModel * brokenLineModel = resultArray.firstObject;
            dayModel.brokenLineModel = brokenLineModel;
            //获取高度的百分比
            CGFloat heightPercent = brokenLineModel.returnCashAmount.floatValue/maxAmountModel.returnCashAmount.floatValue;
            //获取实际高度   （折线图内容高度 - 气泡的高度） * 高度百分比
            CGFloat height = (BrokenLineContentHeight - BrokenLineBubbleHeight) * heightPercent;
            pointY = BrokenLineContentHeight - height;
            
            //记录最后一个，只要有新的出现就会被替换，直至最后一个
            lastMatcheDayModel = dayModel;
        }
        
        //只能记录过去的数据，未来的数据过滤掉
        if (!isFutureTime) {
            CGFloat pointX = BrokenLineSubViewWidth * (i + 1.5);
            //获得点位
            dayModel.point = CGPointMake(pointX, pointY);
            
            //获取折现图
            if (i == 0) {
                [self.brokenLineBezierPath moveToPoint:dayModel.point];
            }
            else
            {
                [self.brokenLineBezierPath addLineToPoint:dayModel.point];
            }
        }
        
        //经过了今天的日期就是未来
        if (dayModel.isToday) {
            isFutureTime = YES;
        }
    }
    [self.brokenLineBezierPath stroke];
    self.monthModel = monthModel;
    self.selectDayModel = lastMatcheDayModel;
    [self.collectionView reloadData];
}


#pragma mark - UICollectionViewDataSource
- (NSInteger)collectionView:(UICollectionView *)collectionView numberOfItemsInSection:(NSInteger)section
{
    return self.monthModel.dayArray.count;
}

- (__kindof UICollectionViewCell *)collectionView:(UICollectionView *)collectionView cellForItemAtIndexPath:(NSIndexPath *)indexPath
{
    QGBrokenLineSubView * cell = (QGBrokenLineSubView *)[collectionView dequeueReusableCellWithReuseIdentifier:NSStringFromClass([QGBrokenLineSubView class]) forIndexPath:indexPath];
    cell.dayModel = self.monthModel.dayArray[indexPath.row];
    
    if (indexPath.row == self.monthModel.dayArray.count - 1) {
        cell.rightLineView.hidden = NO;
    }
    else
    {
        cell.rightLineView.hidden = YES;
    }
    
    return cell;
}

- (void)collectionView:(UICollectionView *)collectionView didSelectItemAtIndexPath:(NSIndexPath *)indexPath
{
    QGBrokenLineDayModel * dayModel = self.monthModel.dayArray[indexPath.row];
    //如果没有点位，则不可选
    if (CGPointEqualToPoint(dayModel.point, CGPointZero)) {
        return;
    }
    self.selectDayModel = dayModel;
    [self.collectionView reloadData];
}

#pragma mark - Setter
- (void)setSelectDayModel:(QGBrokenLineDayModel *)selectDayModel
{
    if (_selectDayModel != selectDayModel) {
        _selectDayModel.isSelect = NO;
        _selectDayModel = selectDayModel;
        _selectDayModel.isSelect = YES;
        
        //设置气泡的位置
        self.bubbleView.centerX = _selectDayModel.point.x;
        self.bubbleView.bottom = _selectDayModel.point.y;
        self.bubbleView.dayModel = _selectDayModel;
        
        
    }
}

#pragma mark - Getter
- (QGBrokenLineBubbleView *)bubbleView
{
    if (!_bubbleView) {
        _bubbleView = [[QGBrokenLineBubbleView alloc] initWithFrame:CGRectMake(0, 0, 100, BrokenLineBubbleHeight)];
        [_bubbleView addTarget:self action:@selector(bubbleViewClick:) forControlEvents:UIControlEventTouchUpInside];
        [self.collectionView addSubview:_bubbleView];
    }
    return _bubbleView;
}

#pragma mark - Action
/// 气泡点击事件
- (void)bubbleViewClick:(QGBrokenLineBubbleView *)bubbleView
{
    if ([NSString isEmpty:bubbleView.dayModel.brokenLineModel.returnCashUrl]) {
        return;
    }
    if (self.bubbleClickBlock) {
        self.bubbleClickBlock(bubbleView.dayModel.brokenLineModel);
    }
    
}

@end




@interface QGBrokenLineSubView ()

/// 底部日期展示
@property (nonatomic, strong) UILabel * dayLabel;

/// 选中的背景图片
@property (nonatomic, strong) UIView * selectBgView;

@end

@implementation QGBrokenLineSubView

- (instancetype)initWithFrame:(CGRect)frame
{
    if (self = [super initWithFrame:frame]) {
        
        //选中背景
        self.selectBgView = [UIView creatViewWithSuperView:self.contentView backgroundColor:[UIColor moduleSeparatorColorF7F7]];
        [self.selectBgView mas_makeConstraints:^(MASConstraintMaker *make) {
            make.left.right.inset(0.5);
            make.top.inset(0);
            make.height.mas_equalTo(BrokenLineContentHeight);
        }];
        
        //左侧线
        UIView * leftLineView = [UIView creatViewWithSuperView:self.contentView backgroundColor:[UIColor lineSeparatorColorDCDC]];
        [leftLineView mas_makeConstraints:^(MASConstraintMaker *make) {
            make.left.inset(0);
            make.width.mas_equalTo(0.5);
            make.top.bottom.mas_equalTo(self.selectBgView);
        }];
        
        self.rightLineView = [UIView creatViewWithSuperView:self.contentView backgroundColor:[UIColor lineSeparatorColorDCDC]];
        [self.rightLineView mas_makeConstraints:^(MASConstraintMaker *make) {
            make.right.inset(0);
            make.width.mas_equalTo(0.5);
            make.top.bottom.mas_equalTo(self.selectBgView);
        }];
        
        //日期
        self.dayLabel = [UILabel creatLabelWithSuperView:self.contentView title:@"11-11" titleFontSize:[UIFont qg_lightFontOfSize:12] textColor:[UIColor textColor9999] alignment:NSTextAlignmentCenter];
        [self.dayLabel mas_makeConstraints:^(MASConstraintMaker *make) {
            make.left.right.inset(0);
            make.height.mas_equalTo(17);
            make.top.mas_equalTo(self.selectBgView.mas_bottom).offset(5);
        }];
        
        
        
    }
    return self;
}

- (void)setDayModel:(QGBrokenLineDayModel *)dayModel
{
    if (_dayModel != dayModel) {
        _dayModel = dayModel;
        self.dayLabel.text = dayModel.monthAndDay;
    }
    
    if (_dayModel.isSelect) {
       self.selectBgView.hidden = NO;
       self.dayLabel.font = [UIFont qg_mediumFontOfSize:12];
    }
    else
    {
       self.selectBgView.hidden = YES;
       self.dayLabel.font = [UIFont qg_lightFontOfSize:12];
    }
}




@end


