//
//  QGActivityPopManager.m
//  VirtualPayment
//
//  Created by silicn on 2019/12/10.
//  Copyright © 2019 bo.zhang. All rights reserved.
//

#import "QGActivityPopManager.h"

#import "QGTalosNetwork.h"
#import "QGCardRequestApi.h"

#import "QGActivityPopView.h"
#import "QGFloatsPopView.h"

#import "QGActivityPopModel.h"
#import "QGCacheManager.h"
#import "QGDateManager.h"

#import "QGAppCoreModelClickEvent.h"
#import "QGAnalyticsService.h"

#import "QGUserInfo.h"
#import "QGGlobalConst.h"

@interface QGActivityPopControlModel : NSObject

/// 弹窗位置
@property (nonatomic, assign) NSInteger position ;
/// 是否需要刷新
@property (nonatomic,assign) BOOL shouldReset;

@end

@implementation QGActivityPopControlModel


@end

@interface QGActivityPopManager ()

/// 所有活动窗口的集合{index:[],index:[]}
@property (nonatomic, strong)NSMutableDictionary *activePops;
/// 用户已经交互过活动窗口id集合
@property (nonatomic, strong)NSMutableDictionary *invalidActivityPops;
/// 等候显示的视图集合[QGActivityPopView] ,一旦用户切换了页面,那么这个页面的弹窗就会暂时进入等待显示集合,
@property (nonatomic, strong)NSMutableArray *waitPops;

/// 显示弹窗的类的集合
@property (nonatomic, strong)NSMutableArray *popIndexs;

///本次获取的已经显示和尚未显示的弹窗集合,便于清除
@property (nonatomic, strong)NSMutableArray *activePopViews;
///本次获取的已经显示和尚未显示的浮窗集合,便于清除
@property (nonatomic, strong)NSMutableArray *floatsPopViews;

/// 所有浮窗视图集合
@property (nonatomic, strong)NSMutableDictionary *floatsPops;
/// 用户已经交互过浮窗id集合
@property (nonatomic, strong)NSMutableDictionary *invalidFloatPops;

/// viewDidAppear出现时的锁
@property (nonatomic, strong) NSLock *lock;
/// 第一次配置时出现时的锁
@property (nonatomic, strong) NSLock *configLock;

@end

/// 弹窗集合标识key
static NSString * activities_pop_sets = @"activities_pop_sets";
/// 浮窗集合标识key
static NSString * floats_pop_sets = @"floats_pop_sets";


@implementation QGActivityPopManager

+ (instancetype)shareInstance
{
    static QGActivityPopManager *manager = nil;
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        manager = [[self alloc] init];
    });
    return manager;
}

- (id)init
{
    self = [super init];
    if (self) {
        /// 初始化弹窗存储空间
        self.activePops  = [NSMutableDictionary dictionary];
        QGCacheManager *cache = [QGCacheManager shareCacheManager];
        NSDictionary *invalidActivities =  [cache appConfigValueForKey:activities_pop_sets];
        /// 无效的弹窗集合
        self.invalidActivityPops = [NSMutableDictionary dictionaryWithDictionary:invalidActivities];
        NSDictionary *invalidFloats = [cache appConfigValueForKey:floats_pop_sets];
        /// 无效的浮窗集合
        self.invalidFloatPops = [NSMutableDictionary dictionaryWithDictionary:invalidFloats];
        /// 等待显示的弹窗集合
        self.waitPops = [NSMutableArray array];
        self.floatsPops = [NSMutableDictionary dictionary];
        
        self.popIndexs = [NSMutableArray arrayWithCapacity:1];
        
        self.floatsPopViews =[NSMutableArray array];
        self.activePopViews = [NSMutableArray array];
        
        [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(removeAllPopViews) name:ModelCenterUserLogOutNotification object:nil];
        [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(resetAllPopViews) name:ModelCenterUserLoginNotification object:nil];
    }
    return self;
}


/// 移除当前所有的弹窗
- (void)removeAllPopViews
{
    /// 移除当前所有的弹窗视图
    if (self.activePopViews.count) {
        [self.activePopViews enumerateObjectsWithOptions:NSEnumerationReverse usingBlock:^(UIView *  _Nonnull obj, NSUInteger idx, BOOL * _Nonnull stop) {
            [obj removeFromSuperview];
        }];
        [self.activePopViews removeAllObjects];
    }
    /// 移除所有浮窗视图
    if (self.floatsPopViews.count) {
        [self.floatsPopViews enumerateObjectsWithOptions:NSEnumerationReverse usingBlock:^(UIView *  _Nonnull obj, NSUInteger idx, BOOL * _Nonnull stop) {
            [obj removeFromSuperview];
        }];
        [self.floatsPopViews removeAllObjects];
    }
    
    [self.activePops removeAllObjects];
    
    /// 清空本地的标记
     [self cleanLocalUniques];
     
     for (QGActivityPopControlModel *model in self.popIndexs) {
         model.shouldReset = YES;
     }
}

/// 重新设置所有的弹窗
- (void)resetAllPopViews
{
    [self removeAllPopViews];
}

#pragma mark - 网络模块

/// 检查当前记录是否包含本次弹窗位置
/// 如果不包含记录本次弹窗的位置,以便于下次刷新
- (QGActivityPopControlModel *)checkPopHadContainsWithIndex:(NSInteger)index
{
    __block QGActivityPopControlModel *model = nil;
    [self.popIndexs enumerateObjectsUsingBlock:^(QGActivityPopControlModel *  _Nonnull obj, NSUInteger idx, BOOL * _Nonnull stop) {
        if (obj.position == index) {
            model = obj;
            *stop = YES;
        }
    }];
    if (!model) {
        model = [[QGActivityPopControlModel alloc]init];
        model.position = index;
        [self.popIndexs addObject:model];
    }
    return model;
}

/// 内部方法,获取指定位置的活动弹窗
/// @param index 位置
- (void)requestActivitiesWithIndex:(NSInteger)index complete:(void(^)(NSArray *activities,NSArray *floats))complete
{
    /// 标记本次弹窗请求位置
    QGActivityPopControlModel *model = [self checkPopHadContainsWithIndex:index];
    /// 请求数据开始
    QGTalosNetwork *networking = [QGTalosNetwork shareManager];
    /// 参数传递
    NSMutableDictionary *param = [NSMutableDictionary dictionary];
    [param setObject:@(index) forKey:@"tabName"];
    
//    NSString * urlString = @"http://yapi.quantgroups.com/mock/297/vcc/app/config/windows";
    NSString * urlString = [QGCardRequestApi urlWithIndex:QGCardRequestServerMethodActivity];
    [networking GET:urlString parameters:param success:^(id  _Nonnull responseObject) {
        /// 请求成功
        if ([QGTalosNetwork checkBusinessCodeAndCodeWithResponseObjectObject:responseObject]) {
            ///----> 弹窗集合逻辑
            NSArray *popups = responseObject[@"data"][@"popups"];  // 源数据
            NSArray *activityModels = [self handleActivitiesWithIndex:index activities:popups]; // 处理数据
            
            ///-----> 浮窗集合逻辑
            NSArray *floats = responseObject[@"data"][@"floats"];  // 源数据
            NSArray *floatsModels = [self handleFloatsWithIndex:index floats:floats];  // 处理数据
            
            if (complete) {
                complete(activityModels,floatsModels);
            }
            /// 弹窗不会再次刷新
            model.shouldReset = NO;
        }
    } failure:^(NSURLSessionDataTask * _Nonnull task, NSError * _Nonnull error) {
        /// 如果失败,不提示,不做处理
    }];
}

#pragma mark - 弹窗和浮窗显示逻辑

/// 配置当前要显示的弹窗,默认是集合[0]的对象
/// @param index 弹窗的位置
/// @param controller 显示的父视图
- (void)configsActivitieViewsWithIndex:(NSInteger)index controller:(UIViewController *)controller
{
    /// 第一次显示防止多线程同时访问,加锁
    [self.configLock lock];
    NSArray *list = [self.activePops objectForKey:[NSString stringWithFormat:@"%ld",(long)index]];
    if (list.count == 0){
        [self.configLock unlock];
        return;
    };
    @weakify(self);
    dispatch_async(dispatch_get_main_queue(), ^{
        /// 获取最上面的弹窗
        for (QGActivityPopModel *model in list) {
            /// 初始化弹窗系统
            QGActivityPopView *pop = [[QGActivityPopView alloc] initWithPopModel:model];
            pop.position = index;
            pop.controller = controller;
            [self.activePopViews addObject:pop];
            /// 加入待展示
            [self.waitPops addObject:pop];
            /// 点击事件
            pop.activityDismissBlock = ^(NSInteger position,QGActivityPopModel *popModel, UIViewController *aController,BOOL isClose) {
               @strongify(self);
                /// 点击关闭后不再弹出( popModel.showType == 2)
                /// 点击内容后不再弹出(!isClose && popModel.showType == 1)
                if ((popModel.showType == 2) || (!isClose && popModel.showType == 1)) {
                    /// 把这个id标记为显示过,时间是今天的日期
                   NSString *today = DateManager().defaultFormatterForTimestamp([[NSDate date] timeIntervalSince1970]);
                   [self.invalidActivityPops setObject:today forKey:[NSString stringWithFormat:@"%@",popModel.uniqueId]];
                    ///重新储存
                    [[QGCacheManager shareCacheManager] setAPPConfigValue:self.invalidActivityPops forKey:activities_pop_sets];
                }
               [self configNextActivityWithIndex:position controller:aController];
            };
        }
        /// 立刻展示第一个
        [self showActiesViewsWithIndex:index inController:controller];
    });
    /// 显示完成解锁
     [self.configLock unlock];
    
    
}

- (BOOL)canShowStatusWithController:(UIViewController *)controller
{
    /// 父视图为空
    if (controller.view == nil || controller.view.window == nil) {
        return NO;
    }
    /// 当前页面已经被push
    if (controller.navigationController) {
        if (controller.navigationController.topViewController != controller) {
            return NO;
        }
    }
    return YES;
}


- (void)cleanLocalUniques
{
    [[QGCacheManager shareCacheManager] removeAppConfigWithKeys:@[floats_pop_sets,activities_pop_sets]];
}

/// 显示立即标记,即showType = 3时
/// @param model 类型
- (void)saveUniqueIdWithModel:(QGActivityPopModel *)model
{
    /// 标记today已经显示,显示就关闭 showType = 3
       NSString *today = DateManager().defaultFormatterForTimestamp([[NSDate date] timeIntervalSince1970]);
       [self.invalidActivityPops setObject:today forKey:[NSString stringWithFormat:@"%@",model.uniqueId]];
       ///重新储存
       [[QGCacheManager shareCacheManager] setAPPConfigValue:self.invalidActivityPops forKey:activities_pop_sets];
}

/// 获取弹窗页面的名称
/// @param index 位置
- (NSString *)pageNameWithIndex:(NSInteger)index
{
    if (index == QGActivityPositionHome) {
        return @"弹窗-首页";
    }
    if (index == QGActivityPositionManager) {
        return @"弹窗-管理页";
    }
    if (index == QGActivityPositionMine) {
        return @"弹窗-我的";
    }
    if (index == QGActivityPositionMall) {
        return @"弹窗-省钱";
    }
    
    if (index == QGActivityPositionCashback) {
        return @"弹窗-赚钱";
    }
    
    return nil;
}

/// 配置下一个活动窗口的活动,清除并标记上一个已经显示的弹窗
/// @param index 活动窗口的显示位置
/// @param controller 窗口的父视图
- (void)configNextActivityWithIndex:(NSInteger)index controller:(UIViewController *)controller
{
    dispatch_async(dispatch_get_global_queue(0, 0), ^{
        /// 移除当前显示的model
          NSArray *list = [self.activePops objectForKey:[NSString stringWithFormat:@"%ld",(long)index]];
          if (list.count == 0) return ;
          NSMutableArray *muList = [NSMutableArray arrayWithArray:list];
          [muList removeObjectAtIndex:0];
          /// 重置position所在位置的活动窗口集合
          [self.activePops setObject:muList forKey:[NSString stringWithFormat:@"%ld",(long)index]];
          sleep(2);
          /// 重新配置新的活动窗口
          [self showActiesViewsWithIndex:index inController:controller];
    });
  
}

/// 配置浮窗
/// @param index 浮窗的显示位置
/// @param controller 窗口的父视图
- (void)configFloatWithIndex:(NSInteger)index controller:(UIViewController *)controller
{
     NSArray *list = [self.floatsPops objectForKey:[NSString stringWithFormat:@"%ld",(long)index]];
    if (list.count == 0) return;
    @weakify(self);
    dispatch_async(dispatch_get_main_queue(), ^{
        @strongify(self);
        /// 判断视图是空的情况
        if (!controller || controller.view == nil) return ;
        /// 初始化浮窗
        QGActivityPopModel *model = [list firstObject];
        QGFloatsPopView *pop = [[QGFloatsPopView alloc]initWithModel:model];
        [self.floatsPopViews addObject:pop];
        /// block触发必定是只显示一次的
        pop.dismissBlock = ^(QGActivityPopModel *model) {
             NSString *today = DateManager().defaultFormatterForTimestamp([[NSDate date] timeIntervalSince1970]);
            [self.invalidFloatPops setObject:today forKey:[NSString stringWithFormat:@"%@",model.uniqueId]];
            ///重新储存
            [[QGCacheManager shareCacheManager] setAPPConfigValue:self.invalidFloatPops forKey:floats_pop_sets];
        };
        [pop showInController:controller];
        /// 统计
        QGAppCoreModelClickEvent *event = [QGAppCoreModelClickEvent acmcEventWithPage:[self floatsPageNameWithIndex:index] path:nil name:nil detail_name:@"显示浮窗"];
        [QGAnalyticsService analyticsTrackWithEvent:event];
        
    });
    
}


/// 获取浮窗页面的名称
/// @param index 位置
- (NSString *)floatsPageNameWithIndex:(NSInteger)index
{
    if (index == QGActivityPositionHome) {
        return @"浮窗-首页";
    }
    if (index == QGActivityPositionManager) {
        return @"浮窗-管理页";
    }
    if (index == QGActivityPositionMine) {
        return @"浮窗-我的";
    }
    
    return nil;
}

#pragma mark - 弹窗和浮窗数据过滤

/// 对弹窗源数据进行数据过滤,并限制最大为2条
- (NSArray *)handleActivitiesWithIndex:(NSInteger)index activities:(NSArray *)popups
{
    NSMutableArray *activityModels = [NSMutableArray array];   // 处理数据
    for (NSDictionary *dic in popups) {
        QGActivityPopModel *model = [QGActivityPopModel yy_modelWithJSON:dic];
        
        // 点击或者关闭后不再显示,需要标记并且忽略日期
        if (model.showType == 1 || model.showType == 2) {
            // 如果没有被标记过
            if (![self.invalidActivityPops valueForKey:[NSString stringWithFormat:@"%@",model.uniqueId]]){
                [activityModels addObject:model];
            }
        }else if (model.showType == 3) { // 每天仅弹出一次,需要标记但是需要比对日期
            NSString *flagDate = [self.invalidActivityPops valueForKey:[NSString stringWithFormat:@"%@",model.uniqueId]];
            NSString *today = DateManager().defaultFormatterForTimestamp([[NSDate date] timeIntervalSince1970]);
            // 今天已经标记过,忽略本条数据
            if (flagDate && [flagDate isEqualToString:today]) {continue;}
            [activityModels addObject:model];
        }else{
            [activityModels addObject:model];
        }
        if (activityModels.count >= 2) break;
    }
    /// 放入弹窗集合里面
    [self.activePops setObject:activityModels forKey:[NSString stringWithFormat:@"%ld",(long)index]];
    return activityModels;
}
/// 对浮窗源数据进行数据过滤
- (NSArray *)handleFloatsWithIndex:(NSInteger )index floats:(NSArray *)floats
{
    NSMutableArray *floatsModels = [NSMutableArray array]; // 处理数据
    for (NSDictionary *dic in floats) {
        QGActivityPopModel *model = [QGActivityPopModel yy_modelWithJSON:dic];
        if (model.showType == 1) { // 点击后不再显示
            if (![self.invalidFloatPops valueForKey:[NSString stringWithFormat:@"%@",model.uniqueId]]) {
                [floatsModels addObject:model];
                break;   // 浮窗只能有一个
            }
        }else if (model.showType == 2){
             [floatsModels addObject:model];
            break;   // 浮窗只能有一个
        }
    }
    
    [self.floatsPops setValue:floatsModels forKey:[NSString stringWithFormat:@"%ld",(long)index]];
    return floatsModels;
}

/// 是否需要请求数据
- (BOOL)ifNeedRequestWithIndex:(NSInteger)index
{
    /// 默认不存在就是需要请求
    __block BOOL shouldReset = YES;
    [self.popIndexs enumerateObjectsUsingBlock:^(QGActivityPopControlModel *  _Nonnull model, NSUInteger idx, BOOL * _Nonnull stop) {
        if (model.position == index) {
            shouldReset = model.shouldReset;
            *stop = YES;
        }
    }];
    return shouldReset;
}

#pragma mark - 查询并展示页面弹窗

/// 查询当前页面的视图展示
- (void)viewDidAppearAtPosition:(QGActivityPosition)position inController:(UIViewController *)controller
{
    
    if (position == 0) return;
    if (controller.view == nil) return;
    /// 加载
    if ([self ifNeedRequestWithIndex:position]) {
         @weakify(self);
        /// 网络获取该位置的活动弹窗
        [self requestActivitiesWithIndex:position complete:^(NSArray *activities,NSArray *floats){
            @strongify(self);
            /// 配置弹窗
            [self configsActivitieViewsWithIndex:position controller:controller];
            /// 配置浮窗
            [self configFloatWithIndex:position controller:controller];
        }];
    }else{
        ///展示
        [self showActiesViewsWithIndex:position inController:controller];
    }
}

/// 展示弹窗
- (void)showActiesViewsWithIndex:(QGActivityPosition)position inController:(UIViewController *)controller
{
    /// 没有等待的窗口
    if (self.waitPops.count == 0) return;
    dispatch_async(dispatch_get_global_queue(0, 0), ^{

        __block QGActivityPopView *popView = nil;
        [self.waitPops enumerateObjectsWithOptions:NSEnumerationConcurrent usingBlock:^(QGActivityPopView *  _Nonnull pop, NSUInteger idx, BOOL * _Nonnull stop) {
            if (pop.controller == controller && pop.position == position) {
                popView = pop;
                *stop =  YES;
            }
        }];
        if (popView) {
            [self showWaitPopWith:popView];
        }
    });
}

/// 展示等待的视图
- (void)showWaitPopWith:(QGActivityPopView *)pop
{
    dispatch_async(dispatch_get_main_queue(), ^{
        if ([self canShowStatusWithController:pop.controller]) {
            [pop showInView:pop.controller.view];
              /// 显示即标记
              if (pop.popModel.showType == 3) {
                  [self saveUniqueIdWithModel:pop.popModel];
              }
              /// 一旦展示就清除
              [self.waitPops removeObject:pop];
            QGAppCoreModelClickEvent *event = [QGAppCoreModelClickEvent acmcEventWithPage:[self pageNameWithIndex:pop.position] path:nil name:nil detail_name:@"显示弹窗"];
            [QGAnalyticsService analyticsTrackWithEvent:event];
        }
    });
}


- (NSLock *)lock
{
    if (!_lock) {
        _lock = [[NSLock alloc]init];
    }
    return _lock;
}

- (NSLock *)configLock
{
    if (!_configLock) {
        _configLock = [[NSLock alloc]init];
    }
    return _configLock;
}


- (void)dealloc
{
    [[NSNotificationCenter defaultCenter] removeObserver:self];
}

@end
