//
//  QGTencentTool.m
//  CreditWallet
//
//  Created by harry on 07/11/2017.
//  Copyright © 2017 QuantGroup. All rights reserved.
//

#import "QGTencentTool.h"
#import "SDWebImageDownloader.h"
#import "UIImage+CreditWallet.h"
#import "QGHUDManager.h"
#import <TencentOpenAPI/QQApiInterface.h>
#import <TencentOpenAPI/QQApiInterfaceObject.h>
#import <TencentOpenAPI/TencentOAuth.h>
#import <TencentOpenAPI/sdkdef.h>

#import "QGAppMacro.h"
@interface QGTencentTool () <TencentSessionDelegate, QQApiInterfaceDelegate>

@property (strong, nonatomic) TencentOAuth *tencentOAuth;
@end

@implementation QGTencentTool
static QGTencentTool *_tencentTool = nil;

+ (instancetype)share {
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        if (!_tencentTool) {
            _tencentTool = [[QGTencentTool alloc] init];
            [_tencentTool qqInit];
        }
    });
    return _tencentTool;
}

/**
 *  QQ初始化
 */
- (void)qqInit {
    self.tencentOAuth = [[TencentOAuth alloc] initWithAppId:kTencentAppID andDelegate:self];
}

/**
 *  QQ是否安装
 *
 *  @return BOOL结果
 */
- (BOOL)isQQInstalled {
    return [QQApiInterface isQQInstalled];
}

/**
 *  QQ分享
 *
 *  @param dic   字典内包含的key @[imgUrl,title,desc,link]
 */
- (void)QQShareWithDic:(NSDictionary *)dic shareType:(QGTencentToolType)shareType {
    if (![self isQQInstalled]) {
        [QGHUDManager showHud:@"未安装QQ客户端" afterDelay:1.5];
        return;
    }

    NSString * urlString = dic[@"imgUrl"];
    UIImage * image = dic[@"image"];
    if (urlString && [urlString isKindOfClass:[NSString class]]) {
        __weak typeof(self) weakSelf = self;
        [self downloaderImageObjectWithUrl:urlString successBlock:^(NSData *data, UIImage *image) {
            [weakSelf sendToQQWithImage:data dic:dic shareType:shareType];
        }];
    }
    else if (image && [image isKindOfClass:[UIImage class]])
    {
         [self sendToQQWithImage:UIImagePNGRepresentation(image) dic:dic shareType:shareType];
    }
    else {
        [QGHUDManager showHud:@"分享失败"];
    }
}

/// 分享图片
/// @param dic   字典内包含的key @[imgUrl,title,desc,link]
- (void)QQShareImageWithDic:(NSDictionary *)dic shareType:(QGTencentToolType)shareType
{
    if (![self isQQInstalled]) {
        [QGHUDManager showHud:@"未安装QQ客户端" afterDelay:1.5];
        return;
    }
    NSString * urlString = dic[@"imgUrl"];
    UIImage * image = dic[@"image"];
    if (urlString && [urlString isKindOfClass:[NSString class]]) {
        __weak typeof(self) weakSelf = self;
        [self downloaderImageObjectWithUrl:urlString successBlock:^(NSData *data, UIImage *image) {
            [weakSelf sendToQQImageObjcWithImage:data dic:dic shareType:shareType];
        }];
    }
    else if (image && [image isKindOfClass:[UIImage class]])
    {
         [self sendToQQImageObjcWithImage:UIImageJPEGRepresentation(image, 0.5) dic:dic shareType:shareType];
    }
    else {
        [QGHUDManager showHud:@"分享失败"];
    }
}



/// 下载图片数据，并且不做任何处理
- (void)downloaderImageObjectWithUrl:(NSString *)urlString successBlock:(void(^)(NSData * data, UIImage * image))successBlock
{
    
    NSURL *imageUrl = [NSURL URLWithString:urlString];
    [[SDWebImageDownloader sharedDownloader] downloadImageWithURL:imageUrl options:SDWebImageDownloaderLowPriority progress:nil completed:^(UIImage * _Nullable image, NSData * _Nullable data, NSError * _Nullable error, BOOL finished) {
        if (data) {
            if (successBlock) {
                successBlock(data,image);
            }
        } else {
            dispatch_async(dispatch_get_main_queue(), ^{
                [QGHUDManager showHud:@"分享失败"];
            });
        }
    }];
}

/// 分析图片信息
- (void)sendToQQImageObjcWithImage:(NSData *)imageData dic:(NSDictionary *)dic shareType:(QGTencentToolType)shareType {
    __weak __typeof(self) weakSelf = self;
    dispatch_async(dispatch_get_main_queue(), ^{
        NSString *title = dic[@"title"];
        NSString *desc = dic[@"desc"];
        NSInteger count = 512;
        if (title.length > 128) {
            title = [title substringToIndex:127];
        }
        if (desc.length > count) {
            desc = [desc substringToIndex:count - 1];
        }

        __strong __typeof(weakSelf) strongSelf = weakSelf;
        

        QQApiSendResultCode sent = 0;
        switch (shareType) {
            case QGTencentToolTypeQQ: {
                QQApiImageObject * imageObj = [QQApiImageObject objectWithData:imageData previewImageData:imageData title:title ? title: @"" description:desc ? desc: @""];
                SendMessageToQQReq *req = [SendMessageToQQReq reqWithContent:imageObj];
                sent = [QQApiInterface sendReq:req];
                break;
            }
            case QGTencentToolTypeQQZone: {
                
                QQApiImageArrayForQZoneObject * imageObj = [QQApiImageArrayForQZoneObject objectWithimageDataArray:@[imageData] title:title ? title: @"" extMap:nil];
                SendMessageToQQReq *req = [SendMessageToQQReq reqWithContent:imageObj];
                sent = [QQApiInterface SendReqToQZone:req];
                break;
            }

            default:
                break;
        }

        [strongSelf handleSendResult:sent];
    });
}


///分享新闻信息
- (void)sendToQQWithImage:(NSData *)imageData dic:(NSDictionary *)dic shareType:(QGTencentToolType)shareType {
    __weak __typeof(self) weakSelf = self;
    dispatch_async(dispatch_get_main_queue(), ^{
        NSString *title = dic[@"title"];
        NSString *desc = dic[@"desc"];
        NSString *link = dic[@"link"];
        NSInteger count = 512;
        if (title.length > 128) {
            title = [title substringToIndex:127];
        }
        if (desc.length > count) {
            desc = [desc substringToIndex:count - 1];
        }

        __strong __typeof(weakSelf) strongSelf = weakSelf;

        QQApiNewsObject *videoObj = [QQApiNewsObject objectWithURL:[NSURL URLWithString:link ?: @""]
                                                             title:title ? title : @""
                                                       description:desc ? desc : @""
                                                  previewImageData:imageData];
        SendMessageToQQReq *req = [SendMessageToQQReq reqWithContent:videoObj];

        QQApiSendResultCode sent = 0;
        switch (shareType) {
            case QGTencentToolTypeQQ: {
                sent = [QQApiInterface sendReq:req];
                break;
            }
            case QGTencentToolTypeQQZone: {
                sent = [QQApiInterface SendReqToQZone:req];
                break;
            }

            default:
                break;
        }

        [strongSelf handleSendResult:sent];
    });
}

/**
 *  分享文本到QQ
 *
 *  @param text  文本信息
 */
- (void)shareTextToQQWithText:(NSString *)text shareType:(QGTencentToolType)shareType {
    QQApiTextObject *txtObj = [QQApiTextObject objectWithText:text];
    SendMessageToQQReq *req = [SendMessageToQQReq reqWithContent:txtObj];
    //分享到QZone
    QQApiSendResultCode sent = 0;
    switch (shareType) {
        case QGTencentToolTypeQQ:
            sent = [QQApiInterface sendReq:req];
            break;
        case QGTencentToolTypeQQZone:
            sent = [QQApiInterface SendReqToQZone:req];
            break;

        default:
            break;
    }

    [self handleSendResult:sent];
}

#pragma mark - TencentLoginDelegate

- (void)tencentDidLogin {
}

- (void)tencentDidNotLogin:(BOOL)cancelled {
}

- (void)tencentDidNotNetWork {
}

#pragma mark - TencentSessionDelegate

- (void)tencentDidLogout {
}

- (void)responseDidReceived:(APIResponse *)response forMessage:(NSString *)message {
    //    URLREQUEST_SUCCEED = 0, /**< 网络请求成功发送至服务器，并且服务器返回数据格式正确
    //                             * \note 这里包括所请求业务操作失败的情况，例如没有授权等原因导致
    //                             */
    //
    //    URLREQUEST_FAILED = 1,  /**< 网络异常，或服务器返回的数据格式不正确导致无法解析 */
}

#pragma mark - QQApiInterfaceDelegate
- (void)onReq:(QQBaseReq *)req {
    switch (req.type) {
        case EGETMESSAGEFROMQQREQTYPE: {
            break;
        }
        default: {
            break;
        }
    }
}

- (void)onResp:(QQBaseResp *)resp {
    switch (resp.type) {
        case ESENDMESSAGETOQQRESPTYPE: {
            SendMessageToQQResp *sendResp = (SendMessageToQQResp *) resp;
            if ([sendResp.result integerValue] == 0) {
                if (self.qqShareResultBlock) {
                    self.qqShareResultBlock(YES);
                }
                [QGHUDManager showHud:@"分享成功" afterDelay:1];
            } else {
                if (self.qqShareResultBlock) {
                    self.qqShareResultBlock(NO);
                }
                [QGHUDManager showHud:@"分享失败" afterDelay:1];
            }
            break;
        }
        default: {
            break;
        }
    }
}

- (void)isOnlineResponse:(NSDictionary *)response {
}

#pragma mark - QQ handleSendResult
- (void)handleSendResult:(QQApiSendResultCode)sendResult {
    NSString *text = @"";
    switch (sendResult) {
        case EQQAPIAPPNOTREGISTED: {
            text = @"App未注册";
            break;
        }
        case EQQAPIMESSAGECONTENTINVALID:
        case EQQAPIMESSAGECONTENTNULL:
        case EQQAPIMESSAGETYPEINVALID: {
            text = @"发送参数错误";
            break;
        }
        case EQQAPIQQNOTINSTALLED: {
            text = @"未安装手Q";
            break;
        }
        case EQQAPIQQNOTSUPPORTAPI: {
            text = @"API接口不支持";
            break;
        }
        case EQQAPISENDFAILD: {
            text = @"发送失败";
            break;
        }
        case EQQAPIQZONENOTSUPPORTTEXT: {
            text = @"空间分享不支持纯文本分享，请使用图文分享";
            break;
        }
        case EQQAPIQZONENOTSUPPORTIMAGE: {
            text = @"空间分享不支持纯图片分享，请使用图文分享";
            break;
        }
        default: {
            break;
        }
    }
    if (text && text.length > 0) {
        [QGHUDManager showHud:text afterDelay:1];
    }
}

/*! @brief 处理QQ通过URL启动App时传递的数据
 *
 * 需要在 application:openURL:sourceApplication:annotation:或者application:handleOpenURL中调用。
 * @param url QQ启动第三方应用时传递过来的URL
 * @return 成功返回YES，失败返回NO。
 */
- (BOOL)handleOpenURL:(NSURL *)url {
    return [QQApiInterface handleOpenURL:url delegate:self];
}

@end
