//
//  QGLocationManager.m
//  VirtualPayment
//
//  Created by silicn on 2020/4/24.
//  Copyright © 2020 bo.zhang. All rights reserved.
//

#import "QGLocationManager.h"
#import <UIKit/UIKit.h>
#import <CoreLocation/CoreLocation.h>

@interface QGLocationManager ()<CLLocationManagerDelegate>

/// 地理位置信息获取
@property (nonatomic, strong)CLLocationManager *location;
/// 地理位置授权状态
@property (nonatomic, assign) QGLocationStatus status;
/// 授权状态回调
@property (nonatomic, copy) QGLocationStatusBlock requestStatusBlock ;
/// 经纬度信息回调
@property (nonatomic, copy) QGLocationLoLaBlock lolaBlock ;
/// 地址信息回调
@property (nonatomic, copy) QGLocationPlacemarkBlock placemarkBlock ;
/// 地理编译
@property (nonatomic, strong)CLGeocoder *geocoder;


@end


@implementation QGLocationManager

+ (instancetype)shareManager
{
    static QGLocationManager *manager = nil;
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        manager = [[self alloc]init];
    });
    return manager;
}

- (id)init
{
    self = [super init];
    if (self) {
        /// 初始化权限状态
        [self locationAuthorizationStatus];
        /// 部分系统做兼容
        [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(applicationBecomeActive) name:UIApplicationDidBecomeActiveNotification object:nil];
    }
    return self;
}

- (void)applicationBecomeActive
{
    BOOL enable = [CLLocationManager locationServicesEnabled];
    if (enable) {
        CLAuthorizationStatus CLstatus = [CLLocationManager authorizationStatus];
        QGLocationStatus status = [self switchStatusWithCLstatus:CLstatus];
        if (self.status != status) {
            self.status = status;
            if (self.statusBlock) self.statusBlock(self.status);
        }
    }
}

///初始化权限状态
- (void)locationAuthorizationStatus
{
    BOOL enable = [CLLocationManager locationServicesEnabled];
    if (enable) {
        CLAuthorizationStatus CLstatus = [CLLocationManager authorizationStatus];
        QGLocationStatus status = [self switchStatusWithCLstatus:CLstatus];
        self.status = status;
    }else{
        self.status = QGLocationStatusNotAvaiable;
    }
}


/// 系统权限转化成自定义权限
/// @param CLstatus CLstatus
- (QGLocationStatus )switchStatusWithCLstatus:(CLAuthorizationStatus)CLstatus
{
    switch (CLstatus) {
        case kCLAuthorizationStatusNotDetermined:
            return QGLocationStatusNotDetermined;
            break;
        case kCLAuthorizationStatusRestricted:  /// 未知原因
            return QGLocationStatusNotAvaiable;
            break;
        case kCLAuthorizationStatusAuthorizedAlways:
            return QGLocationStatusAuthed;
            break;
        case kCLAuthorizationStatusAuthorizedWhenInUse:
            return QGLocationStatusAuthed;
            break;
        case kCLAuthorizationStatusDenied:
            return QGLocationStatusDenied;
            break;
        default:
            break;
    }
}

/// 请求获取权限
- (void)requestLocationAuthorizationStatus:(_Nullable QGLocationStatusBlock)statusBlock
{
    
    if (self.status == QGLocationStatusNotDetermined) {
        self.requestStatusBlock = statusBlock;
        self.location = [[CLLocationManager alloc]init];
        self.location.delegate = self;
        [self.location requestAlwaysAuthorization];
        [self.location requestWhenInUseAuthorization];
    }else{
        if (statusBlock) {
            statusBlock(self.status);
        }
    }
}

/// 获取地理位置经纬度
- (void)requestLocationinformationLoLaBlock:(QGLocationLoLaBlock)lolaBlock placemarkBlock:(QGLocationPlacemarkBlock)placemarkBlock;
{
    self.lolaBlock = lolaBlock;
    self.placemarkBlock = placemarkBlock;
    
    if (self.status == QGLocationStatusAuthed) {
        if (!self.location) {
            self.location = [[CLLocationManager alloc]init];
            self.location.delegate = self;
            self.location.desiredAccuracy = kCLLocationAccuracyBest;
        }
        [self.location startUpdatingLocation];
    }else if (self.status == QGLocationStatusNotDetermined){
        self.location = [[CLLocationManager alloc]init];
        self.location.delegate = self;
        [self.location requestAlwaysAuthorization];
        [self.location requestWhenInUseAuthorization];
        [self.location startUpdatingLocation];
    }
}

- (void)locationManager:(CLLocationManager *)manager didChangeAuthorizationStatus:(CLAuthorizationStatus)status
{
    NSLog(@"locationManager ---- didChangeAuthorizationStatus");
   self.status = [self switchStatusWithCLstatus:status];
    if (self.statusBlock) {
        self.statusBlock(self.status);
    }
    if (self.requestStatusBlock) {
        self.requestStatusBlock(self.status);
    }
}

- (void)locationManager:(CLLocationManager *)manager
didUpdateLocations:(NSArray<CLLocation *> *)locations
{
    CLLocation *location = [locations lastObject];
    CLLocationCoordinate2D coordinate = location.coordinate;
    /// 终止获取地理位置
    [manager stopUpdatingLocation];
    
    if (self.lolaBlock) {
        self.lolaBlock(coordinate.longitude, coordinate.latitude,nil);
    }
    /// 如果self.placemarkBlock 有值,进行反编译,否则不进行反编译
    if (self.placemarkBlock) {
        [self reverseGeocoderLocation:location];
    }
    
}

/// 反编译地理位置
/// @param location 地理位置
- (void)reverseGeocoderLocation:(CLLocation *)location
{
    [self.geocoder reverseGeocodeLocation:location completionHandler:^(NSArray<CLPlacemark *> * _Nullable placemarks, NSError * _Nullable error) {
        if (error != nil || placemarks.count == 0 ){
            self.placemarkBlock(nil);
        }else{
            self.placemarkBlock(placemarks.lastObject);
        }
    }];
}


/// 地址编译
/// @param address 地址信息
/// @param lolaBlock 经纬度信息
- (void)geocoderAddress:(NSString *)address complete:(QGLocationLoLaBlock)lolaBlock
{
    [self.geocoder geocodeAddressString:address completionHandler:^(NSArray<CLPlacemark *> * _Nullable placemarks, NSError * _Nullable error) {
        /// 出错
        if (error != nil || placemarks.count == 0){
            if (lolaBlock) {
                lolaBlock(-1,-1,error);
            }
        }else{
             CLPlacemark *placemark=[placemarks firstObject];
             CLLocationCoordinate2D coordinate = placemark.location.coordinate;
            if (lolaBlock) {
                lolaBlock(coordinate.longitude,coordinate.latitude,nil);
            }
        }
    }];
}

- (void)reverseGeocoderLongitude:(double)longitude latitude:(double)latitude complete:(QGLocationPlacemarkBlock)placemarkBlock
{
    CLLocation *location = [[CLLocation alloc]initWithLatitude:latitude longitude:longitude];
    [self.geocoder reverseGeocodeLocation:location completionHandler:^(NSArray<CLPlacemark *> * _Nullable placemarks, NSError * _Nullable error) {
        if (error != nil || placemarks.count == 0 ){
            if (placemarkBlock) {
                placemarkBlock(nil);
            }
        }else{
            if (placemarkBlock) {
                placemarkBlock(placemarks.lastObject);
            }
        }
    }];
}



- (CLGeocoder *)geocoder
{
    if (!_geocoder) {
        _geocoder = [[CLGeocoder alloc]init];
    }
    return _geocoder;
}


- (void)dealloc
{
    [[NSNotificationCenter defaultCenter] removeObserver:self];
}

@end
