//
//  QGFaceVerifyManager.m
//  CreditWallet
//
//  Created by silicn on 2019/12/20.
//  Copyright © 2019 QuantGroup. All rights reserved.
//

#import "QGFaceVerifyManager.h"
/// 人脸核身
#import <WBCloudReflectionFaceVerify/WBFaceVerifyCustomerService.h>
#import "QGTalosNetwork.h"
#import "QGHUDManager.h"
#import "QGUserInfo.h"

#import "QGAppMacro.h"
#import "QGAppStatic.h"

#import "QGAnalyticsService.h"


@interface QGFaceVerifyManager ()<WBFaceVerifyCustomerServiceDelegate>

/// 用户id
@property (nonatomic, copy) NSString *userId;
/// 一次性随机串
@property (nonatomic, copy) NSString *nonce;
/// 签名
@property (nonatomic, copy) NSString *sign;
/// 订单号
@property (nonatomic, copy) NSString *orderNo ;
/// 凭证
@property (nonatomic, copy) NSString *ticket ;
/// faceId 人脸核身的时候用到
@property (nonatomic, copy) NSString *faceId ;
/// 当前SDK版本号
@property (nonatomic, copy) NSString *versionNo;
/// 活体识别结果
@property (nonatomic, copy) QGFaceVerifyResultBlock faceVerifyResult;
/// 腾讯的AppId
@property (nonatomic, copy)NSString *faceAppId;
/// 腾讯的LICENSE
@property (nonatomic, copy)NSString *faceLicense;

/// 获取faceId的type参数
@property (nonatomic, copy)NSString *bundleType;

/// 上报结果完成回调
@property (nonatomic, copy) void(^faceVerifyCompleteBlock)(void) ;


@end


@implementation QGFaceVerifyManager

+ (instancetype)shareInstance
{
    static QGFaceVerifyManager *manager = nil;
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        manager = [[self alloc]init];
        
    });
    return manager;
}

- (id)init
{
    self = [super init];
    if (self) {
        self.faceAppId = face_appid();
        self.faceLicense = face_license();
        self.bundleType = bundle_type();
        NSParameterAssert(self.faceAppId != nil);
    }
    return self;
}

#pragma mark - OCR Service


- (void)startOCRServiceWithType:(QGOCRIDCardType)type
                   ocrAuthToken:(nullable NSString *)ocrAuthToken
                        success:(void(^)(WBIDCardInfoModel *model))successBlock
                        failure:(void(^)(NSError *error,NSString *orderNo))failureBlock;
{
    /// 从后台拉取服务启动所需参数
    
    [self requestOCRServiceWithAuthToken:ocrAuthToken
                                 success:^{
        /// 获取参数成功,启动SDK
        [self startOCRSDKWithType:type success:successBlock failure:failureBlock];
    } failure:failureBlock];
    
}

/// 获取OCR需要的参数
- (void)requestOCRServiceWithAuthToken:(NSString *)authToken
                               success:(void(^)(void))successBlock
                               failure:(void(^)(NSError *error,NSString *orderNo))failureBlock
{
    dispatch_async(dispatch_get_main_queue(), ^{
        [QGHUDManager showHud:@""];
    });
    
    if (!(QGUserInfo.isLogin)) return;
    ///参数拼接
    NSMutableDictionary *params = [NSMutableDictionary dictionary];
    [params setObject:QGUserInfo.sharedUserInfo.uuid forKey:@"uuid"];  /// userId
    [params setObject:self.bundleType forKey:@"type"];
    [params setObject:registerFrom forKey:@"channelId"];
    [params setObject:@(1) forKey:@"independentFlag"];
    
    QGTalosNetwork *networking = [QGTalosNetwork shareManager];
    NSMutableDictionary *headers = [NSMutableDictionary dictionary];
    if (authToken) {
        [headers setObject:authToken forKey:@"Ind-Auth-Token"];
    }
    [networking GET:[QGFaceVerifyRequestApi urlWithIndex:QGFaceRequestApiGetOCRParams] headers:headers parameters:params success:^(id  _Nonnull responseObject) {
        /// 请求成功
        if ([QGTalosNetwork checkBusinessCodeAndCodeWithResponseObjectObject:responseObject]) {
            NSDictionary *dic = responseObject[@"data"];
            self.nonce = dic[@"nonce"];
            self.sign = dic[@"sign"];
            self.userId = dic[@"userId"];
            self.versionNo = dic[@"versionNo"];
            self.orderNo = dic[@"orderNo"];
            if ([self readyForParams:NO]) {
                if(successBlock) successBlock();
            }else{
                /// 参数不全
                [QGHUDManager showHud:@"内部错误,请稍后再试..."];
                if (failureBlock) failureBlock(nil,nil);
            }
        }else{
            NSString *msg = responseObject[@"msg"];
            if (msg) {
                [QGHUDManager showHud:msg];
            }else{
                [QGHUDManager showHud:@"内部错误,请稍后再试..."];
            }
            if (failureBlock) failureBlock(nil,nil);
        }
    } failure:^(NSURLSessionDataTask * _Nonnull task, NSError * _Nonnull error) {
        if(failureBlock) failureBlock(nil,nil);
        [QGHUDManager showHud:@"网络错误,请稍后重试..."];
    }];
}


/// 启动OCR SDK
/// @param type 人像面 | 国徽面
/// @param successBlock 识别成功
/// @param failureBlock 识别失败
- (void)startOCRSDKWithType:(QGOCRIDCardType)type  success:(void(^)(WBIDCardInfoModel *model))successBlock failure:(void(^)(NSError *error,NSString *orderNo))failureBlock;
{
    dispatch_async(dispatch_get_main_queue(), ^{
        WBOCRConfig *config = [WBOCRConfig sharedConfig];
        config.SDKType = [self switchType:type];
       /// 拉起腾讯OCR服务
        [[WBOCRService sharedService] startOCRServiceWithConfig:config
                                                       version:self.versionNo
                                                         appId:self.faceAppId
                                                         nonce:self.nonce
                                                        userId:self.userId
                                                          sign:self.sign
                                                       orderNo:self.orderNo startSucceed:^{
            [QGHUDManager hideHud];  /// 启动成功
            NSLog(@"[OCR]  启动成功 ");
        } recognizeSucceed:^(id  _Nonnull resultModel, id  _Nullable extension) {
            /// 识别成功
            WBIDCardInfoModel *model = (WBIDCardInfoModel *)resultModel;
            if (successBlock) successBlock(model);
            NSLog(@"[OCR] 识别成功");
        } failed:^(NSError * _Nonnull error, id  _Nullable extension) {
            [QGHUDManager hideHud];
            /// 识别失败或者退出识别
            if (failureBlock) failureBlock(error,self.orderNo);
            NSLog(@"[OCR] 启动失败|退出识别");
       }];
    });
   
}

/// 把自定义枚举值转化成SDK枚举 默认返回人像面
- (WBOCRSDKType)switchType:(QGOCRIDCardType)type
{
    switch (type) {
        case QGOCRIDCardTypeFrontSide:  /// 人像面
            return WBOCRSDKTypeIDCardFrontSide;
            break;
        case QGOCRIDCardTypeBackSide:   /// 国徽面
            return WBOCRSDKTypeIDCardBackSide;
            break;
        default:                        /// 默认人像面
            return WBOCRSDKTypeIDCardFrontSide;
            break;
    }
}

#pragma mark - 人脸核身 Service


/// 启动人脸核身,并最终上报结果
/// @param completeBlock 上报结果完成
- (void)openFaceVerifySuccess:(void(^)(void))completeBlock
{
    /// 活体识别上报完成
    self.faceVerifyCompleteBlock = completeBlock;
    [self faceVerifyEventAnalyticsWithDesc:@"请求启动活体"];
    /// 启动活体识别服务
    [self startFaceVerifyWithStartUpSuccess:^{
        [self faceVerifyEventAnalyticsWithDesc:@"活体启动成功"];
    } startUpFailure:^{
        [self faceVerifyEventAnalyticsWithDesc:@"活体启动失败"];
    } result:^(BOOL isSuccess, NSString * _Nonnull orderNo,NSInteger code) {
         [self reportFaceVerifyResult:isSuccess orderNo:orderNo code:code];
    }];
}

/// 上报后台结果
/// @param isSuccess 是否成功
/// @param orderNo 订单号
- (void)reportFaceVerifyResult:(BOOL)isSuccess orderNo:(NSString *)orderNo code:(NSInteger)code
{
    if (!(QGUserInfo.sharedUserInfo.isLogin) && (orderNo == nil)) return;
    QGTalosNetwork *networking = [QGTalosNetwork shareManager];
    NSMutableDictionary *params = [NSMutableDictionary dictionary];
    
    [params setObject:QGUserInfo.sharedUserInfo.uuid forKey:@"uuid"];
    [params setObject:bundle_type() forKey:@"type"];
    [params setObject:registerFrom forKey:@"channelId"];
    [params setObject:orderNo forKey:@"orderNo"];
    [params setObject:@(1) forKey:@"independentFlag"];
    [params setObject:@(code) forKey:@"code"];
    [QGHUDManager showHud:@""];
    [networking GET:[QGFaceVerifyRequestApi urlWithIndex:QGFaceRequestApiGetFaceVerfiyReult] parameters:params success:^(id  _Nonnull responseObject) {
        [QGHUDManager hideHud];
        [self faceVerifyEventAnalyticsWithDesc:@"活体上报结果成功"];
        if (isSuccess && [QGTalosNetwork checkBusinessCodeAndCodeWithResponseObjectObject:responseObject]) {
            /// 上报完成回调
            if (self.faceVerifyCompleteBlock) {
                 self.faceVerifyCompleteBlock();
            }
        }
      
    } failure:^(NSURLSessionDataTask * _Nonnull task, NSError * _Nonnull error) {
        [QGHUDManager hideHud];
        [self faceVerifyEventAnalyticsWithDesc:@"活体上报结果失败"];
    }];
}


#pragma mark - 活体认证基础流程

/// 启动本地活体检测服务
/// @param startSuccess 启动成功
/// @param startFailure 启动失败
/// @param result 识别结果
- (void)startFaceVerifyWithStartUpSuccess:(void(^)(void))startSuccess
                           startUpFailure:(void(^)(void))startFailure
                                   result:(QGFaceVerifyResultBlock)result;
{
    self.faceVerifyResult = result;  /// 活体识别结果
    /// 请求faceid
    [self requestFaceVerifySuccess:^(BOOL useAuthorSource) {
        ///拉起活体识别SDK
        [self startFaceVerifySDKWithFaceType:useAuthorSource startSuccess:startSuccess startFailure:startFailure];
    } failure:startFailure];
    
}

/// 从后台拉取启动SDK所需要的参数
/// @param successBlock 拉取成功回调
/// @param failureBlock 拉取失败回调
- (void)requestFaceVerifySuccess:(void(^)(BOOL useAuthorSource))successBlock
                         failure:(void(^)(void))failureBlock
{
    dispatch_async(dispatch_get_main_queue(), ^{
        [QGHUDManager showHud:@""];
    });
    
    if (!(QGUserInfo.isLogin)) return;
       ///参数拼接
       NSMutableDictionary *params = [NSMutableDictionary dictionary];
       [params setObject:QGUserInfo.sharedUserInfo.uuid forKey:@"uuid"];  /// userId
       [params setObject:self.bundleType forKey:@"type"];
       [params setObject:registerFrom forKey:@"channelId"];
       [params setObject:@(1) forKey:@"independentFlag"];
       
      QGTalosNetwork *networking = [QGTalosNetwork shareManager];
       /// 获取参数
      [networking GET:[QGFaceVerifyRequestApi urlWithIndex:QGFaceRequestApiGetFaceverify] headers:nil parameters:params success:^(id  _Nonnull responseObject) {
            /// 请求成功
            if ([QGTalosNetwork checkBusinessCodeAndCodeWithResponseObjectObject:responseObject]) {
                NSDictionary *dic = responseObject[@"data"];
                self.nonce = dic[@"nonce"];
                self.sign = dic[@"sign"];
                self.userId = dic[@"userId"];
                self.versionNo = dic[@"versionNo"];
                self.orderNo = dic[@"orderNo"];
                self.faceId = dic[@"faceId"];
                BOOL useAuthorSource = YES;  /// 是否使用权威对比源
                /// 没返回hasSource字段或者hasSource字段值不是1就是自带对比源
                if (!dic[@"hasSource"] || [dic[@"hasSource"] intValue] != 1) {
                     useAuthorSource = NO;
                }
                /// 检查参数是否完整
                if ([self readyForParams:YES]) {
                    if(successBlock) successBlock(useAuthorSource);
                }else{
                    /// 参数不全
                    [QGHUDManager showHud:@"内部错误,请稍后重试..."];
                    if (failureBlock) failureBlock();
                }
            }else{
                NSString *msg = responseObject[@"msg"];
                if (msg) {
                    [QGHUDManager showHud:msg];
                }else{
                    [QGHUDManager showHud:@"内部错误,请稍后再试..."];
                }
                if (failureBlock) failureBlock();
            }
      } failure:^(NSURLSessionDataTask * _Nonnull task, NSError * _Nonnull error) {
            if(failureBlock) failureBlock();
            [QGHUDManager showHud:@"网络错误,请稍后重试..."];
      }];
}

/// 检查参数是否完成
/// @param isFace 是否是活体检测
- (BOOL)readyForParams:(BOOL)isFace
{
    if (self.nonce == nil) return NO;
    if (self.sign == nil) return NO;
    if (self.userId == nil) return NO;
    if (self.versionNo == nil) return NO;
    if (isFace) {
        if (self.faceId == nil) return NO;
    }
    return YES;
}

/// 启动活体识别SDK服务
/// @param useAuthorSource 使用权威对比源
/// @param startSuccess 启动SDK成功
/// @param startFailure 启动SDK失败
- (void)startFaceVerifySDKWithFaceType:(BOOL)useAuthorSource
                          startSuccess:(void(^)(void))startSuccess
                          startFailure:(void(^)(void))startFailure
{
    dispatch_async(dispatch_get_main_queue(), ^{
        /// 初始化人脸核身
        WBFaceVerifyCustomerService *service = [WBFaceVerifyCustomerService sharedInstance];
        service.delegate = self;
        WBFaceVerifySDKConfig *config = [self getSDKSettings];
        if (useAuthorSource) {
            /// 使用权威对比源--> 有源对比
            [service loginInLiveCheckAndCompareWithIdImageService:self.userId
                                                            nonce:self.nonce
                                                             sign:self.sign
                                                            appid:self.faceAppId
                                                          orderNo:self.orderNo
                                                       apiVersion:self.versionNo
                                                          licence:self.faceLicense
                                                         faceType:WBFaceVerifyLivingType_Light
                                                           faceId:self.faceId
                                                        sdkConfig:config
                                                          success:^{
                [QGHUDManager hideHud];
                if (startSuccess) startSuccess();
                NSLog(@"[Face Verify] 权威对比源 启动成功");
            } failure:^(WBFaceError * _Nonnull error) {
                if (startFailure) startFailure();
                [QGHUDManager showHud:@"活体启动失败,请联系客服"];
                NSLog(@"[Face Verify] 权威对比源 启动失败 %@",error);
            }];
        }else{
            /// 自带对比源 --> 无源对比
           [service loginInLiveCheckAndCompareWithSourceImageService:self.userId
                                                                nonce:self.nonce
                                                                 sign:self.sign
                                                                appid:self.faceAppId
                                                              orderNo:self.orderNo
                                                           apiVersion:self.versionNo
                                                              licence:self.faceLicense
                                                             faceType:WBFaceVerifyLivingType_Light
                                                               faceId:self.faceId
                                                            sdkConfig:config
                                                              success:^{
                [QGHUDManager hideHud];
                if (startSuccess) startSuccess();
                NSLog(@"[Face Verify] 自带对比源 启动成功");
            } failure:^(WBFaceError * _Nonnull error) {
                if (startFailure) startFailure();
                 [QGHUDManager showHud:@"活体启动失败,请联系客服"];
                NSLog(@"[Face Verify] 自带对比源 启动失败 %@",error);
            }];
        }
    });
}

/// SDK 配置信息
-(WBFaceVerifySDKConfig *)getSDKSettings{
    WBFaceVerifySDKConfig *config = [WBFaceVerifySDKConfig sdkConfig];
    config.recordVideo = YES;
    config.theme = WBFaceVerifyThemeLightness;
    return  config;
}

#pragma mark - 人脸核身 Delegate 获取人脸核身结果


/// 获取活体对比结果 delegate
/// @param faceVerifyResult 结果
-(void)wbfaceVerifyCustomerServiceDidFinishedWithFaceVerifyResult:(WBFaceVerifyResult *)faceVerifyResult
{
    NSLog(@"result = %@",faceVerifyResult);
    NSInteger code = 0;
    if (!faceVerifyResult.isSuccess) {
        code = faceVerifyResult.error.code;
        [self faceVerifyEventAnalyticsWithDesc:@"活体识别失败"];
        if (faceVerifyResult.error.desc) {
            [QGHUDManager showHud:faceVerifyResult.error.desc];
        }else{
            [QGHUDManager showHud:@"活体识别失败!"];
        }
        
    }else{
        [self faceVerifyEventAnalyticsWithDesc:@"活体识别成功"];
    }
    if (self.faceVerifyResult) {
        self.faceVerifyResult(faceVerifyResult.isSuccess,faceVerifyResult.orderNo,code);
    }
}



#pragma mark - 统计

// 活体检测统计信息
/// @param desc 事件描述
- (void)faceVerifyEventAnalyticsWithDesc:(NSString *)desc
{
    NSDictionary *propertieDic =@{@"event_uuid":[QGUserInfo sharedUserInfo].uuid,
                                  @"event_desc":desc};
      [self faceAnalyticsTrackWithDeatilInfo:propertieDic];
}
/**
 face++ Ocr数据埋点统计
 
 @param info 统计信息
 */
-(void)faceAnalyticsTrackWithDeatilInfo:(NSDictionary *)info{
    
    [QGAnalyticsService analyticsTrack:@"TXSDK_AppFaceVerify_Event" withProperties: info];
}


@end
