//
//  QGCacheManager.m
//  CreditWallet
//
//  Created by 张博 on 16/11/17.
//  Copyright © 2016年 QuantGroup. All rights reserved.
//

#import "QGCacheManager.h"
#import "QGDiskCache.h"


#define APPConfigPlistName @"QGDiskAppConfigCachePlist"
#define UserConfigPlistName @"QGDiskUserConfigCachePlist"

@interface QGCacheManager ()

/**
 *  缓存数据
 */
@property (nonatomic, strong) NSMutableDictionary * appConfigDataDic;

/**
 *  缓存数据
 */
@property (nonatomic, strong) NSMutableDictionary * userConfigDataDic;

@end

@implementation QGCacheManager


/**
 *  缓存管理类
 */
+ (instancetype)shareCacheManager
{
    static QGCacheManager * shareManager = nil;
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        shareManager = [[self alloc]init];
    });
    return shareManager;
}

- (instancetype)init
{
    if (self = [super init]) {
        self.appConfigDataDic = [QGDiskCache loadDicFromPlistFile:APPConfigPlistName];
        if (!self.appConfigDataDic) {
            self.appConfigDataDic  = [NSMutableDictionary dictionaryWithCapacity:0];
        }
        
        self.userConfigDataDic = [QGDiskCache loadDicFromPlistFile:UserConfigPlistName];
        if (!self.userConfigDataDic) {
            self.userConfigDataDic  = [NSMutableDictionary dictionaryWithCapacity:0];
        }
        
    }
    return self;
}



#pragma mark - 用户相关配置
/**
 *  保存用户相关配置到本地
 *
 *  @param value value
 *  @param key   key
 */
- (void)setUserConfigInfoValue:(id)value forKey:(NSString *)key
{
    if (key.length == 0 || !value) {
        return;
    }
    [self.userConfigDataDic setValue:value forKey:key];
    
    [QGDiskCache saveToPlistFileWithData:self.userConfigDataDic fileName:UserConfigPlistName];
}



/**
 *  批量保存用户相关配置到本地
 *
 *  @param infoDic 用户信息字典
 */
- (void)setUserConfigInfoWithDic:(NSDictionary *)infoDic
{
    if (infoDic.count == 0) {
        return;
    }
    [self.userConfigDataDic setValuesForKeysWithDictionary:infoDic];
    [QGDiskCache saveToPlistFileWithData:self.userConfigDataDic fileName:UserConfigPlistName];
    
}



/**
 *  获取用户配置
 *
 *  @param key key
 *
 *  @return value
 */
- (id)userConfigInfoValueForKey:(NSString *)key
{
    if (key.length == 0) {
        return nil;
    }
    id propertyValue = [self.userConfigDataDic objectForKey:key];
    return propertyValue;
}


/**
 *  删除用户配置
 *
 *  @param keys keyArray
 */
- (void)removeUserConfigWithKeys:(NSArray *)keys
{
    if (keys.count == 0) {
        return;
    }
    
    [self.userConfigDataDic removeObjectsForKeys:keys];
    [QGDiskCache saveToPlistFileWithData:self.userConfigDataDic fileName:UserConfigPlistName];
}


#pragma mark - APP相关配置


/**
 *  保存APP配置到本地
 *  此配置与当前登录账户无关，不会因为登录/登出而被清除
 *  @param value value
 *  @param key   key
 */
- (void)setAPPConfigValue:(id)value forKey:(NSString *)key;
{
    if (key.length == 0 || !value) {
        return;
    }
    [self.appConfigDataDic setValue:value forKey:key];
    
    [QGDiskCache saveToPlistFileWithData:self.appConfigDataDic fileName:APPConfigPlistName];
}


/**
 *  批量保存APP配置到本地
 *
 *  @param infoDic 用户信息字典
 */
- (void)setAPPConfigInfoWithDic:(NSDictionary *)infoDic
{
    if (infoDic.count == 0) {
        return;
    }
    [self.appConfigDataDic setValuesForKeysWithDictionary:infoDic];
    [QGDiskCache saveToPlistFileWithData:self.appConfigDataDic fileName:APPConfigPlistName];
}


/**
 *  获取APP配置
 *
 *  @param key key
 *
 *  @return value
 */
- (id)appConfigValueForKey:(NSString *)key;
{
    if (key.length == 0) {
        return nil;
    }
    id propertyValue = [self.appConfigDataDic objectForKey:key];
    return propertyValue;
    
}

/**
 *  删除用户配置
 *
 *  @param keys keyArray
 */
- (void)removeAppConfigWithKeys:(NSArray *)keys
{
    if (keys.count == 0) {
        return;
    }
    
    [self.appConfigDataDic removeObjectsForKeys:keys];
    [QGDiskCache saveToPlistFileWithData:self.appConfigDataDic fileName:APPConfigPlistName];
}




/**
 *  保存图片到本地路径
 *
 *  @param image      图片
 *  @param name       本地名称
 *  @param completion 完成回调
 */
+ (void)saveImage:(UIImage *)image withName:(NSString *)name completion:(void(^)())completion
{
    [QGDiskCache saveImage:image withName:name completion:completion];
}


/**
 *  获取本地保存的图片 异步
 *
 *  @param name 图片名字
 *
 *  @return 返回图片
 */
+ (void)imageWithName:(NSString *)name completion:(void(^)(UIImage * image))completion
{
    [QGDiskCache imageWithName:name completion:completion];
}


/**
 *  获取本地保存图片 同步
 *
 *  @param name 图片名字
 *
 *  @return 图片
 */
+ (UIImage *)imageWithName:(NSString *)name
{
    return [QGDiskCache imageWithName:name];
}


/**
 *  移除本地保存的图片
 *
 *  @param name 图片名
 */
+ (void)removeImageWithName:(NSString *)name
{
    [QGDiskCache removeImageWithName:name];
}


/**
 *  删除本地所有保存的图片
 */
+ (void)removeAllImage
{
    [QGDiskCache removeAllImage];
}

@end
