//
//  NSString+CreditWallet.m
//  CreditWallet
//
//  Created by 逍遥子 on 2016/11/14.
//  Copyright © 2016年 QuantGroup. All rights reserved.
//

#import "AFURLRequestSerialization.h"
#import "NSString+CreditWallet.h"
#import <CommonCrypto/CommonDigest.h>
#include <sys/sysctl.h>

@implementation NSString (CreditWallet)

/**
 汉字转拼音
 */
- (NSString *)transformToPinyin {
    
    NSMutableString *mutableString = [NSMutableString stringWithString:self];
    
    CFStringTransform((CFMutableStringRef) mutableString, NULL, kCFStringTransformToLatin, false);
    
    mutableString = (NSMutableString *) [mutableString stringByFoldingWithOptions:NSDiacriticInsensitiveSearch locale:[NSLocale currentLocale]];
    
    return mutableString;
}


/**
 JSON格式的字符串转换成字典
 
 @param jsonString json字符串
 @return 字典
 */
+ (NSDictionary *)dictionaryWithJsonString:(NSString *)jsonString
{
    if (jsonString == nil) {
        return nil;
    }
    NSData *jsonData = [jsonString dataUsingEncoding:NSUTF8StringEncoding];
    NSError *err;
    NSDictionary *dic = [NSJSONSerialization JSONObjectWithData:jsonData
                                                        options:NSJSONReadingMutableContainers
                                                          error:&err];
    if (err) {
        NSLog(@"json解析失败：%@", err);
        return nil;
    }
    return dic;
}

/**
 字典转换成json格式的字符串
 
 @param dic 字典
 @return json格式字符串
 */
+ (NSString *)dictionaryToJson:(NSDictionary *)dic
{
    NSError *parseError = nil;
    NSData *jsonData = [NSJSONSerialization dataWithJSONObject:dic options:NSJSONWritingPrettyPrinted error:&parseError];
    if (parseError) {
        return nil;
    }
    return [[[[[NSString alloc] initWithData:jsonData encoding:NSUTF8StringEncoding] stringByReplacingOccurrencesOfString:@"\n" withString:@""] stringByReplacingOccurrencesOfString:@" " withString:@""] stringByReplacingOccurrencesOfString:@"\\" withString:@""];
}


/**
 返回不为null的字符串
 
 @param string 需要处理的字符串或者nil
 @return 目标字符串
 */
+ (NSString *)getCheckString:(NSString *)string
{
    if (!string) {
        return @"";
    }
    return string;
}

/**
 获取固定字符串的高度

 @param content 字符串
 @param width 最大宽度
 @param font 字体大小
 @return 目标高度
 */
+ (CGFloat)getHeightWithContent:(NSString *)content width:(CGFloat)width UIFont:(UIFont *)font {
    CGRect rect = [content boundingRectWithSize:CGSizeMake(width, MAXFLOAT)
                                        options:NSStringDrawingUsesLineFragmentOrigin
                                     attributes:@{NSFontAttributeName : font}
                                        context:nil];
    return rect.size.height;
}

/**
 获取固定字符串的宽度

 @param content 字符串
 @param height 最大高度
 @param font 字体大小
 @return 目标宽度
 */
+ (CGFloat)getWidthWithContent:(NSString *)content height:(CGFloat)height UIFont:(UIFont *)font {
    CGRect rect = [content boundingRectWithSize:CGSizeMake(MAXFLOAT, height)
                                        options:NSStringDrawingUsesLineFragmentOrigin
                                     attributes:@{NSFontAttributeName : font}
                                        context:nil];
    return rect.size.width;
}


/**
 计算字符串大小

 @param font 字体
 @param maxSize 最大size
 @return 目标size
 */
- (CGSize)sizeWithFont:(UIFont *)font maxSize:(CGSize)maxSize {
    NSDictionary *dict = @{NSFontAttributeName : font};
    CGSize textSize = [self boundingRectWithSize:maxSize options:NSStringDrawingUsesLineFragmentOrigin attributes:dict context:nil].size;
    return textSize;
}

/**
 是否包含某段字符
 
 @param string 匹配字符串
 @return 目标结果
 */
- (BOOL)isContainOfString:(NSString *)string
{
    NSRange range = [self rangeOfString:string];
    if (range.length > 0) {
        return YES;
    }
    return NO;
}


/**
 是否是身份证号
 
 @return 目标结果
 */
- (BOOL)isIdentityCard
{
    BOOL flag;
    if (self.length <= 0) {
        flag = NO;
        return flag;
    }
    NSString *regex2 = @"^(\\d{14}|\\d{17})(\\d|[xX])$";
    NSPredicate *identityCardPredicate = [NSPredicate predicateWithFormat:@"SELF MATCHES %@", regex2];
    return [identityCardPredicate evaluateWithObject:self];
}

/**
 邮箱正则
 
 @return 目标结果
 */
- (BOOL)isEmail
{
    NSString *emailRegEx =
    @"(?:[a-z0-9!#$%\\&'*+/=?\\^_`{|}~-]+(?:\\.[a-z0-9!#$%\\&'*+/=?\\^_`{|}"
    @"~-]+)*|\"(?:[\\x01-\\x08\\x0b\\x0c\\x0e-\\x1f\\x21\\x23-\\x5b\\x5d-\\"
    @"x7f]|\\\\[\\x01-\\x09\\x0b\\x0c\\x0e-\\x7f])*\")@(?:(?:[a-z0-9](?:[a-"
    @"z0-9-]*[a-z0-9])?\\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?|\\[(?:(?:25[0-5"
    @"]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-"
    @"9][0-9]?|[a-z0-9-]*[a-z0-9]:(?:[\\x01-\\x08\\x0b\\x0c\\x0e-\\x1f\\x21"
    @"-\\x5a\\x53-\\x7f]|\\\\[\\x01-\\x09\\x0b\\x0c\\x0e-\\x7f])+)\\])";
    NSPredicate *regExPredicate = [NSPredicate predicateWithFormat:@"SELF MATCHES %@", emailRegEx];
    return [regExPredicate evaluateWithObject:[self lowercaseString]];
}

- (NSString *)trimmedWhitespaceString {
    return [self stringByTrimmingCharactersInSet:[NSCharacterSet whitespaceCharacterSet]];
}


/**
 是否为数字
 */
- (BOOL)isNumber {
    if ([self isEmptyOrWhitespace]) {
        return NO;
    }
    NSString *integerOrFloatPointRegEx = @"[0-9]+$";
    NSPredicate *regExPredicate = [NSPredicate predicateWithFormat:@"SELF MATCHES %@", integerOrFloatPointRegEx];
    return [regExPredicate evaluateWithObject:[self lowercaseString]];
}

- (BOOL)isEmptyOrWhitespace {
    return self.length == 0 || [[self trimmedWhitespaceString] length] == 0;
}

/**
 md5
 
 @return md5之后的字符串
 */
- (NSString *)md5
{
    const char *cStr = [self UTF8String];
    //开辟一个16字节（128位：md5加密出来就是128位/bit）的空间（一个字节=8字位=8个二进制数）
    unsigned char result[16];
    //把cStr字符串转换成了32位的16进制数列（这个过程不可逆转） 存储到了result这个空间中
    CC_MD5(cStr, (int) strlen(cStr), result);
    return [NSString stringWithFormat:
            @"%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x",
            result[0], result[1], result[2], result[3],
            result[4], result[5], result[6], result[7],
            result[8], result[9], result[10], result[11],
            result[12], result[13], result[14], result[15]];
}

/*
 URLFragmentAllowedCharacterSet  "#%<>[\]^`{|}
 URLHostAllowedCharacterSet      "#%/<>?@\^`{|}
 URLPasswordAllowedCharacterSet  "#%/:<>?@[\]^`{|}
 URLPathAllowedCharacterSet      "#%;<>?[\]^`{|}
 URLQueryAllowedCharacterSet     "#%<>[\]^`{|}
 URLUserAllowedCharacterSet      "#%/:<>?@[\]^`
 */
+ (NSString *)urlEncodeString:(NSString *)string {
    if (string && string.length > 0) {
        NSMutableCharacterSet *allowedCharacterSet = [[[NSCharacterSet characterSetWithCharactersInString:@"`#%^{}\"[]|\\<> "] invertedSet] mutableCopy];

        // FIXME: https://github.com/AFNetworking/AFNetworking/pull/3028
        // return [string stringByAddingPercentEncodingWithAllowedCharacters:allowedCharacterSet];
        static NSUInteger const batchSize = 50;
        NSUInteger index = 0;
        NSMutableString *escaped = @"".mutableCopy;

        while (index < string.length) {
            NSUInteger length = MIN(string.length - index, batchSize);
            NSRange range = NSMakeRange(index, length);

            // To avoid breaking up character sequences such as 👴🏻👮🏽
            range = [string rangeOfComposedCharacterSequencesForRange:range];

            NSString *substring = [string substringWithRange:range];
            //stringByAddingPercentEncodingWithAllowedCharacters苹果默认UTF-8
            NSString *encoded = [substring stringByAddingPercentEncodingWithAllowedCharacters:allowedCharacterSet];
            [escaped appendString:encoded];

            index += range.length;
        }

        return escaped;
    }
    return @"";
}


/**
 拼接URL
 
 @param baseURL 基础字符串
 @param params 字典参数
 @return 目标字符串
 */
+ (NSString *)serializeURL:(NSString *)baseURL params:(NSDictionary *)params {
    if (baseURL && baseURL.length > 0) {
        NSURL *parsedURL = [NSURL URLWithString:baseURL];
        NSString *query = @"";
        if (params && params.count > 0) {
            query = AFQueryStringFromParameters(params);
        }
        return [[NSURL URLWithString:[baseURL stringByAppendingFormat:parsedURL.query ? @"&%@" : @"?%@", query]] absoluteString];
    }
    return @"";
}


/**
 根据URL 解析出query里面的字典
 
 @param queryString 需要解析字符串
 @return 解析字典结果
 */
+ (NSDictionary *)explodeQueryStringToDictionary:(NSString *)queryString {
    NSArray *firstExplode = [queryString componentsSeparatedByString:@"&"];
    NSArray *secondExplode;
    NSInteger count = [firstExplode count];
    NSMutableDictionary *returnDictionary = [NSMutableDictionary dictionaryWithCapacity:count];
    for (NSInteger i = 0; i < count; i++) {
        secondExplode = [(NSString *) [firstExplode objectAtIndex:i] componentsSeparatedByString:@"="];
        if ([secondExplode count] == 2) {
            [returnDictionary setObject:[secondExplode objectAtIndex:1]
                                 forKey:[secondExplode objectAtIndex:0]];
        }
    }
    return returnDictionary;
}

/**
 判断一个字符串中是否包含汉字

 @param aString 待判断的字符串
 @return yes - 包含中文字符  no - 不包含中文字符
 */
+ (BOOL)isContainChinese:(NSString *)aString {
    for (NSInteger i = 0; i < [aString length]; i++) {
        NSInteger character = [aString characterAtIndex:i];
        if (character >= 0x4e00 && character < 0x9fff) {
            return YES;
        }
    }

    return NO;
}

/**
 获取首个字母，如果是汉字，则为汉字对应的拼音首字母

 @param aString 待处理的字符串
 @return 字符串首字母
 */
+ (NSString *)firstCharactor:(NSString *)aString {
    //转成了可变字符串
    NSMutableString *str = [NSMutableString stringWithString:aString];
    //先转换为带声调的拼音
    CFStringTransform((CFMutableStringRef) str, NULL, kCFStringTransformMandarinLatin, NO);
    //再转换为不带声调的拼音
    CFStringTransform((CFMutableStringRef) str, NULL, kCFStringTransformStripDiacritics, NO);
    //转化为大写拼音
    NSString *pinYin = [str capitalizedString];
    //获取并返回首字母
    return [pinYin substringToIndex:1];
}

/**
 七牛云 图片url支持webp

 @param urlStr url
 @return webpUrl
 */
+ (NSString *)addWebPFormatFromUrlStr:(NSString *)urlStr {
    if (urlStr && ([urlStr rangeOfString:@"qnssl.com"].location != NSNotFound || [urlStr rangeOfString:@"jdimage.xyqb.com"].location != NSNotFound)) {
        //imageMogr2 提供一系列高级图片处理功能，包括格式转换、缩放、裁剪、旋转等。imageMogr2 接口可支持处理的原图片格式有 psd、jpeg、png、gif、webp、tiff、bmp
        if ([urlStr rangeOfString:@"?"].location == NSNotFound) {
            urlStr = [NSString stringWithFormat:@"%@?imageMogr2/format/webp", urlStr];
        } else {
            if ([urlStr rangeOfString:@"imageMogr2"].location == NSNotFound) {
                urlStr = [NSString stringWithFormat:@"%@/imageMogr2/format/webp", urlStr];
            } else {
                urlStr = [NSString stringWithFormat:@"%@/format/webp", urlStr];
            }
        }
    }
    if (!urlStr) {
        urlStr = @"";
    }
    return urlStr;
}

+ (BOOL)isEmpty:(NSString *)string{
    if (!string) return YES;
    if (![string isKindOfClass:[NSString class]]) return YES;
    if ( [string isKindOfClass:[NSNull class]] ) return YES;
    if ( [[string stringByTrimmingCharactersInSet:[NSCharacterSet whitespaceCharacterSet]] length]==0 ) return YES;
    return NO;
}


-(BOOL)isValidPasswordString
{
    BOOL result = NO;
    if ([self length] >= 8 && [self length] <= 20){
        //数字条件
        NSRegularExpression *tNumRegularExpression = [NSRegularExpression regularExpressionWithPattern:@"[0-9]" options:NSRegularExpressionCaseInsensitive error:nil];
        
        //符合数字条件的有几个
        NSUInteger tNumMatchCount = [tNumRegularExpression numberOfMatchesInString:self
                                                                           options:NSMatchingReportProgress
                                                                             range:NSMakeRange(0, self.length)];
        
        //英文字条件
        NSRegularExpression *tLetterRegularExpression = [NSRegularExpression regularExpressionWithPattern:@"[A-Za-z]" options:NSRegularExpressionCaseInsensitive error:nil];
        
        //符合英文字条件的有几个
        NSUInteger tLetterMatchCount = [tLetterRegularExpression numberOfMatchesInString:self
                                                                                 options:NSMatchingReportProgress
                                                                                   range:NSMakeRange(0, self.length)];
        
        if(tNumMatchCount >= 1 && tLetterMatchCount >= 1){
            result = YES;
        }
        
    }
    return result;
    
}

- (BOOL)isLegalAddress
{
    // 匹配规则汉字，字母，数字，-，# ,以及 .. ，。
    NSString *regex = @"[A-Za-z0-9\u4e00-\u9fa5-#,.，。]+";
    NSPredicate *pred = [NSPredicate predicateWithFormat:@"SELF MATCHES %@", regex];
    return [pred evaluateWithObject:self];
}

- (BOOL)isOnlyContainsNumberAndLetter
{
    NSString *regex = @"[A-Za-z0-9]+";
    NSPredicate *pred = [NSPredicate predicateWithFormat:@"SELF MATCHES %@", regex];
    return [pred evaluateWithObject:self];
}


/// 格式化银行卡号
- (NSString *)formatterBankCardNum
{
    NSString * tempStr = self;
    NSInteger size = (tempStr.length/4);
    NSMutableArray * tmpStrArr = [[NSMutableArray alloc] init];
    for (int n = 0; n < size; n++)
    {
        [tmpStrArr addObject:[tempStr substringWithRange:NSMakeRange(n * 4, 4)]];
    }
    [tmpStrArr addObject:[tempStr substringWithRange:NSMakeRange(size * 4, (tempStr.length % 4))]];
    tempStr = [tmpStrArr componentsJoinedByString:@" "];
    return tempStr;
}

@end
