package cn.quantgroup.customer.model.order;

import cn.quantgroup.customer.enums.UserRepayType;
import cn.quantgroup.customer.util.DateUtil;
import cn.quantgroup.customer.util.MoneySerializer;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import lombok.Data;

import java.math.BigDecimal;
import java.util.Date;

/**
 * 还款计划信息
 *
 * @author Wang Xiangwei
 * @version 2020/3/9
 */
@Data
public class Repayment {

    /**
     * 还款期数
     */
    private Integer periodNo;

    /**
     * 本金
     */
    @JsonSerialize(using = MoneySerializer.class)
    private BigDecimal principal;

    /**
     * 利息
     */
    @JsonSerialize(using = MoneySerializer.class)
    private BigDecimal interest;

    /**
     * 担保费
     */
    @JsonSerialize(using = MoneySerializer.class)
    private BigDecimal serviceFee;

    /**
     * 其他担保
     */
    @JsonSerialize(using = MoneySerializer.class)
    private BigDecimal otherFee;

    /**
     * 罚息
     */
    @JsonSerialize(using = MoneySerializer.class)
    private BigDecimal overdueInterest;


    /**
     * 当前减免
     */
    @JsonSerialize(using = MoneySerializer.class)
    private BigDecimal reliefAmount;

    /**
     * 还款总额
     */
    @JsonSerialize(using = MoneySerializer.class)
    private BigDecimal repayAmount;

    /**
     * 应还总额
     */
    @JsonSerialize(using = MoneySerializer.class)
    private BigDecimal requiredRepayment;

    /**
     * 应还时间 自己转化
     */
    private String shouldAt;

    /**
     * 还款时间
     */
    private String repaymentReceivedAt;

    /**
     * 还款状态 自己转化
     */
    private String repaymentStatus;

    /**
     * 还款方式 自己转化
     */
    private String repayType;

    /**
     * 逾期天数
     */
    private Integer overdueDays;


    public static Repayment valueOf(RepaymentPlanItem repaymentPlanItem) {

        if (repaymentPlanItem == null) {
            return null;
        }

        Repayment repayment = new Repayment();
        repayment.setPeriodNo(repaymentPlanItem.getPeriodNo());
        repayment.setPrincipal(repaymentPlanItem.getPrincipal());
        repayment.setInterest(repaymentPlanItem.getInterest());
        repayment.setServiceFee(repaymentPlanItem.getServiceFee());
        repayment.setOtherFee(repaymentPlanItem.getOtherFee());
        repayment.setOverdueInterest(repaymentPlanItem.getOverdueInterest());
        repayment.setReliefAmount(repaymentPlanItem.getReliefAmount());
        repayment.setRepayAmount(repaymentPlanItem.getRepayAmount());
        repayment.setRequiredRepayment(repaymentPlanItem.getRequiredRepayment());

        Long dueTime = repaymentPlanItem.getDueTime();
        if (dueTime != null) {
            //java时间戳13位
            String shouldAt = DateUtil.format(new Date(dueTime * 1000), DateUtil.YYYY_MM_DD);
            repayment.setShouldAt(shouldAt);
        }else {
            repayment.setShouldAt("");
        }

        Long repayTime = repaymentPlanItem.getRepayTime();
        if (repayTime != null && repayTime > 0) {
            //java时间戳13位
            String repaymentReceivedAt = DateUtil.format(new Date(repayTime * 1000), DateUtil.DATE_FORMAT_1);
            repayment.setRepaymentReceivedAt(repaymentReceivedAt);
        }else {
            repayment.setRepaymentReceivedAt("");
        }

        Integer repaymentStatus = repaymentPlanItem.getRepaymentStatus();
        String repayStatus = "";
        //还款状态，0 ("未还款")、2 ("已逾期")、3 ("已还款")
        switch (repaymentStatus) {
            case 0:
                repayStatus = "未还款";
                break;
            case 2:
                repayStatus = "已逾期";
                break;
            case 3:
                repayStatus = "已还款";

        }
        repayment.setRepaymentStatus(repayStatus);
        if(repaymentStatus == 3){
            repayment.setRepayType(UserRepayType.valueOf(repaymentPlanItem.getRepaymentType()).getDesc());
        }else {
            repayment.setRepayType("");
        }
        repayment.setOverdueDays(repaymentPlanItem.getOverdueDays());
        return repayment;

    }
}
