package cn.quantgroup.customer.service.impl;

import cn.quantgroup.customer.common.adapter.LocalDateTypeAdapter;
import cn.quantgroup.customer.entity.User;
import cn.quantgroup.customer.enums.ErrorCodeEnum;
import cn.quantgroup.customer.exception.BusinessException;
import cn.quantgroup.customer.repo.UserRepo;
import cn.quantgroup.customer.rest.param.phone.ModifyPhoneAudit;
import cn.quantgroup.customer.rest.param.phone.ModifyPhoneFeedback;
import cn.quantgroup.customer.rest.param.phone.ModifyPhoneQuery;
import cn.quantgroup.customer.rest.vo.JsonResult;
import cn.quantgroup.customer.service.IUserService;
import cn.quantgroup.customer.service.http.IHttpService;
import cn.quantgroup.riskcontrol.model.AuthenticationUserDetail;
import com.google.common.collect.Maps;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.stereotype.Service;

import java.time.LocalDate;
import java.util.Map;

import static cn.quantgroup.customer.constant.Constant.GSON;

@Slf4j
@Service("userService")
public class UserServiceImpl implements IUserService {

    @Value("${passportapi2.http}")
    private String userSysUrl;

    private final UserRepo userRepo;
    private final IHttpService httpService;


    @Autowired
    public UserServiceImpl(UserRepo userRepo, IHttpService httpService) {
        this.userRepo = userRepo;
        this.httpService = httpService;
    }

    @Override
    public UserDetails loadUserByUsername(String userName) throws UsernameNotFoundException {
        User user = userRepo.findByUsername(userName);
        if (user == null) {
            throw new UsernameNotFoundException("user: " + userName + " do not exist!");
        }
        return new AuthenticationUserDetail(user);
    }

    @Override
    public String modifyPhoneQuery(ModifyPhoneQuery modifyPhoneQuery) {
        String url = userSysUrl + "/v1/user/modify/phone_no";
        Gson gson = new GsonBuilder()
                .registerTypeAdapter(LocalDate.class, new LocalDateTypeAdapter())
                .create();
        Map param = gson.fromJson(gson.toJson(modifyPhoneQuery), Map.class);
        try {
            String response = httpService.get(url, param);
            if (StringUtils.isEmpty(response)) {
                log.error("[user][query modify phone list] 请求业务系统返回值为空,modifyPhoneQuery:{}", modifyPhoneQuery);
            }
            return response;
        } catch (Exception e) {
            log.error("[user][query modify phone list] 网络通讯异常,modifyPhoneQuery:{},ex:{}", modifyPhoneQuery, ExceptionUtils.getStackTrace(e));
            throw new BusinessException(ErrorCodeEnum.NET_ERROR);
        }
    }

    @Override
    public String modifyPhoneAudit(ModifyPhoneAudit modifyPhoneAudit) {
        String url = userSysUrl + "/v1/user/modify/phone_no/audit";
        Map param = GSON.fromJson(GSON.toJson(modifyPhoneAudit), Map.class);
        try {
            Map<String, String> header = Maps.newHashMap();
            header.put("Content-type", "application/json");
            String result = httpService.post(url, header, param);
            log.info("[user][query modify audit ] 请求业务系统返回值:{}", result);
            return result;
        } catch (Exception e) {
            log.error("[user][query modify audit ] 网络通讯异常,modifyPhoneAudit:{},ex:{}", modifyPhoneAudit, ExceptionUtils.getStackTrace(e));
            throw new BusinessException(ErrorCodeEnum.NET_ERROR);
        }
    }

    @Override
    public String modifyPhoneFeedback(ModifyPhoneFeedback modifyPhoneFeedback) {
        String url = userSysUrl + "/v1/user/modify/phone_no/{id}/feedback";
        try {
            String id = modifyPhoneFeedback.getId();
            url = url.replace("{id}", id);
            Map<String, String> header = Maps.newHashMap();
            header.put("Content-type", "application/json");
            String result = httpService.post(url, header, null);
            log.info("[user][query modify feedback ] 请求业务系统返回值:{}", result);
            return result;
        } catch (Exception e) {
            log.error("[user][query modify feedback ] 网络通讯异常,modifyPhoneFeedback:{},ex:{}", modifyPhoneFeedback, ExceptionUtils.getStackTrace(e));
            throw new BusinessException(ErrorCodeEnum.NET_ERROR);
        }
    }

    @Override
    public String findUuidByIdNo(String idNo) {
        String url = userSysUrl + "/innerapi/fetchUuid";
        try {
            Map<String, Object> param = Maps.newHashMap();
            param.put("idNo", idNo);
            String result = httpService.post(url, param);
            log.info("[user][query uuid by idNo ] 请求业务系统返回值:{}", result);
            JsonResult jsonResult = GSON.fromJson(result, JsonResult.class);
            Object uuid = jsonResult.getData();
            return String.valueOf(uuid);
        } catch (Exception e) {
            log.error("[user][query uuid by idNo ] 网络通讯异常,idNo:{},ex:{}", idNo, ExceptionUtils.getStackTrace(e));
            throw new BusinessException(ErrorCodeEnum.NET_ERROR);
        }
    }

}
