package cn.quantgroup.customer.service.impl;

import cn.quantgroup.customer.entity.Authority;
import cn.quantgroup.customer.entity.RoleAuthority;
import cn.quantgroup.customer.entity.User;
import cn.quantgroup.customer.entity.UserRole;
import cn.quantgroup.customer.enums.ErrorCodeEnum;
import cn.quantgroup.customer.exception.NetCommunicationException;
import cn.quantgroup.customer.repo.AuthorityRepo;
import cn.quantgroup.customer.repo.RoleAuthorityRepo;
import cn.quantgroup.customer.repo.UserRepo;
import cn.quantgroup.customer.repo.UserRoleRepo;
import cn.quantgroup.customer.rest.param.ModifyPhoneAudit;
import cn.quantgroup.customer.rest.param.ModifyPhoneFeedback;
import cn.quantgroup.customer.rest.param.ModifyPhoneQuery;
import cn.quantgroup.customer.service.IUserService;
import cn.quantgroup.customer.service.http.IHttpService;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;

import static cn.quantgroup.customer.constant.Constant.GSON;

@Slf4j
@Service("userService")
public class UserServiceImpl implements IUserService {

    @Value("${passportapi2.https}")
    private String userSysUrl;

    private final UserRepo userRepo;
    private final UserRoleRepo userRoleRepo;
    private final AuthorityRepo authorityRepo;
    private final RoleAuthorityRepo roleAuthorityRepo;
    private final IHttpService httpService;


    @Autowired
    public UserServiceImpl(UserRepo userRepo, UserRoleRepo userRoleRepo,
                           AuthorityRepo authorityRepo, RoleAuthorityRepo roleAuthorityRepo,
                           IHttpService httpService) {
        this.userRepo = userRepo;
        this.userRoleRepo = userRoleRepo;
        this.authorityRepo = authorityRepo;
        this.roleAuthorityRepo = roleAuthorityRepo;
        this.httpService = httpService;
    }

    @Override
    public UserDetails loadUserByUsername(String userName) throws UsernameNotFoundException {
        User user = userRepo.findByUsername(userName);
        if (user == null) {
            throw new UsernameNotFoundException("user: " + userName + " do not exist!");
        }
        /*
        List<UserRole> userRoles = findUserRoleByUserId(user.getId());
        if (CollectionUtils.isNotEmpty(userRoles)) {
            List<RoleAuthority> roleAuthorityList = findRoleAuthorityByRoleIds(
                    userRoles.stream().map(UserRole::getRoleId).collect(Collectors.toList()));

            if (CollectionUtils.isNotEmpty(roleAuthorityList)) {
                List<Authority> authorities = findAuthorityByAuthorityIds(
                        roleAuthorityList.stream().map(RoleAuthority::getAuthorityId).collect(Collectors.toList()));

                user.setAuthorities(authorities);
            }
        }
         */
        return user;
    }

    @Override
    public List<UserRole> findUserRoleByUserId(Long userId) {
        return userRoleRepo.findByUserId(userId);
    }

    @Override
    public List<RoleAuthority> findRoleAuthorityByRoleIds(List<Long> roleIdList) {
        return roleAuthorityRepo.findByRoleIdIn(roleIdList);
    }

    @Override
    public List<Authority> findAuthorityByAuthorityIds(List<Long> authorityIdList) {
        return authorityRepo.findByIdIn(authorityIdList);
    }

    @Override
    public String modifyPhoneQuery(ModifyPhoneQuery modifyPhoneQuery) {
        String url = userSysUrl + "/v1/user/modify/phone_no";
        Map param = GSON.fromJson(GSON.toJson(modifyPhoneQuery), Map.class);
        try {
            String response = httpService.get(url, param);
            if (StringUtils.isEmpty(response)) {
                log.error("[user][query modify phone list] 请求业务系统返回值为空,queryModifyPhone:{}", modifyPhoneQuery);
            }
            return response;
        } catch (Exception e) {
            log.error("[user][query modify phone list] 网络通讯异常,queryModifyPhone:{},ex:{}", modifyPhoneQuery, ExceptionUtils.getStackTrace(e));
            throw new NetCommunicationException(ErrorCodeEnum.NET_ERROR.getMessage());
        }
    }

    @Override
    public String modifyPhoneAudit(ModifyPhoneAudit modifyPhoneAudit) {
        String url = userSysUrl + "/v1/user/modify/phone_no/audit";
        Map param = GSON.fromJson(GSON.toJson(modifyPhoneAudit), Map.class);
        try {
            Map<String, String> header = Maps.newHashMap();
            header.put("Content-type", "application/json");
            String result = httpService.put(url, header, param, String.class);
            log.info("[user][query modify audit ] 请求业务系统返回值:{}", result);
            return result;
        } catch (Exception e) {
            log.error("[user][query modify audit ] 网络通讯异常,modifyPhoneAudit:{},ex:{}", modifyPhoneAudit, ExceptionUtils.getStackTrace(e));
            throw new NetCommunicationException(ErrorCodeEnum.NET_ERROR.getMessage());
        }
    }

    @Override
    public String modifyPhoneFeedback(ModifyPhoneFeedback modifyPhoneFeedback) {
        String url = "http://127.0.0.1:7067/test/modify/{id}/feedback";
        //String url = userSysUrl + "/v1/user/modify/phone_no/{id}/feedback";
        try {
            String id = modifyPhoneFeedback.getId();
            url = url.replace("{id}", id);
            String result = httpService.put(url, null, null, String.class);
            log.info("[user][query modify feedback ] 请求业务系统返回值:{}", result);
            return result;
        } catch (Exception e) {
            log.error("[user][query modify feedback ] 网络通讯异常,modifyPhoneFeedback:{},ex:{}", modifyPhoneFeedback, ExceptionUtils.getStackTrace(e));
            throw new NetCommunicationException(ErrorCodeEnum.NET_ERROR.getMessage());
        }
    }
}
