package cn.quantgroup.customer.rest;

import cn.quantgroup.customer.entity.User;
import cn.quantgroup.customer.enums.ErrorCodeEnum;
import cn.quantgroup.customer.model.user.AuthBean;
import cn.quantgroup.customer.rest.enums.phone.ModifyPhoneApplyStatus;
import cn.quantgroup.customer.rest.enums.phone.ModifyPhoneAuditStatus;
import cn.quantgroup.customer.rest.enums.phone.ModifyPhoneProcessingStatus;
import cn.quantgroup.customer.rest.param.phone.ModifyPhoneAudit;
import cn.quantgroup.customer.rest.param.phone.ModifyPhoneFeedback;
import cn.quantgroup.customer.rest.param.phone.ModifyPhoneQuery;
import cn.quantgroup.customer.rest.param.user.UserCombinationParam;
import cn.quantgroup.customer.rest.vo.JsonResult;
import cn.quantgroup.customer.service.IUserService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;
import java.security.Principal;
import java.time.LocalDate;

import static cn.quantgroup.customer.constant.Constant.GSON;

@Slf4j
@RestController
@RequestMapping("/user")
public class UserRest {

    private final IUserService userService;


    @Autowired
    public UserRest(IUserService userService) {
        this.userService = userService;
    }

    @PostMapping(value = "/login")
    public JsonResult login(Principal principal, HttpServletRequest request) {
        if (principal == null) {
            throw new RuntimeException("账户名或密码不正确");
        }
        User user = (User) ((UsernamePasswordAuthenticationToken) principal).getPrincipal();
        if (!user.isEnable()) {
            throw new RuntimeException("账号已被禁用");
        }
        return new JsonResult(new AuthBean(request.getSession().getId(), user));
    }

    /**
     * 修改手机号-查询工单
     *
     * @param modifyPhoneQuery
     * @return
     */
    @GetMapping(value = "/modify/phone_no")
    public JsonResult modifyPhoneQuery(ModifyPhoneQuery modifyPhoneQuery) {
        LocalDate startAt = modifyPhoneQuery.getStartAt();
        LocalDate endAt = modifyPhoneQuery.getEndAt();
        JsonResult paramErrorResult = JsonResult.buildErrorStateResult(ErrorCodeEnum.PARAM_ERROR.getMessage(), ErrorCodeEnum.PARAM_ERROR.getCode());

        if (startAt == null && endAt != null) {
            return paramErrorResult;
        }
        if (startAt != null && endAt != null && startAt.isAfter(endAt)) {
            return paramErrorResult;
        }
        String applyStatus = modifyPhoneQuery.getApplyStatus();
        if (StringUtils.isNotEmpty(applyStatus) && !ModifyPhoneApplyStatus.validate(applyStatus)) {
            return paramErrorResult;
        }
        String processingStatus = modifyPhoneQuery.getProcessingStatus();
        if (StringUtils.isNotEmpty(processingStatus) && !ModifyPhoneProcessingStatus.validate(processingStatus)) {
            return paramErrorResult;
        }
        String response = userService.modifyPhoneQuery(modifyPhoneQuery);
        if (StringUtils.isEmpty(response)) {
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.RETURN_ERROR.getMessage(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }
        return GSON.fromJson(response, JsonResult.class);
    }

    /**
     * 修改手机号-人工审核
     *
     * @param modifyPhoneAudit
     * @return
     */
    @PostMapping(value = "/modify/phone_no/audit")
    public JsonResult modifyPhoneAudit(@Valid ModifyPhoneAudit modifyPhoneAudit) {
        if (!ModifyPhoneAuditStatus.validate(modifyPhoneAudit.getApplyStatus(), modifyPhoneAudit.getApplyStatusReason())) {
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.PARAM_ERROR.getMessage(), ErrorCodeEnum.PARAM_ERROR.getCode());
        }
        String response = userService.modifyPhoneAudit(modifyPhoneAudit);
        if (StringUtils.isEmpty(response)) {
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.RETURN_ERROR.getMessage(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }
        return GSON.fromJson(response, JsonResult.class);
    }

    /**
     * 修改手机号-反馈跟进
     *
     * @param modifyPhoneFeedback
     * @return
     */
    @PostMapping(value = "/modify/phone_no/feedback")
    public JsonResult modifyPhoneFeedback(@Valid ModifyPhoneFeedback modifyPhoneFeedback) {
        String response = userService.modifyPhoneFeedback(modifyPhoneFeedback);
        if (StringUtils.isEmpty(response)) {
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.RETURN_ERROR.getMessage(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }
        return GSON.fromJson(response, JsonResult.class);
    }


    /**
     * 用户综合查询
     *
     * @param combinationParam
     * @return
     */
    @GetMapping(value = "/info")
    public JsonResult findUserCombination(UserCombinationParam combinationParam) {
        return userService.findUserCombination(combinationParam);

    }


}
