package cn.quantgroup.customer.rest;

import cn.quantgroup.customer.enums.ErrorCodeEnum;
import cn.quantgroup.customer.rest.param.repay.RepayOrderInfoQuery;
import cn.quantgroup.customer.rest.param.repay.RepayOrderQuery;
import cn.quantgroup.customer.rest.vo.JsonResult;
import cn.quantgroup.customer.service.IRepayService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.validation.Valid;

import java.util.StringJoiner;

import static cn.quantgroup.customer.constant.Constant.GSON;

@Slf4j
@RestController
@RequestMapping("/repay")
public class RepayRest {

    private final IRepayService repayService;

    @Autowired
    public RepayRest(IRepayService repayService) {
        this.repayService = repayService;
    }

    @GetMapping("/repayment/repay_orders")
    public JsonResult repayOrders(String phoneNo, String idNo) {
        if (StringUtils.isEmpty(phoneNo) && StringUtils.isEmpty(idNo)) {
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.PARAM_ERROR.getMessage(), ErrorCodeEnum.PARAM_ERROR.getCode());
        }
        RepayOrderQuery repayOrderQuery = new RepayOrderQuery();
        repayOrderQuery.setPhoneNo(phoneNo);
        repayOrderQuery.setIdNo(idNo);
        String response = repayService.findRepayOrders(repayOrderQuery);
        return GSON.fromJson(response, JsonResult.class);
    }

    /**
     * 还款信息
     *
     * @param repayOrderQuery
     * @return
     */
    @GetMapping("/info")
    public JsonResult queryRepayInfo(@Valid RepayOrderInfoQuery repayOrderQuery, BindingResult bindingResult) {
        log.info("[查询还款信息],请求参数:repayOrderQuery={}", repayOrderQuery);
        if (bindingResult.hasErrors()) {
            StringJoiner errorStr = new StringJoiner(";");
            bindingResult.getFieldErrors().forEach(fieldError -> errorStr.add(fieldError.getDefaultMessage()));
            log.info("[查询还款信息],请求参数验证异常:errorStr={}", errorStr);
            return JsonResult.buildErrorStateResult(errorStr.toString(), ErrorCodeEnum.PARAM_ERROR.getCode());
        }
        return repayService.queryRepayInfo(repayOrderQuery);
    }

    /**
     * 查询还款操作记录
     *
     * @param businessFlowNo
     * @return
     */
    @GetMapping("/oplog/{businessFlowNo}")
    public JsonResult queryRepayOptRecords(@PathVariable String businessFlowNo) {
        log.info("[查询还款操作记录],请求参数:businessFlowNo={}", businessFlowNo);
        return repayService.queryRepayOptRecords(businessFlowNo);
    }
}
