package cn.quantgroup.customer.service.impl;

import cn.quantgroup.customer.constant.Constant;
import cn.quantgroup.customer.enums.ErrorCodeEnum;
import cn.quantgroup.customer.exception.BusinessException;
import cn.quantgroup.customer.model.xyqbuser.UserBasicInfo;
import cn.quantgroup.customer.rest.param.ordermapping.OrderQueryParam;
import cn.quantgroup.customer.rest.param.repay.RepayOrderInfoQuery;
import cn.quantgroup.customer.rest.param.repay.RepayOrderQuery;
import cn.quantgroup.customer.rest.param.user.UserCombinationParam;
import cn.quantgroup.customer.rest.param.user.UserQueryParam;
import cn.quantgroup.customer.rest.vo.JsonResult;
import cn.quantgroup.customer.rest.vo.repay.RepayInfoResult;
import cn.quantgroup.customer.rest.vo.repay.RepayOptRecord;
import cn.quantgroup.customer.rest.vo.repay.XyqbRepayInfo;
import cn.quantgroup.customer.service.IRepayService;
import cn.quantgroup.customer.service.IUserService;
import cn.quantgroup.customer.service.IXyqbService;
import cn.quantgroup.customer.util.JSONTools;
import cn.quantgroup.user.retbean.XUser;
import cn.quantgroup.user.vo.UserSysResult;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.lang.reflect.Type;
import java.util.List;
import java.util.Objects;

@Slf4j
@Service("repayService")
public class RepayServiceImpl implements IRepayService {

    private final IUserService userService;
    private final UserSdkImpl userSdk;
    private final IXyqbService xyqbService;

    @Autowired
    public RepayServiceImpl(IUserService userService, UserSdkImpl userSdk, IXyqbService xyqbService) {
        this.userService = userService;
        this.userSdk = userSdk;
        this.xyqbService = xyqbService;
    }

    @Override
    public String findRepayOrders(RepayOrderQuery repayOrderQuery) {
        String phoneNo = repayOrderQuery.getPhoneNo();
        String idNo = repayOrderQuery.getIdNo();
        if (StringUtils.isEmpty(phoneNo) && StringUtils.isEmpty(idNo)) {
            throw new BusinessException(ErrorCodeEnum.PARAM_ERROR);
        }
        if (StringUtils.isEmpty(phoneNo)) {
            String uuid = userService.findUuidByIdNo(idNo);
            UserSysResult<XUser> userUserSysResult = userSdk.getService().findUserByUuid(uuid);
            if (!userUserSysResult.isSuccess()) {
                throw new BusinessException(ErrorCodeEnum.RETURN_ERROR);
            }
            Long userId = userUserSysResult.getData().getId();
            repayOrderQuery.setUserId(userId);
        }

        return xyqbService.findRepayOrders(repayOrderQuery);
    }

    @Override
    public JsonResult<RepayInfoResult> queryRepayInfo(RepayOrderInfoQuery repayOrderQuery) {

        final String logPre = "[queryRepayInfo] 查询还款信息";
        // 参数校验
        if (Objects.isNull(repayOrderQuery)) {
            log.error("{} 请求参数 repayOrderQuery 为空", logPre);
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.PARAM_ERROR.getMessage(), ErrorCodeEnum.PARAM_ERROR.getCode());
        }

        if (Objects.isNull(repayOrderQuery.getLoanId())) {
            log.error("{} 请求参数 loanId 不能为空，repayOrderQuery:{}", logPre, repayOrderQuery);
            return JsonResult.buildErrorStateResult("借款订单号不能为空", ErrorCodeEnum.PARAM_ERROR.getCode());
        }

        Long loanId = repayOrderQuery.getLoanId();
        // 处理用户信息
        OrderQueryParam orderQueryParam = new OrderQueryParam();
        orderQueryParam.setLoanId(loanId);
        JsonResult<UserBasicInfo> userJsonResult = userService.findUserInfoByOrderParam(orderQueryParam);
        log.info("{} 查询用户信息 userJsonResult={} ", logPre, userJsonResult);
        if (Objects.isNull(userJsonResult) || !userJsonResult.isSuccess()) {
            log.error("{} 获取用户信息失败 orderQueryParam={},userJsonResult={}", logPre, orderQueryParam, userJsonResult);
            return JsonResult.buildErrorStateResult("获取用户信息失败", ErrorCodeEnum.RETURN_ERROR.getCode());
        }
        UserBasicInfo userbasicinfo;
        if (Objects.isNull(userbasicinfo = userJsonResult.getData())) {
            log.warn("{} 不存在对应用户 orderQueryParam={},userJsonResult={}", logPre, orderQueryParam, userJsonResult);
            return JsonResult.buildSuccessResult("不存在对应用户信息", null);
        }

        //处理还款信息
        JsonResult<List<XyqbRepayInfo>> xyqbRepayInfoResult = xyqbService.queryRepayInfo(repayOrderQuery);
        log.info("{} 信用钱包返回信息 xyqbRepayInfoResult:{}", logPre, xyqbRepayInfoResult);
        if (Objects.isNull(xyqbRepayInfoResult)) {
            log.error("{} 返回为空 repayOrderQuery={}", logPre, repayOrderQuery);
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.RETURN_ERROR.getMessage(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }
        if (!xyqbRepayInfoResult.isSuccess()) {
            log.error("{} 信息异常 repayOrderQuery={},xyqbRepayInfoResult={}", logPre, repayOrderQuery, xyqbRepayInfoResult);
            return JsonResult.buildErrorStateResult(xyqbRepayInfoResult.getMsg(), xyqbRepayInfoResult.getCode());
        }

        RepayInfoResult repayInfoResult = new RepayInfoResult();
        repayInfoResult.setUserInfo(userbasicinfo);
        repayInfoResult.setRepaymentList(xyqbRepayInfoResult.getData());
        log.info("{}，返回结果 repayInfoResult={}", logPre, repayInfoResult);
        return JsonResult.buildSuccessResult("处理成功", repayInfoResult);
    }

    @Override
    public JsonResult<List<RepayOptRecord>> queryRepayOptRecords(String businessFlowNo) {
        final String logPre = "[queryRepayOptRecords] 查询还款操作记录";
        if (StringUtils.isBlank(businessFlowNo)) {
            log.error("{} 请求参数 流水号为空", logPre);
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.PARAM_ERROR.getMessage(), ErrorCodeEnum.PARAM_ERROR.getCode());
        }

        //发起xyqb查询
        JsonResult<List<RepayOptRecord>> repayOptRecordResult = xyqbService.queryRepayOptRecords(businessFlowNo);
        log.info("{} 信用钱包返回信息 repayOptRecordResult:{}", logPre, repayOptRecordResult);
        if (Objects.isNull(repayOptRecordResult)) {
            log.error("{} 返回为空 businessFlowNo={}", logPre, businessFlowNo);
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.RETURN_ERROR.getMessage(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }
        if (!repayOptRecordResult.isSuccess()) {
            log.error("{} 信息异常 businessFlowNo={},repayOptRecordResult={}", logPre, businessFlowNo, repayOptRecordResult);
            return JsonResult.buildErrorStateResult(repayOptRecordResult.getMsg(), repayOptRecordResult.getCode());
        }

        log.info("{}，返回信息 repayOptRecordResult={}", repayOptRecordResult);
        return repayOptRecordResult;
    }
}
