package cn.quantgroup.customer.rest;

import cn.quantgroup.customer.enums.ErrorCodeEnum;
import cn.quantgroup.customer.rest.enums.phone.ModifyPhoneApplyStatus;
import cn.quantgroup.customer.rest.enums.phone.ModifyPhoneAuditStatus;
import cn.quantgroup.customer.rest.enums.phone.ModifyPhoneProcessingStatus;
import cn.quantgroup.customer.rest.param.phone.ModifyPhoneAudit;
import cn.quantgroup.customer.rest.param.phone.ModifyPhoneFeedback;
import cn.quantgroup.customer.rest.param.phone.ModifyPhoneQuery;
import cn.quantgroup.customer.rest.param.user.LoginParam;
import cn.quantgroup.customer.rest.vo.JsonResult;
import cn.quantgroup.customer.rest.vo.user.AuthUserVo;
import cn.quantgroup.customer.service.IUserService;
import cn.quantgroup.riskcontrol.model.AuthenticationUserDetail;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import javax.validation.Valid;
import java.time.LocalDate;

import static cn.quantgroup.customer.constant.Constant.*;

@Slf4j
@RestController
@RequestMapping("/user")
public class UserRest {

    private final IUserService userService;
    private final AuthenticationManager authenticationManager;

    @Autowired
    public UserRest(AuthenticationManager authenticationManager, IUserService userService) {
        this.authenticationManager = authenticationManager;
        this.userService = userService;
    }

    @PostMapping(value = "/login")
    public JsonResult login(@Valid @ModelAttribute LoginParam loginParam, HttpServletRequest request) {
        UsernamePasswordAuthenticationToken authRequest = new UsernamePasswordAuthenticationToken(loginParam.getUsername(), loginParam.getPassword());
        Authentication authentication = authenticationManager.authenticate(authRequest);
        SecurityContextHolder.getContext().setAuthentication(authentication);
        HttpSession session = request.getSession();
        session.setAttribute(SPRING_SECURITY_CONTEXT, SecurityContextHolder.getContext());

        if (authentication != null && authentication.isAuthenticated()) {
            AuthenticationUserDetail principal = (AuthenticationUserDetail) authentication.getPrincipal();
            AuthUserVo authUserVo = AuthUserVo.parse(principal);
            return JsonResult.buildSuccessResult(LOGIN_SUCCESS, authUserVo);
        }
        return JsonResult.buildErrorStateResult(LOGIN_FAIL, null);
    }

    /**
     * 修改手机号-查询工单
     *
     * @param modifyPhoneQuery
     * @return
     */
    @GetMapping(value = "/modify/phone_no")
    public JsonResult modifyPhoneQuery(ModifyPhoneQuery modifyPhoneQuery) {
        LocalDate startAt = modifyPhoneQuery.getStartAt();
        LocalDate endAt = modifyPhoneQuery.getEndAt();
        JsonResult paramErrorResult = JsonResult.buildErrorStateResult(ErrorCodeEnum.PARAM_ERROR.getMessage(), ErrorCodeEnum.PARAM_ERROR.getCode());

        if (startAt == null && endAt != null) {
            return paramErrorResult;
        }
        if (startAt != null && endAt != null && startAt.isAfter(endAt)) {
            return paramErrorResult;
        }
        String applyStatus = modifyPhoneQuery.getApplyStatus();
        if (StringUtils.isNotEmpty(applyStatus) && !ModifyPhoneApplyStatus.validate(applyStatus)) {
            return paramErrorResult;
        }
        String processingStatus = modifyPhoneQuery.getProcessingStatus();
        if (StringUtils.isNotEmpty(processingStatus) && !ModifyPhoneProcessingStatus.validate(processingStatus)) {
            return paramErrorResult;
        }
        String response = userService.modifyPhoneQuery(modifyPhoneQuery);
        if (StringUtils.isEmpty(response)) {
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.RETURN_ERROR.getMessage(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }
        return GSON.fromJson(response, JsonResult.class);
    }

    /**
     * 修改手机号-人工审核
     *
     * @param modifyPhoneAudit
     * @return
     */
    @PostMapping(value = "/modify/phone_no/audit")
    public JsonResult modifyPhoneAudit(@Valid ModifyPhoneAudit modifyPhoneAudit) {
        if (!ModifyPhoneAuditStatus.validate(modifyPhoneAudit.getApplyStatus(), modifyPhoneAudit.getApplyStatusReason())) {
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.PARAM_ERROR.getMessage(), ErrorCodeEnum.PARAM_ERROR.getCode());
        }
        String response = userService.modifyPhoneAudit(modifyPhoneAudit);
        if (StringUtils.isEmpty(response)) {
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.RETURN_ERROR.getMessage(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }
        return GSON.fromJson(response, JsonResult.class);
    }

    /**
     * 修改手机号-反馈跟进
     *
     * @param modifyPhoneFeedback
     * @return
     */
    @PostMapping(value = "/modify/phone_no/feedback")
    public JsonResult modifyPhoneFeedback(@Valid ModifyPhoneFeedback modifyPhoneFeedback) {
        String response = userService.modifyPhoneFeedback(modifyPhoneFeedback);
        if (StringUtils.isEmpty(response)) {
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.RETURN_ERROR.getMessage(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }
        return GSON.fromJson(response, JsonResult.class);
    }


}
