package cn.quantgroup.customer.service.impl;

import cn.quantgroup.customer.model.order.*;
import cn.quantgroup.customer.moice.enums.BusinessType;
import cn.quantgroup.customer.moice.enums.LoanProgress;
import cn.quantgroup.customer.moice.service.IXyqbRepaymentPlanService;
import cn.quantgroup.customer.moice.vo.LoanOrderVo;
import cn.quantgroup.customer.moice.vo.StatusMap;
import cn.quantgroup.customer.moice.vo.UserOrders;
import cn.quantgroup.customer.moice.model.XyqbLoanApplicationHistory;
import cn.quantgroup.customer.moice.model.XyqbLoanApplicationManifestHistory;
import cn.quantgroup.customer.moice.model.XyqbRepaymentPlan;
import cn.quantgroup.customer.moice.repo.IFundingCorpRepository;
import cn.quantgroup.customer.moice.service.IXyqbLoanApplicationHistoryService;
import cn.quantgroup.customer.moice.service.IXyqbLoanApplicationManifestHistoryService;
import cn.quantgroup.customer.rest.param.applyorder.ApplyOrderQuery;
import cn.quantgroup.customer.rest.vo.JsonResult;
import cn.quantgroup.customer.service.IFileService;
import cn.quantgroup.customer.service.IIceService;
import cn.quantgroup.customer.service.http.IHttpService;
import cn.quantgroup.customer.util.JSONTools;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import javax.servlet.http.HttpServletResponse;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;

@Slf4j
@Service("iceService")
public class IceServiceImpl implements IIceService {

    @Value("${mo.ice.http}")
    private String iceUrl;

    @Value("${mo-sidecar.http}")
    private String sidecarUrl;

    private final IHttpService httpService;

    @Autowired
    private IFileService fileService;

    @Autowired
    private IXyqbLoanApplicationHistoryService loanApplicationHistoryService;

    @Autowired
    private IXyqbLoanApplicationManifestHistoryService manifestHistoryService;

    @Autowired
    private IFundingCorpRepository fundingCorpRepository;

    @Autowired
    private IXyqbRepaymentPlanService repaymentPlanService;

    @Autowired
    public IceServiceImpl(IHttpService httpService) {
        this.httpService = httpService;
    }

    @Override
    public JsonResult<ApplyOrder> findApplyOrders(ApplyOrderQuery applyOrderQuery) {
        String logPre = "IceService.findApplyOrders";
        log.info("{} 申请订单查询 applyOrderQuery={}", logPre, applyOrderQuery);
        String orderNo = applyOrderQuery.getOrderNo();
        Long loanId = applyOrderQuery.getLoanId();
        Long userId = applyOrderQuery.getUserId();
        String url = iceUrl + "/ex/customer_sys/query/applyOrder";
        Map<String, Object> param = Maps.newHashMap();
        if (StringUtils.isNotEmpty(orderNo)) {
            param.put("orderNo", orderNo);
        }
        if (Objects.nonNull(userId)) {
            param.put("userId", userId.toString());
        }
        if (Objects.nonNull(loanId)) {
            param.put("loanId", loanId.toString());
        }
        Map<String, String> header = Maps.newHashMap();
        header.put("Content-Type", "application/x-www-form-urlencoded");
        String result = httpService.post(url, header, param);
        log.info("{} 用户申请订单列表 url={}, header={},param={},result={}", logPre, url, header, param, result);
        if (StringUtils.isBlank(result)) {
            log.error("{} 调用信用钱包失败 url={}, header={},param={},result={}", logPre, url, header, param, result);
            return JsonResult.buildErrorStateResult("申请订单查询失败", null);
        }

        TypeReference<JsonResult<ApplyOrder>> typeToken = new TypeReference<JsonResult<ApplyOrder>>() {
        };
        JsonResult<ApplyOrder> jsonResult = JSONTools.deserialize(result, typeToken);
        return jsonResult;
    }

    @Override
    public JsonResult<LoanOrder> findLoanOrder(Long loanId) {
        String logPre = "IceService.findLoanOrder";
        log.info("{} 提现订单详情 loanId={}", logPre, loanId);
        String url = iceUrl + "/ex/customer_sys/loan/detail";
        if (Objects.isNull(loanId)) {
            log.error("{} 借款订单号为空 orderNo={}", logPre, loanId);
            return JsonResult.buildErrorStateResult("借款订单号为空", null);
        }
        Map<String, Object> param = Maps.newHashMap();
        param.put("loanId", loanId.toString());

        Map<String, String> header = Maps.newHashMap();
        header.put("Content-Type", "application/x-www-form-urlencoded");
        String result = null;
        try {
            result = httpService.post(url, header, param);
            log.info("{} 提现订单详情 loanId={},result:{}", logPre, loanId, result);
        } catch (Exception e) {
            log.error("{} 远程调用异常 url={},param={}", logPre, url, param, e);
            return JsonResult.buildErrorStateResult("网络异常", null);
        }

        if (StringUtils.isBlank(result)) {
            log.error("{} 调用信用钱包失败 url={}, header={},param={},result={}", logPre, url, header, param, result);
            return JsonResult.buildErrorStateResult("提现订单详情查询失败", null);
        }
        TypeReference<JsonResult<LoanOrder>> typeToken = new TypeReference<JsonResult<LoanOrder>>() {
        };
        JsonResult<LoanOrder> jsonResult = JSONTools.deserialize(result, typeToken);

        return jsonResult;
    }

    @Override
    public JsonResult<List<FlowNode>> findFlowChart(String orderNo) {
        String logPre = "IceService.findFlowChart";
        log.info("{} 流程日志查询 orderNo={}", logPre, orderNo);
        String url = iceUrl + "/ex/customer_sys/query/flow";
        if (StringUtils.isEmpty(orderNo)) {
            log.error("{} 申请订单号为空 orderNo={}", logPre, orderNo);
            return JsonResult.buildErrorStateResult("申请订单号为空", null);
        }
        Map<String, Object> param = Maps.newHashMap();
        param.put("orderNo", orderNo);

        Map<String, String> header = Maps.newHashMap();
        header.put("Content-Type", "application/x-www-form-urlencoded");
        String result = null;
        try {
            result = httpService.post(url, header, param);
            log.info("{} 流程日志查询 orderNo={},result:{}", logPre, orderNo, result);
        } catch (Exception e) {
            log.error("{} 通讯异常 url={},param={}", logPre, url, param, e);
            return JsonResult.buildErrorStateResult("通讯异常", null);
        }

        if (StringUtils.isBlank(result)) {
            log.error("{} 调用信用钱包失败 url={}, header={},param={},result={}", logPre, url, header, param, result);
            return JsonResult.buildErrorStateResult("流程日志查询失败", null);
        }
        TypeReference<JsonResult<List<FlowNode>>> typeToken = new TypeReference<JsonResult<List<FlowNode>>>() {
        };
        JsonResult<List<FlowNode>> jsonResult = JSONTools.deserialize(result, typeToken);
        return jsonResult;

    }

    @Override
    public JsonResult<List<SettleStatus>> findSettleFlag(List<Long> loanIds) {
        String logPre = "IceService.findSettleFlag";
        log.info("{} 结清状态查询 loanIds={}", logPre, loanIds);
        String url = sidecarUrl + "/middle_office/settle_proof/status";
        if (CollectionUtils.isEmpty(loanIds)) {
            log.error("{} 订单号全为空 loanIds={}", logPre, loanIds);
            return JsonResult.buildErrorStateResult("申请订单号为空", null);
        }
        Map<String, Object> param = Maps.newHashMap();
        param.put("loanIds", StringUtils.join(loanIds, ","));

        Map<String, String> header = Maps.newHashMap();
        header.put("Content-Type", "application/x-www-form-urlencoded");
        String result;
        try {
            result = httpService.get(url, header, param);
            log.info("{} 结清状态查询 loanIds={},result:{}", logPre, loanIds, result);
        } catch (Exception e) {
            log.error("{} 通讯异常 url={},param={}", logPre, url, param, e);
            return JsonResult.buildErrorStateResult("通讯异常", null);
        }

        if (StringUtils.isBlank(result)) {
            log.error("{} 调用失败 url={}, header={},param={},result={}", logPre, url, header, param, result);
            return JsonResult.buildErrorStateResult("结清状态查询失败", null);
        }
        TypeReference<JsonResult<List<SettleStatus>>> typeToken = new TypeReference<JsonResult<List<SettleStatus>>>() {
        };
        JsonResult<List<SettleStatus>> jsonResult = JSONTools.deserialize(result, typeToken);
        return jsonResult;
    }

    @Override
    public JsonResult applySettle(Long loanId) {
        String logPre = "IceService.applySettle";
        log.info("{} 申请结清下载 loanIds={}", logPre, loanId);
        String url = sidecarUrl + "/middle_office/settle_proof/apply";
        if (loanId == null) {
            log.error("{} 申请订单号loanId为空", logPre);
            return JsonResult.buildErrorStateResult("申请订单号为空", null);
        }
        Map<String, Object> param = Maps.newHashMap();
        param.put("loanId", String.valueOf(loanId));

        Map<String, String> header = Maps.newHashMap();
        header.put("Content-Type", "application/x-www-form-urlencoded");
        String result;
        try {
            result = httpService.post(url, header, param);
            log.info("{} 申请结清下载 loanId={},result:{}", logPre, loanId, result);
        } catch (Exception e) {
            log.error("{} 通讯异常 url={},param={}", logPre, url, param, e);
            return JsonResult.buildErrorStateResult("通讯异常", null);
        }

        if (StringUtils.isBlank(result)) {
            log.error("{} 调用失败 url={}, header={},param={},result={}", logPre, url, header, param, result);
            return JsonResult.buildErrorStateResult("申请结清下载失败", null);
        }
        TypeReference<JsonResult> typeToken = new TypeReference<JsonResult>() {
        };
        return JSONTools.deserialize(result, typeToken);
    }

    @Override
    public JsonResult downloadSettle( HttpServletResponse response, Long loanId) {
        String logPre = "IceService.downloadSettle";
        log.info("{} 结清证明下载 loanIds={}", logPre, loanId);
        String url = sidecarUrl + "/middle_office/settle_proof/download";
        if (loanId == null) {
            log.error("{} 申请订单号loanId为空 loanId", logPre);
            return JsonResult.buildErrorStateResult("申请订单号为空", null);
        }
        Map<String, Object> param = Maps.newHashMap();
        param.put("loanId", String.valueOf(loanId));

        Map<String, String> header = Maps.newHashMap();
        header.put("Content-Type", "application/x-www-form-urlencoded");
        byte[] result;
        try {
            result = httpService.getByte(url, header, param);
            log.info("{} 结清证明下载 loanId={},result:{}", logPre, loanId, result);
        } catch (Exception e) {
            log.error("{} 通讯异常 url={},param={}", logPre, url, param, e);
            return JsonResult.buildErrorStateResult("通讯异常", null);
        }

        if (result == null) {
            log.error("{} 调用失败 url={}, header={},param={},result={}", logPre, url, header, param, result);
            return JsonResult.buildErrorStateResult("结清证明下载失败,返回值为空", null);
        }
        return fileService.outputFile(response,  System.currentTimeMillis() + String.valueOf(loanId) +"settle.pdf", result);
    }

    /**
     * 用户提现信息查询
     * @param applyOrderQuery
     * @return
     */
    @Override
    public JsonResult<ApplyOrder> findUserCashApplyOrders(ApplyOrderQuery applyOrderQuery) {
        JsonResult<ApplyOrder> jsonResult = new JsonResult<>();
        try {
            String logPre = "IceService.findUserCashApplyOrders";
            log.info("{} 申请订单查询 applyOrderQuery={}", logPre, applyOrderQuery);
            Long userId = applyOrderQuery.getUserId();
            log.info("queryUserCashApplyOrders:begin");
            UserOrders userOrders = queryUserCashApplyOrders(userId+"");
            log.info("queryUserCashApplyOrders:end,userOrders="+userOrders);
            log.info("转化为json字符串begin");
            ObjectMapper mapper = new ObjectMapper();
            String result = mapper.writeValueAsString(userOrders);
            log.info("转化为json字符串,result="+result);
//            TypeReference<JsonResult<ApplyOrder>> typeToken = new TypeReference<JsonResult<ApplyOrder>>() {
//            };
//            log.info("转化为json字符串,typeToken="+typeToken);
//            jsonResult = JSONTools.deserialize(result, typeToken);
            jsonResult = JsonResult.buildSuccessResult("请求成功",JSONTools.deserialize(result,ApplyOrder.class));
            log.info("转化为json字符串,jsonResult="+jsonResult);
            return jsonResult;
        } catch (JsonProcessingException e) {
            e.printStackTrace();
        }
        return jsonResult;
    }

    /**
     * 用户提现信息查询
     * @param userId
     * @return
     */
    private UserOrders queryUserCashApplyOrders(String userId){
        log.info("[客服系统] 用户申请订单查询, userId:{},userId"+userId);
        UserOrders orders = new UserOrders();
        List<LoanOrderVo> voList = new ArrayList<>();
        List<LoanOrderVo> voList1 = new ArrayList<>();
        List<XyqbLoanApplicationHistory> activedOldOrder = null;
        if (StringUtils.isNotBlank(userId)) {
            // 查询出18年迁移前的出老订单.
            log.info("查询出18年迁移前的出老订单start");
            activedOldOrder = loanApplicationHistoryService.findActivedOldOrder(userId);
            log.info("查询出18年迁移前的出老订单end,activedOldOrder="+activedOldOrder);
        }
        int i = 1;
        if (org.springframework.util.CollectionUtils.isEmpty(activedOldOrder)) {
            log.info("111111");
            orders.setActiveOrders(voList);
            orders.setInvalidOrders(voList1);
            log.info("222222");
            return orders;
        }
        if (!org.springframework.util.CollectionUtils.isEmpty(activedOldOrder)) {
            log.info("3333333333");
            for (XyqbLoanApplicationHistory loan : activedOldOrder) {
                log.info("4444444444444");
                LoanOrderVo lo = buildOldLoan(loan);
                lo.setNum(i);
                voList.add(lo);
                i++;
            }
        }
        log.info("voList1排序start：voList1="+voList1);
        voList1.sort((o1, o2) -> o2.getUpdateAt().compareTo(o1.getUpdateAt()));
        log.info("voList1排序end：voList1="+voList1);
        log.info("voList排序start：voList="+voList);
        voList.sort((o1, o2) -> o2.getUpdateAt().compareTo(o1.getUpdateAt()));
        log.info("voList排序end：voList="+voList);
        orders.setInvalidOrders(voList1);
        orders.setActiveOrders(voList);
        return orders;
    }

    private LoanOrderVo buildOldLoan(XyqbLoanApplicationHistory loan) {

        LoanOrderVo lo = new LoanOrderVo();
        log.info("manifestHistoryService："+manifestHistoryService);
        XyqbLoanApplicationManifestHistory manifest = manifestHistoryService.findByLoanApplicationHistoryId(loan.getId());
        log.info("repaymentPlanService："+manifestHistoryService);
        List<XyqbRepaymentPlan> plans = repaymentPlanService.findByLoanApplicationHistoryId(loan.getId());
        lo.setAmount(manifest.getContractLoanAmount());
        if (loan.getActive()) {
            lo.setLoanStatus(StatusMap.mapLoanStatus(LoanProgress.CHECK_REPAYMENT_PLAN));
        } else {
            lo.setLoanStatus(StatusMap.mapLoanStatus(LoanProgress.LOAN_COMPLETE));
        }
        lo.setApplyStatus(StatusMap.DEFAULT_STATUS);
        lo.setPaidAt(manifest.getLoanPaidAt());
        if (!org.springframework.util.CollectionUtils.isEmpty(plans)) {
            lo.setTermRepayment(plans.get(0).getRequiredRepayment());
        } else {
            lo.setTermRepayment(manifest.getPrincipalAndInterestPerTerm());
        }
        lo.setApplyAt(loan.getCreatedAt());
        lo.setApplyOrderNo(String.valueOf(loan.getId()));
        lo.setShowBtn(true);
        lo.setLoanId(loan.getId());
        String channelName = fundingCorpRepository.findChannelName(loan.getCreatedFrom());
        lo.setChannelName(channelName == null ? loan.getCreatedFrom().toString() : channelName.concat("(").concat(loan.getCreatedFrom().toString()).concat(")"));
        lo.setUserId(loan.getUserId());
        lo.setProductDesc(BusinessType.CASH.getDesc());
        lo.setUpdateAt(manifest.getCreatedAt());
        return lo;
    }

}
