package cn.quantgroup.customer.service.impl;

import cn.quantgroup.customer.constant.Constant;
import cn.quantgroup.customer.enums.ErrorCodeEnum;
import cn.quantgroup.customer.model.xyqbuser.UserBasicInfo;
import cn.quantgroup.customer.rest.param.card.ModifyCardsQuery;
import cn.quantgroup.customer.rest.param.card.ModifyReservePhone;
import cn.quantgroup.customer.rest.param.card.UnBindCardModel;
import cn.quantgroup.customer.rest.param.user.UserCombinationParam;
import cn.quantgroup.customer.rest.param.user.UserQueryParam;
import cn.quantgroup.customer.rest.vo.JsonResult;
import cn.quantgroup.customer.rest.vo.card.CardResult;
import cn.quantgroup.customer.rest.vo.card.XyqbCardRelation;
import cn.quantgroup.customer.rest.vo.card.XyqbCardsInfo;
import cn.quantgroup.customer.service.ICardService;
import cn.quantgroup.customer.service.IUserService;
import cn.quantgroup.customer.service.IXyqbService;
import cn.quantgroup.customer.util.JSONTools;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.lang.reflect.Type;
import java.util.List;
import java.util.Objects;

/**
 * @author yangjun
 * @Date 2020/3/9 10:52
 * @Desc
 * @Update
 */
@Slf4j
@Service("cardService")
public class CardServiceImpl implements ICardService {
    @Autowired
    private IXyqbService xyqbService;
    @Autowired
    private IUserService userService;


    @Override
    public JsonResult<CardResult> queryCardLists(ModifyCardsQuery cardsQuery) {
        final String logPre = "[queryCardLists] 查询用户绑卡信息";
        if (Objects.isNull(cardsQuery)) {
            log.error("{} 请求参数 cardsQuery 为空", logPre);
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.PARAM_ERROR.getMessage(), ErrorCodeEnum.PARAM_ERROR.getCode());
        }

        if (StringUtils.isBlank(cardsQuery.getPhoneNo()) && Objects.isNull(cardsQuery.getUserId())) {
            log.error("{} 请求参数 phoneNo 和 userId 同时为空 cardsQuery={}", logPre, cardsQuery);
            return JsonResult.buildErrorStateResult("phoneNo 和 userId 不能同时为空", ErrorCodeEnum.PARAM_ERROR.getCode());
        }

        UserQueryParam userQueryParam = JSONTools.convert(cardsQuery, UserQueryParam.class);
        JsonResult<UserBasicInfo> userJsonResult = userService.findUserInfo(userQueryParam);
        log.info("{} 查询用户信息 userJsonResult={} ", logPre, userJsonResult);
        if (Objects.isNull(userJsonResult) || !userJsonResult.isSuccess()) {
            log.error("{} 获取用户信息失败 userQueryParam={},userJsonResult={}", logPre, userQueryParam, userJsonResult);
            return JsonResult.buildErrorStateResult("获取用户信息失败", ErrorCodeEnum.RETURN_ERROR.getCode());
        }
        UserBasicInfo userbasicinfo;
        if (Objects.isNull(userbasicinfo = userJsonResult.getData())) {
            log.warn("{} 不存在对应用户 userQueryParam={},userJsonResult={}", logPre, userQueryParam, userJsonResult);
            return JsonResult.buildSuccessResult("不存在对应用户信息", null);

        }

        //发起xyqb查询
        JsonResult<List<XyqbCardsInfo>> xyqbCardInfoResult = xyqbService.queryCardLists(userbasicinfo.getUserId());
        log.info("{} 信用钱包返回信息 xyqbCardInfoResult:{}", logPre, xyqbCardInfoResult);
        if (Objects.isNull(xyqbCardInfoResult)) {
            log.error("{} 返回为空 userId={}", logPre, userbasicinfo.getUserId());
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.RETURN_ERROR.getMessage(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }
        if (!xyqbCardInfoResult.isSuccess()) {
            log.error("{} 信息异常 userId={},xyqbCardInfoResult={}", logPre, userbasicinfo.getUserId(), xyqbCardInfoResult);
            return JsonResult.buildErrorStateResult(xyqbCardInfoResult.getMsg(), xyqbCardInfoResult.getCode());
        }

        CardResult cardResult = new CardResult();
        cardResult.setUserInfo(userbasicinfo);
        cardResult.setBindCardInfo(xyqbCardInfoResult.getData());
        log.info("{} 返回cardResult={}", logPre, cardResult);
        return JsonResult.buildSuccessResult("处理成功", cardResult);
    }


    /**
     * 查询卡关联信息
     *
     * @param bankId
     * @return
     */
    @Override
    public JsonResult<List<XyqbCardRelation>> queryCardRelation(Long bankId) {
        final String logPre = "[queryCardRelation] 查询卡关联信息";
        if (Objects.isNull(bankId)) {
            log.error("{} 请求参数 bankId 为空", logPre);
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.PARAM_ERROR.getMessage(), ErrorCodeEnum.PARAM_ERROR.getCode());
        }

        //发起xyqb查询
        JsonResult<List<XyqbCardRelation>> xyqbCardRelationResult = xyqbService.queryCardRelation(bankId);
        log.info("{} 信用钱包返回信息 xyqbCardRelationResult:{}", logPre, xyqbCardRelationResult);
        if (Objects.isNull(xyqbCardRelationResult)) {
            log.error("{} 返回为空 bankId={}", logPre, bankId);
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.RETURN_ERROR.getMessage(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }
        if (!xyqbCardRelationResult.isSuccess()) {
            log.error("{} 信息异常 bankId={},xyqbCardRelationResult={}", logPre, bankId, xyqbCardRelationResult);
            return JsonResult.buildErrorStateResult(xyqbCardRelationResult.getMsg(), xyqbCardRelationResult.getCode());
        }

        return JsonResult.buildSuccessResult("处理成功", xyqbCardRelationResult.getData());
    }

    @Override
    public JsonResult<String> unBindCardRelation(UnBindCardModel unBindCardModel) {
        final String logPre = "[unBindCardRelation] 解除绑卡";

        if (Objects.isNull(unBindCardModel)) {
            log.error("{} 请求参数 unBindCardModel 为空", logPre);
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.PARAM_ERROR.getMessage(), ErrorCodeEnum.PARAM_ERROR.getCode());
        }

        if (Objects.isNull(unBindCardModel.getBindCardRecordId()) || Objects.isNull(unBindCardModel.getUserId())) {
            log.error("{} 请求参数 bindCardRecordId 和 userId 为空 unBindCardModel:{}", logPre, unBindCardModel);
            return JsonResult.buildErrorStateResult("请求参数 bindCardRecordId 和 userId 不能为空", ErrorCodeEnum.PARAM_ERROR.getCode());
        }

        //发起xyqb查询
        JsonResult<String> unBindCardResult = xyqbService.unBindCardRelation(unBindCardModel);
        log.info("{} 信用钱包返回信息 unBindCardResult:{}", logPre, unBindCardResult);
        if (Objects.isNull(unBindCardResult)) {
            log.error("{} 返回为空 unBindCardModel={}", logPre, unBindCardModel);
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.RETURN_ERROR.getMessage(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }
        if (!unBindCardResult.isSuccess()) {
            log.error("{} 信息异常 unBindCardModel={},unBindCardResult={}", logPre, unBindCardModel, unBindCardResult);
            return JsonResult.buildErrorStateResult(unBindCardResult.getMsg(), unBindCardResult.getCode());
        }

        return JsonResult.buildSuccessResult("处理成功", unBindCardResult.getData());
    }

    @Override
    public JsonResult<String> modifyReservePhone(ModifyReservePhone modifyReservePhone) {
        final String logPre = "[modifyReservePhone] 修改预留手机号";

        if (Objects.isNull(modifyReservePhone)) {
            log.error("{} 请求参数 modifyReservePhone 为空", logPre);
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.PARAM_ERROR.getMessage(), ErrorCodeEnum.PARAM_ERROR.getCode());
        }


        if (Objects.isNull(modifyReservePhone.getBankId())
                || Objects.isNull(modifyReservePhone.getUserId())
                || StringUtils.isBlank(modifyReservePhone.getPhoneNo())) {
            log.error("{} 请求参数 bankId,userId,phoneNo 为空 modifyReservePhone:{}", logPre, modifyReservePhone);
            return JsonResult.buildErrorStateResult("请求参数 cbankId,userId,phoneNo 不能为空", ErrorCodeEnum.PARAM_ERROR.getCode());
        }

        //发起xyqb查询
        JsonResult<String> modifyReservePhoneResult = xyqbService.modifyReservePhone(modifyReservePhone);
        log.info("{} 信用钱包返回信息 modifyReservePhoneResult:{}", logPre, modifyReservePhoneResult);
        if (Objects.isNull(modifyReservePhoneResult)) {
            log.error("{} 返回为空 modifyReservePhone={}", logPre, modifyReservePhone);
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.RETURN_ERROR.getMessage(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }
        if (!modifyReservePhoneResult.isSuccess()) {
            log.error("{} 信息异常 modifyReservePhone={},modifyReservePhoneResult={}", logPre, modifyReservePhone, modifyReservePhoneResult);
            return JsonResult.buildErrorStateResult(modifyReservePhoneResult.getMsg(), modifyReservePhoneResult.getCode());
        }

        return JsonResult.buildSuccessResult("处理成功", modifyReservePhoneResult.getData());
    }


}
