package cn.quantgroup.customer.service.impl;

import cn.quantgroup.customer.entity.TransactionReceiptRecord;
import cn.quantgroup.customer.exception.BusinessException;
import cn.quantgroup.customer.rest.vo.transaction.TransactionReceiptRecordVO;
import cn.quantgroup.customer.repo.TransactionReceiptRecordRepo;
import cn.quantgroup.customer.rest.param.transactionreceipt.TransactionReceiptRecordQuery;
import cn.quantgroup.customer.rest.vo.JsonResult;
import cn.quantgroup.customer.rest.vo.transaction.*;
import cn.quantgroup.customer.service.ITransactionReceiptRecordService;
import cn.quantgroup.customer.service.http.IHttpService;
import cn.quantgroup.customer.util.ExcelUtil;
import cn.quantgroup.customer.util.FileToZip;
import cn.quantgroup.customer.util.ITextPDFUtil;
import cn.quantgroup.customer.util.SFTPUtil;
import cn.quantgroup.user.retbean.XUser;
import cn.quantgroup.user.vo.UserSysResult;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Maps;
import com.itextpdf.text.*;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;
import com.itextpdf.text.pdf.PdfWriter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.formula.functions.T;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;
import javax.persistence.criteria.Predicate;
import java.io.ByteArrayInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

@Slf4j
@Service("transactionReceiptRecordService")
public class TransactionReceiptRecordServiceImpl implements ITransactionReceiptRecordService {

    @Autowired
    private TransactionReceiptRecordRepo transactionReceiptRecordRepo;
    @Autowired
    private ExcelUtil excelUtil;
    @Autowired
    private  IHttpService httpService;

    @Autowired
    private  UserSdkImpl userSdk;

    @Value("${kdsp-operation}")
    private String kdspOperationUrl;

    @Value("${passportapi2.http}")
    private String userSysUrl;

    //@Value("${customer.host}")
    private String host;
    //@Value("${customer.port}")
    private int port;
    //@Value("${customer.username}")
    private String username;
    //@Value("${customer.privateKey}")
    private String password;
    //@Value("${customer.basePath}")
    private String basePath;

    @Override
    public JsonResult<Page<TransactionReceiptRecordVO>> findRecordsByQuery(TransactionReceiptRecordQuery query) {
        Page<TransactionReceiptRecord> page = transactionReceiptRecordRepo.findAll((root, criteriaQuery, criteriaBuilder) -> {
            List<Predicate> predicates = new ArrayList<>();
            //交易单号
            if (StringUtils.isNotEmpty(query.getOrderNo())) {
                predicates.add(criteriaBuilder.equal(root.get("orderNo"), query.getOrderNo()));
            }
            if (StringUtils.isNotEmpty(query.getImportStatus())) {
                predicates.add(criteriaBuilder.equal(root.get("importStatus"), query.getImportStatus()));
            }

            if (StringUtils.isNotEmpty(query.getExportStatus())) {
                predicates.add(criteriaBuilder.equal(root.get("exportStatus"), query.getExportStatus()));
            }
            // 设置查询条件
            criteriaQuery.where(criteriaBuilder.and(predicates.toArray(new Predicate[predicates.size()])));
            // 指定排序
            //criteriaQuery.orderBy(criteriaBuilder.desc(root.get("id")));

            return criteriaQuery.getRestriction();

        }, new PageRequest(query.getPageNo(), query.getPageSize()));

        Page<TransactionReceiptRecordVO> map = page.map(it -> {
            TransactionReceiptRecordVO workOrderVO = new TransactionReceiptRecordVO();
            workOrderVO.setTradeNo(it.getOrderNo());
            workOrderVO.setUserId(it.getUserId());
            workOrderVO.setUserName(it.getUserName());
            workOrderVO.setSerialNo(it.getSerialNo());
            workOrderVO.setExportStatus(it.getExportStatus());
            workOrderVO.setImportStatus(it.getImportStatus());
            workOrderVO.setId(workOrderVO.getId());
            return workOrderVO;
        });
        if (map != null) {
            return JsonResult.buildSuccessResult("查询成功", map);
        } else {
            return JsonResult.buildErrorStateResult("查询失败");
        }
    }

    @Override
    public JsonResult importTransactionReceiptRecord(MultipartFile file) throws IOException {
        //查询未导出数据最大批次号
        Long maxBatchNo = transactionReceiptRecordRepo.findMaxbatchNo(0);
        if(maxBatchNo == null || maxBatchNo.longValue() ==0l){
            maxBatchNo = 1l;
        }
        //读取所有数据
        List<TransactionReceiptRecord> readList = read(file.getOriginalFilename(), file.getInputStream());
        //0代表初始状态，1代表导入成功，2代表导入失败
        for(TransactionReceiptRecord record : readList){
            //保存时如果有其中一项为空则默认为导入失败，其他则默认初始状态
            if(record.getUserId() ==null || record.getUserId() ==0 || StringUtils.isEmpty(record.getUserName()) || StringUtils.isEmpty(record.getOrderNo())){
                record.setImportStatus(2);
                //默认导出失败
                record.setExportStatus(2);
            }else{
                record.setImportStatus(0);
                //默认导出初始状态
                record.setExportStatus(0);
            }
            //存储批次号
            record.setBatchNo(maxBatchNo);
            transactionReceiptRecordRepo.save(record);
        }
        return JsonResult.buildSuccessResult("success",null);
    }

    public  List<TransactionReceiptRecord> read(String filename, InputStream is) throws IOException {
        /**
         * 总行数
         */
        int totalRows = 0;
        /**
         * 总列数
         */
        int totalCols = 0;

        /** 验证文件是否合法 */
        if (!excelUtil.validateExcel(filename)) {
            throw new RuntimeException("文件格式不合法 ");
        }
        /** 判断文件的类型，是2003还是2007 */
        Workbook wb = null;
        if (!excelUtil.isExcel2007(filename)) {
            wb = new HSSFWorkbook(is);
        } else {
            wb = new XSSFWorkbook(is);
        }
        List<TransactionReceiptRecord> recordList = new ArrayList<>();
        /** 得到第一个shell */
        Sheet sheet = wb.getSheetAt(0);
        /** 得到Excel的行数 */
        totalRows = sheet.getPhysicalNumberOfRows();
        /** 得到Excel的列数 */
        if (totalRows >= 1 && sheet.getRow(0) != null) {
            totalCols = sheet.getRow(0).getPhysicalNumberOfCells();
        }
        /** 循环Excel的行 */
        for (int r = 0; r < totalRows; r++) {

            if(r == 0){
                continue;
            }
            Row row = sheet.getRow(r);
            if (row == null) {
                continue;
            }
            /** 循环Excel的列 */
            for (int c = 0; c < totalCols; c++) {
                Cell cell = row.getCell(c);
                String cellValue = "";
                if (null != cell) {
                    // 以下是判断数据的类型
                    switch (cell.getCellType()) {

                        case 0: // 数字
                            cellValue = cell.getNumericCellValue() + "";
                            break;
                        case 1: // 字符串
                            cellValue = cell.getStringCellValue();
                            break;
                        case  4: // Boolean
                            cellValue = cell.getBooleanCellValue() + "";
                            break;
                        case 2: // 公式
                            cellValue = cell.getCellFormula() + "";
                            break;
                        case 3: // 空值
                            cellValue = "";
                            break;
                        case 5: // 故障
                            cellValue = "非法字符";
                            break;
                        default:
                            cellValue = "未知类型";
                            break;
                    }
                }

                if(r == 1){
                    continue;
                }
                if (org.springframework.util.StringUtils.isEmpty(cellValue)) {
                    continue;
                }
                TransactionReceiptRecord record = new TransactionReceiptRecord();
                if(c == 1){
                    record.setSerialNo(cellValue);
                }else if(c == 2){
                    record.setUserId(Integer.valueOf(cellValue));
                }else if(c == 3){
                    record.setUserName(cellValue);
                }else if(c == 4){
                    record.setOrderNo(cellValue);
                }
                recordList.add(record);
            }
        }
        return recordList;
    }

    //导出zip压缩包文件
    public JsonResult  exportZipFile() throws Exception {
        //调用电商接口获取对应数据
        final String logPre = "TransactionReceiptRecordServiceImpl.exportZipFile";
        //查询需要导出的数据-导入成功的数据
        List<TransactionReceiptRecord> transactionReceiptRecordList= transactionReceiptRecordRepo.selectRecordsByImportStatus(1);
        if(CollectionUtils.isEmpty(transactionReceiptRecordList)){
            return JsonResult.buildErrorStateResult("没有可以导出得数据");
        }
        for(TransactionReceiptRecord record : transactionReceiptRecordList){
            Integer userId = record.getUserId();
            String orderNo = record.getOrderNo();
            //调用kdsp接口获取交易凭证订单信息
            String url = kdspOperationUrl + "/api/kdsp/op/fa-cui/transaction-proof/query";
            try {
                Map<String, String> header = Maps.newHashMap();
                header.put("Content-type", "application/json");
                header.put("qg-tenant-id", "560761");
                Map param = Maps.newHashMap();
                if (Objects.nonNull(userId)) {
                    param.put("userId", userId);
                }
                if (StringUtils.isNotBlank(orderNo)) {
                    param.put("orderNo", orderNo);
                }
                List<Map> paramList =new ArrayList<>();
                paramList.add(param);
                //得到json字符串
                String result = httpService.post(url, header, paramList);
                //转换成json 对象
                JSONObject json = JSONObject.parseObject(result);
                String jsonResult = json.getString("data");
                String businessCode = json.getString("businessCode");
                //查询报错
                if(!"0000".equals(businessCode)){
                    return JsonResult.buildErrorStateResult("订单信息查询报错");
                }
                if(jsonResult != null){
                    JSONObject jsonObject = JSONObject.parseObject(jsonResult);
                    JSONArray jsonArray = jsonObject.getJSONArray("dataList");
                    if(jsonArray.isEmpty()){
                        return JsonResult.buildErrorStateResult("未查询到对应的订单信息");
                    }
                    List<TransactionReceiptVO> transactionReceiptVOList = jsonArray.toJavaList(TransactionReceiptVO.class);
                    if(transactionReceiptVOList.isEmpty()){
                        return JsonResult.buildErrorStateResult("未查询到对应的订单信息");
                    }
                    //生成pdf

                    String basicPath="D:\\交易凭证0222";
                    String filePath =basicPath +"\\"+record.getBatchNo()+record.getUserName();
                    FileToZip.mkdir(filePath);
                    filePath = filePath+"\\"+record.getOrderNo()+".pdf";
                    generatePDF(transactionReceiptVOList.get(0),filePath);
                    //调用电子签章接口
                    //todo
                    //保存到文件服务器上
                    SFTPUtil sftpUtil = new SFTPUtil(username,password,host,port);
                    sftpUtil.login();
                    log.info("uploadFile | 成功连接ftp");
                    //sftpUtil.upload(basePath,directory,okFileName,new ByteArrayInputStream(okFileData));
                    sftpUtil.logout();
                }
            } catch (Exception e) {
                log.error("[TransactionReceiptRecordServiceImpl][exportZipFile] 网络通讯异常,userId:{},ex:{}", ExceptionUtils.getStackTrace(e));
                return JsonResult.buildErrorStateResult(e.getMessage());
            }
        }
        return JsonResult.buildSuccessResult(null,null);
    }

    /**
     * 校验是否可以导出
     * @return
     */
    @Override
    public JsonResult checkCanExport() {
        //查询初始状态的数据-如果存在则说明未全部更新状态
        Long count = transactionReceiptRecordRepo.selectCountByImportStatus(0);
        if(count != null && count > 0){
            //异步多线程更新数据
            return JsonResult.buildErrorStateResult("导入处理中，请稍后重试");
        }else{
            return JsonResult.buildSuccessResult("数据全部导入成功",null);
        }
    }

    /**
     * 更新交易凭证导入状态
     * @return
     */
    @Override
    public JsonResult updateTransactionRecordsStatus() {
        //查询所有初始化的数据
        List<TransactionReceiptRecord> transactionReceiptRecordList
                = transactionReceiptRecordRepo.selectRecordsByImportStatus(0);
        //没有校验成功的放入一个list
        List<TransactionReceiptRecord> failList = new ArrayList<>();
        if(CollectionUtils.isNotEmpty(transactionReceiptRecordList)){
            //创建线程池
            //ThreadPoolExecutor threadPool = new ThreadPoolExecutor(20, 50,
            //       4, TimeUnit.SECONDS, new ArrayBlockingQueue(10), new ThreadPoolExecutor.AbortPolicy());
            // 记录单个任务的执行次数
            //CountDownLatch countDownLatch = new CountDownLatch(splitNList.size());
            // 对拆分的集合进行批量处理, 先拆分的集合, 再多线程执行
            /*for (List<TransactionReceiptRecord> singleList : splitNList) {
                // 线程池执行
                threadPool.execute(new Thread(new Runnable(){
                    @Override
                    public void run() {
                        for (TransactionReceiptRecord record : singleList) {
                            // 将每一个对象进行数据封装, 并添加到一个用于存储更新数据的list
                            // ......

                            // 任务个数 - 1, 直至为0时唤醒await()
                            countDownLatch.countDown();
                        }
                    }
                }));
            }
            try {
                // 让当前线程处于阻塞状态，直到锁存器计数为零
                countDownLatch.await();
            } catch (InterruptedException e) {
                throw new BusinessException("");
            }*/
            //为了性能每次最多传50个
            List<List<TransactionReceiptRecord>> splitNList= splitList(transactionReceiptRecordList,50);
            //分组校验
            for(List<TransactionReceiptRecord> recordList : splitNList){
                //调用金融用户中心校验userId是否一致
                String userIds ="";
                //循环获取userid进行拼接
                for(int i= 0;i < recordList.size();i++){
                    TransactionReceiptRecord record =recordList.get(i);
                    if(i != recordList.size()-1){
                        userIds += record.getUserId()+",";
                    }else{
                        userIds += record.getUserId()+"";
                    }
                }
                Map<String, String> header = Maps.newHashMap();
                header.put("Content-type", "application/x-www-form-urlencoded");
                Map param = Maps.newHashMap();
                param.put("userIds",userIds);
                String userUrl = userSysUrl + "/api/sync/listByUserIds";
                String userResult = httpService.get(userUrl, param);
                if (StringUtils.isEmpty(userResult)) {
                    return JsonResult.buildErrorStateResult("批量查询userId报错");
                }
                //对比拿到的
                JSONObject userJsonObject = JSONObject.parseObject(userResult);
                String dataStr = userJsonObject.getString("data");
                JSONArray userArray = JSONArray.parseArray(dataStr);
                for(int i=0;i<userArray.size();i++){
                    JSONObject object= (JSONObject) userArray.get(i);
                    Integer userId =object.getInteger("userId");
                    String userName = object.getString("name");
                    TransactionReceiptRecord record =recordList.get(i);
                    if(record.getUserId().intValue() != userId.intValue() || userName.equals(record.getUserName())){
                        failList.add(record);
                    }
                }

                //调用kdsp接口校验userId和交易单号是否一致
                String kUrl = kdspOperationUrl + "/api/kdsp/op/fa-cui/transaction-proof/check";
                //得到json字符串
                String kResult = httpService.post(kUrl, header, param);
                JSONObject kJson =JSONObject.parseObject(kResult);
                String kJsonResult = kJson.getString("data");
                String businessCode = kJson.getString("businessCode");
                if(!"0000".equals(businessCode)){
                    return JsonResult.buildErrorStateResult("订单信息查询报错");
                }
                JSONObject kJsonObject = JSONObject.parseObject(kJsonResult);
                failList = (List<TransactionReceiptRecord>) kJsonObject.get("verifyErrorList");
            }
            //校验全部通过则更新状态
            List<TransactionReceiptRecord> sucessList = listrem(transactionReceiptRecordList,failList);
            for(TransactionReceiptRecord record :failList){
                transactionReceiptRecordRepo.updateTransactionStatusByImportStatus(2,record.getOrderNo());
            }
            for(TransactionReceiptRecord record :sucessList){
                transactionReceiptRecordRepo.updateTransactionStatusByImportStatus(2,record.getOrderNo());
            }
        }
        return JsonResult.buildSuccessResult("数据全部导入成功",null);
    }

    @Override
    public JsonResult updatePDFSignStatus(List<String> sucessList,List<String>failList) {
        //pdf签章成功
        if(!sucessList.isEmpty()){
            for(String orderNo :sucessList){
                transactionReceiptRecordRepo.updatePDFStatusByOrderNo(1,orderNo);
            }
        }
        //pdf签章失败
        if(!failList.isEmpty()){
            for(String orderNo :failList){
                transactionReceiptRecordRepo.updatePDFStatusByOrderNo(2,orderNo);
            }
        }
        return JsonResult.buildSuccessResult("pdf签章状态更新失败",null);
    }

    public  List<TransactionReceiptRecord> listrem(List<TransactionReceiptRecord> listA,List<TransactionReceiptRecord> listB){
        HashSet hs1 = new HashSet(listA);
        HashSet hs2 = new HashSet(listB);
        hs1.removeAll(hs2);
        List<TransactionReceiptRecord> listC = new ArrayList<>();
        listC.addAll(hs1);
        return listC;
    }


    /**
     * 切分list
     * @param records
     * @param groupSize
     * @return
     */
    private List<List<TransactionReceiptRecord>> splitList(List<TransactionReceiptRecord> records,int groupSize){
        int length = records.size();
        //计算可以分为多少组
        int num = (length+groupSize-1)/groupSize;
        List<List<TransactionReceiptRecord>> newList = new ArrayList<>();
        for(int i= 0;i < num; i++){
            //开始位置
            int fromIndex = i * groupSize;
            //结束位置
            int toIndex = (i+1) * groupSize < length ? (i+1) * groupSize : length;
            newList.add(records.subList(fromIndex,toIndex));
        }
        return newList;
    }

    /**
     * 生成PDF
     * @param transactionReceiptVO
     */
    public void generatePDF(TransactionReceiptVO transactionReceiptVO,String filePath){
        final String LOG_PRE = "TransactionReceiptRecordServiceImpl.generatePDF";
        log.info("{} 传入参数 transactionReceiptVO={}", LOG_PRE, transactionReceiptVO);
        //订单信息
        OrderVO orderVO = transactionReceiptVO.getOrder();
        //商品信息
        List<SkuVO> skuVOList = transactionReceiptVO.getSkuList();
        //收货人信息
        ReceiverVO receiverVO =transactionReceiptVO.getReceiver();
        //物流信息
        List<LogisticsVO> logisticsVOList = transactionReceiptVO.getLogisticsList();
        //物流详情
        //List<logisticsDetailVO> logisticsDetails = logisticsVO.getDetailList();
        Document document = new Document(PageSize.A4);
        try {
            PdfWriter.getInstance(document, new FileOutputStream(filePath));
            document.open();
            document.addTitle("交易凭证");//标题
            PdfPTable table = new PdfPTable(6);
            Font font = ITextPDFUtil.getColorFont(BaseColor.BLACK,7,"微软雅黑");

            table.setWidths(ITextPDFUtil.getColumnWiths(35,35,35,35,25,35));
            //订单信息
            //第一行合并单元格
            PdfPCell cell;
            cell = new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("订单信息"), font));
            cell.setColspan(6);
            cell.setMinimumHeight(18);
            table.addCell(cell);
            //第二行标题
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("订单ID"), font))).setMinimumHeight(18);
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("供应商订单ID"), font))).setMinimumHeight(18);
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("供应链订单号"), font))).setMinimumHeight(18);
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("商品渠道"), font))).setMinimumHeight(18);
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("下单平台"), font))).setMinimumHeight(18);
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("平台账号id"), font))).setMinimumHeight(18);

            //第三行数据iTextITextPDFUtil.
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(orderVO.getOrderNo())?"/":orderVO.getOrderNo()), font))).setMinimumHeight(18);
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(orderVO.getMerchantOrderId()) ?"/":orderVO.getMerchantOrderId()), font))).setMinimumHeight(18);
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(orderVO.getKeyStoneOrderNo())?"":orderVO.getKeyStoneOrderNo()), font))).setMinimumHeight(18);
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(orderVO.getSkuSourceName())?"/":orderVO.getOrderNo()), font))).setMinimumHeight(18);
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(orderVO.getTenantName())?"/":orderVO.getTenantName()), font))).setMinimumHeight(18);
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String(orderVO.getUserId() ==null ? "/":orderVO.getUserId().toString()), font))).setMinimumHeight(18);


            //第4行标题iTextITextPDFUtil.
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("平台绑定手机号"), font))).setMinimumHeight(18);
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("订单创建时间"), font))).setMinimumHeight(18);
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("支付完成时间"), font))).setMinimumHeight(18);
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("订单金额"), font))).setMinimumHeight(18);
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("实付金额"), font))).setMinimumHeight(18);
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("支付方式"), font))).setMinimumHeight(18);
            //查询手机号
            UserSysResult<XUser> user = userSdk.getService().findUserByUserId(orderVO.getUserId());
            //第5行数据iTextITextPDFUtil.
            if(user !=null && user.getData() !=null){
                table.addCell(new Phrase(ITextPDFUtil.getUTF8String(user.getData().getPhoneNo()), font));
            }else{
                table.addCell(new Phrase(ITextPDFUtil.getUTF8String("/"), font));
            }

            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(orderVO.getCreatedAt())?"/":orderVO.getCreatedAt()), font))).setMinimumHeight(18);
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(orderVO.getPayTime())?"/":orderVO.getPayTime()), font))).setMinimumHeight(18);
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(orderVO.getOrderAmount())?"":orderVO.getOrderAmount()), font))).setMinimumHeight(18);
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(orderVO.getPayAmount())?"/":orderVO.getPayAmount()), font))).setMinimumHeight(18);
            table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(orderVO.getPayTypeName())?"/":orderVO.getPayTypeName()), font))).setMinimumHeight(18);
            //有多个物流信息,重复展示物流信息，商品信息，收货信息，物流详情
            if(logisticsVOList.size() > 1){
                //展示多个
                for(int j= 0;j<logisticsVOList.size();j++){
                    table = generateMultiParagraph(cell,font,table,logisticsVOList,skuVOList,receiverVO,j);
                }
            }else{
                //展示一个
                table=generateMultiParagraph(cell,font,table,logisticsVOList,skuVOList,receiverVO,0);
            }

            document.add(table);
            document.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private PdfPTable generateMultiParagraph(PdfPCell cell,Font font,
                                        PdfPTable table,List<LogisticsVO> logisticsVOList,List<SkuVO> skuVOList,ReceiverVO receiverVO,int count){
        //物流信息
        cell = new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("物流信息"), font));
        cell.setColspan(6);
        cell.setMinimumHeight(18);
        table.addCell(cell);
        if(logisticsVOList.size()>0){
            //第一行标题
            table.addCell(new Phrase(ITextPDFUtil.getUTF8String("发货时间"), font));
            table.addCell(new Phrase(ITextPDFUtil.getUTF8String("确认收货时间"), font));
            table.addCell(new Phrase(ITextPDFUtil.getUTF8String("订单状态"), font));
            table.addCell(new Phrase(ITextPDFUtil.getUTF8String("订单类型"), font));
            table.addCell(new Phrase(ITextPDFUtil.getUTF8String("物流公司"), font));
            table.addCell(new Phrase(ITextPDFUtil.getUTF8String("物流单号"), font));
            //第二行数据

            LogisticsVO logisticsVO=logisticsVOList.get(count);
            table.addCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(logisticsVO.getPostTime())?"/":logisticsVO.getPostTime()), font));
            table.addCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(logisticsVO.getCompleteTime())?"/":logisticsVO.getCompleteTime()), font));
            table.addCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(logisticsVO.getStatusName())?"/":logisticsVO.getStatusName()), font));
            table.addCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(logisticsVO.getTypeName())?"/":logisticsVO.getTypeName()), font));
            table.addCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(logisticsVO.getExpressCompany())?"/":logisticsVO.getExpressCompany()), font));
            table.addCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(logisticsVO.getDeliveryNo())?"/":logisticsVO.getDeliveryNo()), font));
        }

        //商品信息
        cell = new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("商品信息"),font));
        cell.setColspan(6);
        cell.setMinimumHeight(18);
        table.addCell(cell);
        //商品信息标题
        //合并3行
        cell = new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("商品名称"), font));
        cell.setColspan(3);
        cell.setMinimumHeight(18);
        table.addCell(cell);
        //合并2行
        cell = new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("商品属性"), font));
        cell.setColspan(2);
        cell.setMinimumHeight(18);
        table.addCell(cell);
        table.addCell(new Phrase(ITextPDFUtil.getUTF8String("商品件数"), font));

        //商品信息数据
        for(SkuVO skuVO:skuVOList){
            //商品名称取值
            cell = new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(skuVO.getSkuName())?"/":skuVO.getSkuName()), font));
            cell.setColspan(3);
            cell.setMinimumHeight(18);
            table.addCell(cell);
            //商品属性取值
            cell = new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(skuVO.getSkuAttr())?"/":skuVO.getSkuAttr()), font));
            cell.setColspan(2);
            cell.setMinimumHeight(18);
            table.addCell(cell);
            //商品件数取值
            table.addCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(skuVO.getCount())?"/":skuVO.getCount()), font));
        }

        //收货人信息
        cell = new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("收货人信息"), font));
        cell.setMinimumHeight(18);
        cell.setColspan(6);
        table.addCell(cell);

        //收货人信息标题
        table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("收货人手机"), font))).setMinimumHeight(18);
        table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("收货人姓名"), font))).setMinimumHeight(18);
        cell = new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("收货地址"), font));
        cell.setMinimumHeight(18);
        cell.setColspan(4);
        table.addCell(cell);

        //收货人信息数据
        table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(receiverVO.getReceiverMobile())?"/":receiverVO.getReceiverMobile()), font))).setMinimumHeight(18);
        table.addCell(new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(receiverVO.getReceiverName())?"/":receiverVO.getReceiverName()), font))).setMinimumHeight(18);
        cell = new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(receiverVO.getFullAddress())?"/":receiverVO.getFullAddress()), font));
        cell.setMinimumHeight(18);
        cell.setColspan(4);
        table.addCell(cell);

        //物流详情
        if(logisticsVOList.size()>0){
            cell = new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("物流详情"), font));
            cell.setColspan(6);
            cell.setMinimumHeight(18);
            table.addCell(cell);
            //物流详情标题
            table.addCell(new Phrase(ITextPDFUtil.getUTF8String("时间"), font));
            cell = new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String("状态"), font));
            cell.setColspan(5);
            cell.setMinimumHeight(18);
            table.addCell(cell);
            List<logisticsDetailVO> logisticsDetails =logisticsVOList.get(count).getDetailList();
            //物流详情数据
            for(logisticsDetailVO vo : logisticsDetails){
                table.addCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(vo.getTime())?"/":vo.getTime()), font));
                cell = new PdfPCell(new Phrase(ITextPDFUtil.getUTF8String(StringUtils.isEmpty(vo.getDesc())?"/":vo.getDesc()), font));
                cell.setColspan(5);
                cell.setMinimumHeight(18);
                table.addCell(cell);
            }
        }
        return table;
    }
}
