package cn.quantgroup.customer.service.impl;

import cn.quantgroup.customer.aop.OperateLog;
import cn.quantgroup.customer.entity.OpLog;
import cn.quantgroup.customer.entity.OperateLogModel;
import cn.quantgroup.customer.enums.ErrorCodeEnum;
import cn.quantgroup.customer.model.kaordermapping.ApplyRequestHistory;
import cn.quantgroup.customer.model.kaordermapping.LoanOrderMapping;
import cn.quantgroup.customer.model.order.*;
import cn.quantgroup.customer.model.xyqbuser.UserBasicInfo;
import cn.quantgroup.customer.rest.param.ordermapping.EarlySettleUpOrderQueryParam;
import cn.quantgroup.customer.rest.param.ordermapping.OperateEntryParam;
import cn.quantgroup.customer.rest.param.ordermapping.OrderQueryParam;
import cn.quantgroup.customer.rest.param.user.UserQueryParam;
import cn.quantgroup.customer.rest.vo.JsonResult;
import cn.quantgroup.customer.service.*;
import cn.quantgroup.customer.util.DateUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.security.Timestamp;
import java.util.*;

/**
 * @author Wang Xiangwei
 * @version 2020/3/10
 */
@Slf4j
@Service
public class OrderServiceImpl implements IOrderService {
    @Autowired
    private IKaService kaService;

    @Autowired
    private IXyqbService xyqbService;

    @Autowired
    private IOperateLogService operateLogService;

    @Autowired
    private IUserService userService;

    @Override
    public JsonResult<List<FlowNode>> findFlowChart(String applyOrderNo) {
        String logPre = "OrderServiceImpl.findFlowChart";
        log.info("{} 查询流程图 applyOrderNo={}", logPre, applyOrderNo);
        OrderQueryParam param = new OrderQueryParam();
        param.setApplyOrderNo(applyOrderNo);
        JsonResult<LoanOrderMapping> orderMappingResult = kaService.findOrderMapping(param);
        LoanOrderMapping head = null;
        if (orderMappingResult.isSuccess()) {
            head = orderMappingResult.getData();
            log.info("{},申请订单号查询orderMapping applyOrderNo={},data={}", logPre, applyOrderNo, head);
        }

        JsonResult<List<FlowNode>> flowChart = xyqbService.findFlowChart(applyOrderNo);
        if (!flowChart.isSuccess()) {
            log.error("{} 流程图查询失败 result={}", logPre, flowChart);
            return flowChart;
        }

        List<FlowNode> flowNodeList = flowChart.getData();
        if (Objects.nonNull(head)) {
            FlowNode flowNode = new FlowNode();
            flowNode.setName("渠道用户信息导入");
            flowNode.setStatus(1);
            flowNode.setNode("IMPORT_INFO");
            flowNode.setRemark("进件成功");
            flowNodeList.add(0, flowNode);

        }

        return JsonResult.buildSuccessResult("", flowNodeList);
    }


    @Override
    public JsonResult<LoanOrderDetail> getLoanOrderDetail(Long loanId) {
        String logPre = "OrderServiceImpl.getLoanOrderDetail";
        log.info("{} 提现详情 loanId={}", logPre, loanId);
        JsonResult<LoanOrder> loanOrder = xyqbService.findLoanOrder(loanId);

        if (!loanOrder.isSuccess()) {
            log.error("{} 提现订单查询失败 loanId={}, result={}", logPre, loanId, loanOrder);
            return JsonResult.buildErrorStateResult(loanOrder.getMsg(), loanOrder.getData());
        }

        LoanOrderDetail loanOrderDetail = new LoanOrderDetail();
        List<Repayment> repaymentList = new ArrayList<>();
        LoanOrder loanOrderData = loanOrder.getData();
        loanOrderDetail.setLoanOrder(loanOrderData);
        loanOrderDetail.setRepaymentList(repaymentList);

        if (Objects.isNull(loanOrderData) || !loanOrderData.getShowPlans()) {
            log.info("{} 不需查询还款计划", logPre);
            return JsonResult.buildSuccessResult("查询成功", loanOrderDetail);
        }

        //还款计划查询
        JsonResult<OrderRepayment> orderRepaymentJsonResult = xyqbService.repaymentPlanQuery(loanId);
        if (!orderRepaymentJsonResult.isSuccess()) {
            log.error("{} 还款计划查询失败 result={}", logPre, orderRepaymentJsonResult);
            return JsonResult.buildErrorStateResult(orderRepaymentJsonResult.getMsg(), orderRepaymentJsonResult.getData());
        }

        List<RepaymentPlanItem> repaymentPlans = orderRepaymentJsonResult.getData().getRepaymentPlans();
        repaymentPlans.forEach(e -> repaymentList.add(Repayment.valueOf(e)));

        return JsonResult.buildSuccessResult("查询成功", loanOrderDetail);
    }

    @Override
    public JsonResult<OrderInfoVo> queryOrderInfo(OrderQueryParam orderQuery) {
        String logPre = "OrderServiceImpl.queryOrderInfo 查询渠道订单信息";
        if (Objects.isNull(orderQuery)
                || (StringUtils.isBlank(orderQuery.getApplyOrderNo())
                && StringUtils.isBlank(orderQuery.getApplyOrderNo())
                && Objects.isNull(orderQuery.getLoanId())
                && Objects.isNull(orderQuery.getChannelId()))) {
            log.error("{} 请求参数为空 orderQuery={}", logPre, orderQuery);
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.PARAM_ERROR.getMessage(), ErrorCodeEnum.PARAM_ERROR.getCode());
        }

        // 查询订单信息  查询ka
        JsonResult<LoanOrderMapping> orderMappingResult = kaService.findOrderMapping(orderQuery);
        LoanOrderMapping loanOrderMapping;
        if (!orderMappingResult.isSuccess()) {
            log.warn("{} 查询订单信息orderMapping失败 result={} orderQuery={}", logPre, orderMappingResult, orderQuery);
            return JsonResult.buildErrorStateResult(orderMappingResult.getMsg(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }

        loanOrderMapping = orderMappingResult.getData();
        if (Objects.nonNull(loanOrderMapping)) {
            return buildHasOrderMapping(orderQuery, loanOrderMapping);
        } else {
            // 可能两种结果 1)进件失败 此时需要根据渠道订单号和渠道号查询进件流水  2)不存在该订单
            if (StringUtils.isNotBlank(orderQuery.getChannelOrderNo()) && Objects.nonNull(orderQuery.getChannelId())) {
                return buildNoOrderMapping(orderQuery.getChannelOrderNo(), orderQuery.getChannelId());
            } else {
                // 不存在订单信息
                return JsonResult.buildSuccessResult("订单不存在", null);
            }
        }
    }

    @Override
    public JsonResult queryEarlySettleUpOrders(EarlySettleUpOrderQueryParam orderQuery) {
        String logPre = "OrderServiceImpl.queryEarlySettleUpOrders";
        //如果输入了渠道号，必须输入渠道订单号
        if(!existAtLestOneParam(orderQuery)){
            return JsonResult.buildErrorStateResult("至少输入一个请求参数", null);

        }

        // 渠道订单号和渠道号必须同时存在
        if (!existChannelOrderNoAndChannelId(orderQuery))
        {
            return JsonResult.buildErrorStateResult("渠道号和渠道订单号必须同时存在", null);
        }

        // 获取用户userId
        JsonResult<UserBasicInfo> userInfo = this.queryAndCheckUserInfo(orderQuery);

        // 请求xyqb

        // 处理返回数据


        return JsonResult.buildSuccessResult("处理成功", null);
    }

    // 查询用户信息
    private JsonResult<UserBasicInfo> queryAndCheckUserInfo(EarlySettleUpOrderQueryParam orderQuery) {
        String logPre = "OrderServiceImpl.queryAndCheckUserInfo";

        JsonResult<UserBasicInfo> userInfoByOrder = null;
        // 有订单信息存在，根据订单查询用户userId
        if (!(Objects.isNull(orderQuery.getLoanId()) && Objects.isNull(orderQuery.getChannelId()) && StringUtils.isBlank(orderQuery.getChannelOrderNo()))) {
            OrderQueryParam orderQueryParam = new OrderQueryParam();
            orderQueryParam.setLoanId(orderQuery.getLoanId());
            orderQueryParam.setChannelId(orderQuery.getChannelId());
            orderQueryParam.setChannelOrderNo(orderQuery.getChannelOrderNo());
            userInfoByOrder = this.userService.findUserInfoByOrderParam(orderQueryParam);
            if (!userInfoByOrder.isSuccess()) {
                log.error("{} 查询用户失败 result={}", logPre, userInfoByOrder);
                return userInfoByOrder;
            }
        }

        JsonResult<UserBasicInfo> userInfoByPhone = null;
        if (Objects.nonNull(orderQuery.getPhoneNo())) {
            //用户参数查询
            UserQueryParam userQueryParam = new UserQueryParam();
            userQueryParam.setPhoneNo(orderQuery.getPhoneNo());
            userQueryParam.setUserId(orderQuery.getUserId());
            userInfoByPhone = this.userService.findUserInfo(userQueryParam);
            if (!userInfoByPhone.isSuccess()) {
                log.error("{} 查询用户失败 result={}", logPre, userInfoByPhone);
                return userInfoByPhone;
            }
        }

        //TODO 需要联合判断

        if(Objects.isNull(userInfoByOrder) && Objects.isNull(userInfoByPhone) ){

        }

        return null;
    }

    private boolean existChannelOrderNoAndChannelId(EarlySettleUpOrderQueryParam orderQuery) {
        if ( Objects.isNull(orderQuery.getChannelId()) && StringUtils.isNotBlank(orderQuery.getChannelOrderNo())
                || Objects.nonNull(orderQuery.getChannelId()) && StringUtils.isBlank(orderQuery.getChannelOrderNo())
                ) {
            return false;
        }
        return true;
    }

    private boolean existAtLestOneParam(EarlySettleUpOrderQueryParam orderQuery) {
        if (Objects.isNull(orderQuery) ||
                (StringUtils.isBlank(orderQuery.getChannelOrderNo())
                        && Objects.isNull(orderQuery.getChannelId())
                        && Objects.isNull(orderQuery.getLoanId())
                        && Objects.isNull(orderQuery.getUserId())
                        && StringUtils.isBlank(orderQuery.getFundOrderNo())
                        && StringUtils.isBlank(orderQuery.getPhoneNo())
                )
                ) {
            return false;
        }
        return true;
    }

    @Override
    public JsonResult operateEntry(OperateEntryParam operateEntryParam) {
        return JsonResult.buildSuccessResult("处理成功", Boolean.TRUE);
    }

    @Override
    public JsonResult earlySettleUpTrial(Long loanId) {
        List<Map<String, Object>> datas = new ArrayList<>();

        for (int i = 0; i < 3; i++) {
            Map<String, Object> map = new HashMap<>();
            if (i == 0) {
                map.put("title", "本金");
                map.put("value", "10000");
            } else if (i == 1) {
                map.put("title", "利息");
                map.put("value", "100");
            } else {
                map.put("title", "违约金");
                map.put("value", "500");
            }
            datas.add(map);
        }

        return JsonResult.buildSuccessResult("处理成功", datas);
    }

    @Override
    public JsonResult queryOperateLog(Long loanId) {
        List<OpLog> list = operateLogService.findLogsByLoanId(loanId);
        return JsonResult.buildSuccessResult("处理成功", list);
    }

    /**
     * 处理查询不到orderMapping的情况，可能是进件失败，也可能不存在该笔订单
     *
     * @param channelOrderNo
     * @param channelId
     * @return
     */
    private JsonResult<OrderInfoVo> buildNoOrderMapping(String channelOrderNo, Long channelId) {
        String logPre = "OrderServiceImpl.buildNoOrderMapping";
        // 查询进件流水信息  查询ka
        JsonResult<ApplyRequestHistory> orderMappingResult = kaService.queryLatestChannelRequestFlows(channelOrderNo, channelId);
        if (!orderMappingResult.isSuccess()) {
            log.error("{} 查询渠道进件流水信息异常 result={} channelOrderNo={},channelId={}", logPre, orderMappingResult, channelOrderNo, channelId);
            return JsonResult.buildErrorStateResult("查询渠道进件流水信息异常", ErrorCodeEnum.RETURN_ERROR.getCode());
        }

        ApplyRequestHistory applyRequestHistory = orderMappingResult.getData();
        // 不存在进件流水信息 返回查询不到该笔订单
        if (Objects.isNull(applyRequestHistory)) {
            return JsonResult.buildSuccessResult("订单不存在", null);
        }

        OrderInfo orderInfo = OrderInfo.builder()
                .channelOrderNo(channelOrderNo)
                .channelId(channelId)
                .status(applyRequestHistory.getIsSuccess() ? "进件成功" : "进件失败")
                .remark(applyRequestHistory.getDescriptionDetail())
                .build();

        OrderInfoVo orderInfoVo = OrderInfoVo.builder()
                .orderInfo(orderInfo)
                .build();
        log.info("{} 查询订单信息,返回信息 orderInfoVo={}", logPre, orderInfoVo);
        return JsonResult.buildSuccessResult("处理成功", orderInfoVo);
    }

    /**
     * 处理渠道订单表有数据的情形
     *
     * @param orderQuery
     * @param loanOrderMapping
     * @return
     */
    private JsonResult<OrderInfoVo> buildHasOrderMapping(OrderQueryParam orderQuery, LoanOrderMapping loanOrderMapping) {
        String logPre = "OrderServiceImpl.buildHasOrderMapping";

        // 查询订单状态  xyqb
        orderQuery.setApplyOrderNo(loanOrderMapping.getApplyNo());
        orderQuery.setLoanId(loanOrderMapping.getLoanId());
        orderQuery.setChannelId(loanOrderMapping.getRegisteredFrom());
        JsonResult<OrderStatus> orderStatusResult = xyqbService.orderStatusQuery(orderQuery);
        OrderStatus orderStatus;
        if (!orderStatusResult.isSuccess()) {
            log.error("{} 查询订单状态 orderStatus 失败 result={} orderQuery={}", logPre, orderStatusResult, orderQuery);
            return JsonResult.buildErrorStateResult(orderStatusResult.getMsg(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }

        orderStatus = orderStatusResult.getData();

        // 查询通知记录  查询ka
        JsonResult<CallbackRecordList> callbackListResult = kaService.querySendRecords(loanOrderMapping.getApplyNo(), loanOrderMapping.getRegisteredFrom());
        if (!callbackListResult.isSuccess()) {
            log.error("{} 查询发送记录失败  result={} orderQuery={}", logPre, callbackListResult, orderQuery);
            return JsonResult.buildErrorStateResult("查询通知记录失败", ErrorCodeEnum.RETURN_ERROR.getCode());
        }

        final String defaultStatusStr = "进件成功";
        String status = Objects.nonNull(orderStatus) ? orderStatus.getStatus().getValue() : defaultStatusStr;
        OrderInfo orderInfo = OrderInfo.builder()
                .channelOrderNo(loanOrderMapping.getChannelOrderNo())
                .applyNo(loanOrderMapping.getApplyNo())
                .channelId(loanOrderMapping.getRegisteredFrom())
                .loanId(loanOrderMapping.getLoanId())
                .status(status)
                .remark(status)
                .build();

        OrderInfoVo orderInfoVo = OrderInfoVo.builder()
                .orderInfo(orderInfo)
                .noticeList(callbackListResult.getData())
                .build();
        log.info("{} 查询订单信息,返回信息 orderInfoVo={}", logPre, orderInfoVo);
        return JsonResult.buildSuccessResult("处理成功", orderInfoVo);
    }


}
