package cn.quantgroup.customer.service.impl;

import cn.quantgroup.customer.common.adapter.LocalDateTypeAdapter;
import cn.quantgroup.customer.entity.User;
import cn.quantgroup.customer.enums.ErrorCodeEnum;
import cn.quantgroup.customer.exception.BusinessException;
import cn.quantgroup.customer.model.Tuple;
import cn.quantgroup.customer.model.kaordermapping.LoanOrderMapping;
import cn.quantgroup.customer.model.xyqbuser.UserBasicInfo;
import cn.quantgroup.customer.model.xyqbuser.UserCombination;
import cn.quantgroup.customer.repo.UserRepo;
import cn.quantgroup.customer.rest.param.applyorder.ApplyOrderQuery;
import cn.quantgroup.customer.rest.param.ordermapping.OrderMappingQueryParam;
import cn.quantgroup.customer.rest.param.phone.ModifyPhoneAudit;
import cn.quantgroup.customer.rest.param.phone.ModifyPhoneFeedback;
import cn.quantgroup.customer.rest.param.phone.ModifyPhoneQuery;
import cn.quantgroup.customer.rest.param.user.UserCombinationParam;
import cn.quantgroup.customer.rest.vo.JsonResult;
import cn.quantgroup.customer.service.IKaService;
import cn.quantgroup.customer.service.IUserService;
import cn.quantgroup.customer.service.IXyqbService;
import cn.quantgroup.customer.service.http.IHttpService;
import cn.quantgroup.customer.util.DateUtil;
import cn.quantgroup.customer.util.IdcardUtils;
import cn.quantgroup.customer.util.ValidationUtil;
import cn.quantgroup.riskcontrol.model.AuthenticationUserDetail;
import cn.quantgroup.user.retbean.XUserDetail;
import cn.quantgroup.user.retbean.XUserFullInfo;
import cn.quantgroup.user.vo.UserSysResult;
import com.google.common.collect.Maps;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.stereotype.Service;

import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import static cn.quantgroup.customer.constant.Constant.GSON;

@Slf4j
@Service("userService")
public class UserServiceImpl implements IUserService {

    @Value("${passportapi2.http}")
    private String userSysUrl;

    private final UserRepo userRepo;
    private final IHttpService httpService;

    private final UserSdkImpl userSdk;

    private final IXyqbService xyqbService;
    private final IKaService kaService;

    @Autowired
    public UserServiceImpl(UserRepo userRepo, IHttpService httpService, UserSdkImpl userSdk, IXyqbService xyqbService, IKaService kaService) {
        this.userRepo = userRepo;
        this.httpService = httpService;
        this.userSdk = userSdk;
        this.xyqbService = xyqbService;
        this.kaService = kaService;
    }

    @Override
    public UserDetails loadUserByUsername(String userName) throws UsernameNotFoundException {
        User user = userRepo.findByUsername(userName);
        if (user == null) {
            throw new UsernameNotFoundException("user: " + userName + " do not exist!");
        }
        return new AuthenticationUserDetail(user);
    }

    @Override
    public String modifyPhoneQuery(ModifyPhoneQuery modifyPhoneQuery) {
        String url = userSysUrl + "/v1/user/modify/phone_no";
        Gson gson = new GsonBuilder()
                .registerTypeAdapter(LocalDate.class, new LocalDateTypeAdapter())
                .create();
        Map param = gson.fromJson(gson.toJson(modifyPhoneQuery), Map.class);
        try {
            String response = httpService.get(url, param);
            if (StringUtils.isEmpty(response)) {
                log.error("[user][query modify phone list] 请求业务系统返回值为空,modifyPhoneQuery:{}", modifyPhoneQuery);
            }
            return response;
        } catch (Exception e) {
            log.error("[user][query modify phone list] 网络通讯异常,modifyPhoneQuery:{},ex:{}", modifyPhoneQuery, ExceptionUtils.getStackTrace(e));
            throw new BusinessException(ErrorCodeEnum.NET_ERROR);
        }
    }

    @Override
    public String modifyPhoneAudit(ModifyPhoneAudit modifyPhoneAudit) {
        String url = userSysUrl + "/v1/user/modify/phone_no/audit";
        Map param = GSON.fromJson(GSON.toJson(modifyPhoneAudit), Map.class);
        try {
            Map<String, String> header = Maps.newHashMap();
            header.put("Content-type", "application/json");
            String result = httpService.post(url, header, param);
            log.info("[user][query modify audit ] 请求业务系统返回值:{}", result);
            return result;
        } catch (Exception e) {
            log.error("[user][query modify audit ] 网络通讯异常,modifyPhoneAudit:{},ex:{}", modifyPhoneAudit, ExceptionUtils.getStackTrace(e));
            throw new BusinessException(ErrorCodeEnum.NET_ERROR);
        }
    }

    @Override
    public String modifyPhoneFeedback(ModifyPhoneFeedback modifyPhoneFeedback) {
        String url = userSysUrl + "/v1/user/modify/phone_no/{id}/feedback";
        try {
            String id = modifyPhoneFeedback.getId();
            url = url.replace("{id}", id);
            Map<String, String> header = Maps.newHashMap();
            header.put("Content-type", "application/json");
            String result = httpService.post(url, header, null);
            log.info("[user][query modify feedback ] 请求业务系统返回值:{}", result);
            return result;
        } catch (Exception e) {
            log.error("[user][query modify feedback ] 网络通讯异常,modifyPhoneFeedback:{},ex:{}", modifyPhoneFeedback, ExceptionUtils.getStackTrace(e));
            throw new BusinessException(ErrorCodeEnum.NET_ERROR);
        }
    }

    @Override
    public String findUuidByIdNo(String idNo) {
        String url = userSysUrl + "/innerapi/fetchUuid";
        try {
            Map<String, Object> param = Maps.newHashMap();
            param.put("idNo", idNo);
            String result = httpService.post(url, param);
            log.info("[user][query uuid by idNo ] 请求业务系统返回值:{}", result);
            JsonResult jsonResult = GSON.fromJson(result, JsonResult.class);
            Object uuid = jsonResult.getData();
            return String.valueOf(uuid);
        } catch (Exception e) {
            log.error("[user][query uuid by idNo ] 网络通讯异常,idNo:{},ex:{}", idNo, ExceptionUtils.getStackTrace(e));
            throw new BusinessException(ErrorCodeEnum.NET_ERROR);
        }
    }


    @Override
    public JsonResult findUserInfo(UserCombinationParam userCombinationParam) {
        final String LOG_PRE = "UserServiceImpl.findUserInfo";
        log.info("{} 查询用户信息 userCombinationParam={}", LOG_PRE, userCombinationParam);
        Tuple<Boolean, String> tuple = validate(userCombinationParam);

        if (!tuple.getKey()) {
            log.error("{},参数验证失败，{}", LOG_PRE, tuple.getValue());
            return JsonResult.buildErrorStateResult(tuple.getValue(), null);
        }

        //通过userId查询
        if (!Objects.isNull(userCombinationParam.getUserId())) {
            log.info("{} 通过userId查询 userId={}", LOG_PRE, userCombinationParam.getUserId());
            UserSysResult<XUserDetail> userDetailByUserId = userSdk.getService().findUserDetailByUserId(userCombinationParam.getUserId());
            return getUserBasicInfoResult(userDetailByUserId);
        }

        if (StringUtils.isNotBlank(userCombinationParam.getPhoneNo())) {
            log.info("{} 通过phoneNo查询 phoneNo={}", LOG_PRE, userCombinationParam.getPhoneNo());
            UserSysResult<XUserDetail> userDetailByPhone = userSdk.getService().findUserDetailByPhone(userCombinationParam.getPhoneNo());
            return getUserBasicInfoResult(userDetailByPhone);
        }

        if (StringUtils.isNotBlank(userCombinationParam.getUuid())) {
            log.info("{} 通过uuid查询 uuid={}", LOG_PRE, userCombinationParam.getUuid());
            UserSysResult<XUserFullInfo> userFullInfoByUuid = userSdk.getService().findUserFullInfoByUuid(userCombinationParam.getUuid());
            Object jsonResultData = getJsonResultData(userFullInfoByUuid);
            if (jsonResultData instanceof XUserFullInfo) {
                XUserFullInfo xUserFullInfo = (XUserFullInfo) jsonResultData;
                UserSysResult<XUserDetail> userDetailByPhone = userSdk.getService().findUserDetailByPhone(xUserFullInfo.getPhoneNo());
                return getUserBasicInfoResult(userDetailByPhone);
            } else {
                return (JsonResult) jsonResultData;
            }

        }

        if (!Objects.isNull(userCombinationParam.getLoanId()) ||
                StringUtils.isNotBlank(userCombinationParam.getChannelOrderNo()) ||
                StringUtils.isNotBlank(userCombinationParam.getOrderNo())) {
            //通过ka获得userId然后通过userId查询
            OrderMappingQueryParam param = new OrderMappingQueryParam();
            param.setApplyOrderNo(userCombinationParam.getOrderNo());
            param.setChannelOrderNo(userCombinationParam.getChannelOrderNo());
            param.setLoanId(userCombinationParam.getLoanId());
            param.setChannelId(userCombinationParam.getChannelId());
            JsonResult<LoanOrderMapping> orderMapping = kaService.findOrderMapping(param);
            if (!orderMapping.isSuccess() || Objects.isNull(orderMapping.getData())) {
                log.error("{} 查询用户失败 param={} result={}", LOG_PRE, param, orderMapping);
                return orderMapping;
            }
            Long qgUserId = orderMapping.getData().getQgUserId();
            UserSysResult<XUserDetail> userDetailByUserId = userSdk.getService().findUserDetailByUserId(qgUserId);
            return getUserBasicInfoResult(userDetailByUserId);
        }

        if (!Objects.isNull(userCombinationParam.getIdNo())) {
            //通过业务系统获得
            log.info("{} 通过phoneNo查询 idNo={}", LOG_PRE, userCombinationParam.getIdNo());
            return findUserDetailByIdNo(userCombinationParam.getIdNo());
        }

        return JsonResult.buildErrorStateResult("无有效查询参数", null);
    }


    private JsonResult findUserDetailByIdNo(String idNo) {
        String url = userSysUrl + "innerapi/user_detail/fuzzyQuery";
        try {
            Map<String, Object> param = Maps.newHashMap();
            param.put("idNo", idNo);
            String result = httpService.post(url, param);
            log.info("[user][findUserDetailByIdNo ] 请求业务系统返回值:{}", result);
            JsonResult jsonResult = GSON.fromJson(result, JsonResult.class);
            if (!Objects.isNull(jsonResult) && jsonResult.isSuccess()) {
                Object data = jsonResult.getData();
                if (data instanceof List) {

                    List list = (List) data;
                    if (list.size() > 1) {
                        return JsonResult.buildErrorStateResult("身份证查询返回多条数据，请用其他参数查询", null);
                    }
                    if (list.size() == 0) {
                        return JsonResult.buildErrorStateResult("不存在相关用户信息", null);
                    }
                    Object o = list.get(0);
                    XUserDetail userDetail = GSON.fromJson(o.toString(), XUserDetail.class);
                    UserBasicInfo userBasicInfo = new UserBasicInfo();
                    String createAt = DateUtil.format(userDetail.getCreatedAt(), DateUtil.DATE_FORMAT_1);
                    userBasicInfo.setCreateAt(createAt);
                    userBasicInfo.setGender(userDetail.getGender().getName());
                    userBasicInfo.setIdNo(userDetail.getIdNo());
                    userBasicInfo.setName(userDetail.getName());
                    userBasicInfo.setPhoneNo(userDetail.getPhoneNo());
                    userBasicInfo.setStatus(userDetail.getEnable() ? "正常" : "封禁");
                    return JsonResult.buildSuccessResult(null, userBasicInfo);
                }

            }

        } catch (Exception e) {
            log.error("[user][findUserDetailByIdNo ] 网络通讯异常,idNo:{},ex:{}", idNo, ExceptionUtils.getStackTrace(e));
            throw new BusinessException(ErrorCodeEnum.NET_ERROR);
        }

        return JsonResult.buildErrorStateResult("查询用户信息错误", null);
    }

    private JsonResult getUserBasicInfoResult(UserSysResult userSysResult) {
        Object data = getJsonResultData(userSysResult);
        UserBasicInfo userBasicInfo = new UserBasicInfo();
        if (data instanceof XUserDetail) {
            XUserDetail userDetail = (XUserDetail) data;
            String createAt = DateUtil.format(userDetail.getCreatedAt(), DateUtil.DATE_FORMAT_1);
            userBasicInfo.setCreateAt(createAt);
            userBasicInfo.setGender(userDetail.getGender().getName());
            userBasicInfo.setIdNo(userDetail.getIdNo());
            userBasicInfo.setName(userDetail.getName());
            userBasicInfo.setPhoneNo(userDetail.getPhoneNo());
            userBasicInfo.setStatus(userDetail.getEnable() ? "正常" : "封禁");
        } else {
            return (JsonResult) data;
        }


        return JsonResult.buildSuccessResult(null, userBasicInfo);
    }

    private Object getJsonResultData(UserSysResult userSysResult) {
        String logPre = "UserServiceImpl.getJsonResultData";
        log.info("{} 转换为对象 userSysResult={}", logPre, userSysResult);

        if (Objects.isNull(userSysResult)) {
            log.error("{} 转换参数为空 userSysResult={}", logPre, userSysResult);
            return JsonResult.buildErrorStateResult("远程调用结果为空", null);
        }

        if (!userSysResult.isSuccess() || Objects.isNull(userSysResult.getData())) {
            log.error("{} 远程调用失败 userSysResult={}", logPre, userSysResult);
            return JsonResult.buildErrorStateResult(userSysResult.getMsg(), userSysResult.getData());
        }
        return userSysResult.getData();
    }

    /**
     * 验证参数
     * 分两类 订单类查询参数  用户类查询参数 订单类参数优先级高
     * 每类参数只能有一个 如 订单类参数只能有一个（有申请订单号就不能有借款订单号和渠道订单号）   用户类同理
     *
     * @param userCombinationParam
     * @return
     */
    private Tuple<Boolean, String> validate(UserCombinationParam userCombinationParam) {
        Tuple<Boolean, String> result = new Tuple<>();
        int paramCount = 0;
        result.setKey(false);
        if (Objects.isNull(userCombinationParam)) {
            result.setValue("对象为空");
            return result;
        }

        if (StringUtils.isNotBlank(userCombinationParam.getChannelOrderNo())) {
            if (Objects.isNull(userCombinationParam.getChannelId())) {
                result.setValue("查询参数渠道订单号和渠道号要同时存在");
                return result;
            }
            paramCount++;
        }

        if (StringUtils.isNotBlank(userCombinationParam.getIdNo())) {
            if (IdcardUtils.validateIdCard18(userCombinationParam.getIdNo()) || IdcardUtils.validateIdCard15(userCombinationParam.getIdNo())) {
                paramCount++;
            } else {
                result.setValue("身份证号格式有误");
                return result;
            }
        }

        if (StringUtils.isNotBlank(userCombinationParam.getOrderNo())) {
            paramCount++;
        }

        if (!Objects.isNull(userCombinationParam.getUserId())) {
            paramCount++;
        }

        if (StringUtils.isNotBlank(userCombinationParam.getPhoneNo())) {
            if (ValidationUtil.validatePhoneNo(userCombinationParam.getPhoneNo())) {
                paramCount++;
            } else {
                result.setValue("手机号格式错误");
                return result;
            }
        }

        if (StringUtils.isNotBlank(userCombinationParam.getUuid())) {
            paramCount++;
        }

        if (!Objects.isNull(userCombinationParam.getLoanId())) {
            paramCount++;
        }

        //参数个数验证
        if (paramCount == 0) {
            result.setValue("参数全部为空");
            return result;
        }
        if (paramCount > 1) {
            result.setValue("参数个数大于一个");
            return result;
        }

        result.setKey(true);
        return result;
    }


    @Override
    public JsonResult findUserCombination(UserCombinationParam userCombinationParam) {
        String logPre = "UserServiceImpl.findUserCombination";
        log.info("{},综合查询 userCombinationParam={}", logPre, userCombinationParam);
        JsonResult userInfo = this.findUserInfo(userCombinationParam);
        if (!userInfo.isSuccess()) {
            return userInfo;
        }
        UserBasicInfo userBasicInfo = (UserBasicInfo) userInfo.getData();


        //申请订单查询（如果查询参数是渠道订单号）将渠道订单号转为借款订单号和申请订单号
        Long loanId = userCombinationParam.getLoanId();
        String applyNo = userCombinationParam.getOrderNo();
        if (StringUtils.isNotBlank(userCombinationParam.getChannelOrderNo())) {
            OrderMappingQueryParam orderMappingQueryParam = new OrderMappingQueryParam();
            orderMappingQueryParam.setChannelId(userCombinationParam.getChannelId());
            orderMappingQueryParam.setChannelOrderNo(userCombinationParam.getChannelOrderNo());

            JsonResult<LoanOrderMapping> orderMappingJsonResult = kaService.findOrderMapping(orderMappingQueryParam);
            LoanOrderMapping data;
            if (!orderMappingJsonResult.isSuccess() || Objects.isNull(data = orderMappingJsonResult.getData())) {
                log.error("{} 根据渠道号查询orderMapping失败 result={}", logPre, orderMappingJsonResult);
                return orderMappingJsonResult;
            }
            loanId = data.getLoanId();
            applyNo = data.getApplyNo();
        }
        //申请订单查询
        ApplyOrderQuery applyOrderQuery = new ApplyOrderQuery();
        applyOrderQuery.setLoanId(loanId);
        applyOrderQuery.setOrderNo(applyNo);
        applyOrderQuery.setUserId(userBasicInfo.getUserId());
        JsonResult applyOrders = xyqbService.findApplyOrders(applyOrderQuery);
        if (!applyOrders.isSuccess()) {
            return applyOrders;
        }

        List applyOrderList = (List) applyOrders.getData();

        UserCombination userCombination = new UserCombination();
        userCombination.setUserInfo(userBasicInfo);
        userCombination.setApplyOrderList(applyOrderList);

        return JsonResult.buildSuccessResult("", userCombination);
    }
}
