package cn.quantgroup.customer.service.impl;

import cn.quantgroup.customer.model.order.ApplyOrder;
import cn.quantgroup.customer.model.order.FlowNode;
import cn.quantgroup.customer.rest.param.applyorder.ApplyOrderQuery;
import cn.quantgroup.customer.rest.param.card.ModifyReservePhone;
import cn.quantgroup.customer.rest.param.card.UnBindCardModel;
import cn.quantgroup.customer.rest.param.repay.RepayOrderQuery;
import cn.quantgroup.customer.rest.vo.JsonResult;
import cn.quantgroup.customer.service.IXyqbService;
import cn.quantgroup.customer.service.http.IHttpService;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import static cn.quantgroup.customer.constant.Constant.GSON;

@Slf4j
@Service("xyqbService")
public class XyqbServiceImpl implements IXyqbService {

    @Value("${api.http}")
    private String xyqbSysUrl;

    private final IHttpService httpService;

    @Autowired
    public XyqbServiceImpl(IHttpService httpService) {
        this.httpService = httpService;
    }

    @Override
    public String findRepayOrders(RepayOrderQuery repayOrderQuery) {
        String phoneNo = repayOrderQuery.getPhoneNo();
        Long userId = repayOrderQuery.getUserId();
        String url = xyqbSysUrl + "/ex/repayment/repay_orders";
        Map<String, Object> param = Maps.newHashMap();
        if (StringUtils.isNotEmpty(phoneNo)) {
            param.put("phoneNo", phoneNo);
        }
        if (Objects.nonNull(userId)) {
            param.put("userId", userId);
        }
        String response = httpService.get(url, param);
        log.info("[xyqb][query repay orders ] param:{},请求业务系统返回值:{}", param, response);
        return response;
    }


    @Override
    public JsonResult<List<ApplyOrder>> findApplyOrders(ApplyOrderQuery applyOrderQuery) {
        String logPre="XyqbServiceImpl.findApplyOrders";
        log.info("{} 申请订单查询 applyOrderQuery={}",logPre,applyOrderQuery);
        String orderNo = applyOrderQuery.getOrderNo();
        Long loanId = applyOrderQuery.getLoanId();
        Long userId = applyOrderQuery.getUserId();
        String url = xyqbSysUrl + "/ex/inside/customer_sys/query/applyOrder";
        Map<String, Object> param = Maps.newHashMap();
        if (StringUtils.isNotEmpty(orderNo)) {
            param.put("orderNo", orderNo);
        }
        if (Objects.nonNull(userId)) {
            param.put("userId", userId);
        }
        if (Objects.nonNull(loanId)) {
            param.put("loanId", loanId);
        }
        Map<String, String> header = Maps.newHashMap();
        header.put("Content-type", "application/x-www-form-urlencoded");
        JsonResult jsonResult = httpService.post(url, header,param, JsonResult.class);
        if(Objects.isNull(jsonResult) || jsonResult.isSuccess()){
            log.error("{} 调用信用钱包失败 url={}, header={},param={},result={}",logPre,url, header,param,jsonResult);
            return JsonResult.buildErrorStateResult("申请订单查询失败",null);
        }
        List<ApplyOrder> applyOrderList = new ArrayList<>();
        List list = (List) jsonResult.getData();
        list.forEach(e ->{
            ApplyOrder applyOrder = GSON.fromJson(e.toString(), ApplyOrder.class);
            applyOrderList.add(applyOrder);
        });
        jsonResult.setData(applyOrderList);
        return jsonResult;
    }

    @Override
    public JsonResult<List<FlowNode>> findFlowChart(String orderNo) {
        String logPre="XyqbServiceImpl.findFlowChart";
        log.info("{} 流程日志查询 orderNo={}",logPre,orderNo);
        String url = xyqbSysUrl + "/ex/inside/customer_sys/query/applyOrder";
        if (StringUtils.isEmpty(orderNo)) {
            log.error("{} 申请订单号为空 orderNo={}",logPre,orderNo);
            return JsonResult.buildErrorStateResult("申请订单号为空",null);
        }
        Map<String, Object> param = Maps.newHashMap();
        param.put("orderNo", orderNo);

        Map<String, String> header = Maps.newHashMap();
        header.put("Content-type", "application/x-www-form-urlencoded");
        JsonResult jsonResult = httpService.post(url, header,param, JsonResult.class);

        if(Objects.isNull(jsonResult) || !jsonResult.isSuccess()){
            log.error("{} 调用信用钱包失败 url={}, header={},param={},result={}",logPre,url, header,param,jsonResult);
            return JsonResult.buildErrorStateResult("流程日志查询失败",null);
        }

        List data = (List) jsonResult.getData();
        List<FlowNode> flowNodeList = new ArrayList<>();
        data.forEach( e ->{
            flowNodeList.add(GSON.fromJson(e.toString(),FlowNode.class));
        });
        jsonResult.setData(flowNodeList);
        return jsonResult;

    }

    @Override
    public JsonResult findLoanDetail(Long loanId) {
        String logPre="XyqbServiceImpl.findLoanDetail";
        log.info("{} 提现订单详情 loanId={}",logPre,loanId);
        String url = xyqbSysUrl + "/ex/inside/customer_sys/loan/detail";
        if (Objects.isNull(loanId)) {
            log.error("{} 借款订单号为空 orderNo={}",logPre,loanId);
            return JsonResult.buildErrorStateResult("借款订单号为空",null);
        }
        Map<String, Object> param = Maps.newHashMap();
        param.put("loanId", loanId);

        Map<String, String> header = Maps.newHashMap();
        header.put("Content-type", "application/x-www-form-urlencoded");
        JsonResult jsonResult = httpService.post(url, header,param, JsonResult.class);

        if(Objects.isNull(jsonResult) || jsonResult.isSuccess()){
            log.error("{} 调用信用钱包失败 url={}, header={},param={},result={}",logPre,url, header,param,jsonResult);
            return JsonResult.buildErrorStateResult("提现订单详情查询失败",null);
        }
        //data数据结构 LoanOrderDetail
        return jsonResult;
    }

    /**
     * 查询绑定卡列表
     *
     * @param userId
     * @return
     */
    @Override
    public String queryCardLists(Long userId) {
        String url = xyqbSysUrl + "/ex/inside/customer_sys/query/bankcard";
        Map<String, Object> param = new HashMap<>(4);
        if (Objects.nonNull(userId)) {
            param.put("userId", userId);
        }
        String response = null;
        try {
            response = httpService.post(url, param);
            log.info("[queryCardLists] param:{},请求业务系统返回值:{}", param, response);
        } catch (Exception e) {
            log.error("[queryCardLists] param:{},请求业务系统异常:{}", param, e);
        }
        return response;
    }

    @Override
    public String queryCardRelation(Long bankId) {
        String url = xyqbSysUrl + "/ex/inside/customer_sys/query/cardrelation";
        Map<String, Object> param = new HashMap<>(4);
        if (Objects.nonNull(bankId)) {
            param.put("bankId", bankId);
        }

        String response = null;
        try {
            response = httpService.post(url, param);
            log.info("[queryCardRelation] param:{},请求业务系统返回值:{}", param, response);
        } catch (Exception e) {
            log.error("[queryCardRelation] param:{},请求业务系统异常:{}", param, e);
        }
        return response;
    }

    @Override
    public String unBindCardRelation(UnBindCardModel unBindCardModel) {
        String url = xyqbSysUrl + "/ex/inside/customer_sys/unbundle/relation";
        Map<String, Object> param = new HashMap<>(4);
        Long userId = unBindCardModel.getUserId();
        Long bindCardRecordId = unBindCardModel.getBindCardRecordId();
        if (Objects.nonNull(bindCardRecordId)) {
            param.put("bindCardRecordId", bindCardRecordId);
        }
        if (Objects.nonNull(userId)) {
            param.put("userId", userId);
        }
        String response = null;
        try {
            response = httpService.post(url, param);
            log.info("[unBindCardRelation] param:{},请求业务系统返回值:{}", param, response);
        } catch (Exception e) {
            log.error("[unBindCardRelation] param:{},请求业务系统异常:{}", param, e);
        }
        return response;
    }

    @Override
    public String modifyReservePhone(ModifyReservePhone modifyReservePhone) {
        String url = xyqbSysUrl + "/ex/inside/customer_sys/modify/reservedphone";
        Map<String, Object> param = new HashMap<>(8);
        Long userId = modifyReservePhone.getUserId();
        Long bankId = modifyReservePhone.getBankId();
        String phoneNo = modifyReservePhone.getPhoneNo();
        if (Objects.nonNull(bankId)) {
            param.put("bankId", bankId);
        }
        if (Objects.nonNull(userId)) {
            param.put("userId", userId);
        }
        if (StringUtils.isNotBlank(phoneNo)) {
            param.put("phoneNo", phoneNo);
        }

        String response = null;
        try {
            response = httpService.post(url, param);
            log.info("[modifyReservePhone] param:{},请求业务系统返回值:{}", param, response);
        } catch (Exception e) {
            log.error("[modifyReservePhone] param:{},请求业务系统异常:{}", param, e);
        }
        return response;
    }


}
