package cn.quantgroup.customer.service.impl;

import cn.quantgroup.customer.enums.ErrorCodeEnum;
import cn.quantgroup.customer.model.kaordermapping.ApplyRequestHistory;
import cn.quantgroup.customer.model.kaordermapping.LoanOrderMapping;
import cn.quantgroup.customer.model.order.*;
import cn.quantgroup.customer.rest.param.ordermapping.EarlySettleUpOrderQueryParam;
import cn.quantgroup.customer.rest.param.ordermapping.OperateEntryParam;
import cn.quantgroup.customer.rest.param.ordermapping.OrderQueryParam;
import cn.quantgroup.customer.rest.vo.JsonResult;
import cn.quantgroup.customer.service.IKaService;
import cn.quantgroup.customer.service.IOrderService;
import cn.quantgroup.customer.service.IXyqbService;
import cn.quantgroup.customer.util.DateUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.security.Timestamp;
import java.util.*;

/**
 * @author Wang Xiangwei
 * @version 2020/3/10
 */
@Slf4j
@Service
public class OrderServiceImpl implements IOrderService {
    @Autowired
    private IKaService kaService;

    @Autowired
    private IXyqbService xyqbService;

    @Override
    public JsonResult<List<FlowNode>> findFlowChart(String applyOrderNo) {
        String logPre = "OrderServiceImpl.findFlowChart";
        log.info("{} 查询流程图 applyOrderNo={}", logPre, applyOrderNo);
        OrderQueryParam param = new OrderQueryParam();
        param.setApplyOrderNo(applyOrderNo);
        JsonResult<LoanOrderMapping> orderMappingResult = kaService.findOrderMapping(param);
        LoanOrderMapping head = null;
        if (orderMappingResult.isSuccess()) {
            head = orderMappingResult.getData();
            log.info("{},申请订单号查询orderMapping applyOrderNo={},data={}", logPre, applyOrderNo, head);
        }

        JsonResult<List<FlowNode>> flowChart = xyqbService.findFlowChart(applyOrderNo);
        if (!flowChart.isSuccess()) {
            log.error("{} 流程图查询失败 result={}", logPre, flowChart);
            return flowChart;
        }

        List<FlowNode> flowNodeList = flowChart.getData();
        if (Objects.nonNull(head)) {
            FlowNode flowNode = new FlowNode();
            flowNode.setName("渠道用户信息导入");
            flowNode.setStatus(1);
            flowNode.setNode("IMPORT_INFO");
            flowNode.setRemark("进件成功");
            flowNodeList.add(0, flowNode);

        }

        return JsonResult.buildSuccessResult("", flowNodeList);
    }


    @Override
    public JsonResult<LoanOrderDetail> getLoanOrderDetail(Long loanId) {
        String logPre = "OrderServiceImpl.getLoanOrderDetail";
        log.info("{} 提现详情 loanId={}", logPre, loanId);
        JsonResult<LoanOrder> loanOrder = xyqbService.findLoanOrder(loanId);

        if (!loanOrder.isSuccess()) {
            log.error("{} 提现订单查询失败 loanId={}, result={}", logPre, loanId, loanOrder);
            return JsonResult.buildErrorStateResult(loanOrder.getMsg(), loanOrder.getData());
        }

        LoanOrderDetail loanOrderDetail = new LoanOrderDetail();
        List<Repayment> repaymentList = new ArrayList<>();
        LoanOrder loanOrderData = loanOrder.getData();
        loanOrderDetail.setLoanOrder(loanOrderData);
        loanOrderDetail.setRepaymentList(repaymentList);

        if (Objects.isNull(loanOrderData) || !loanOrderData.getShowPlans()) {
            log.info("{} 不需查询还款计划", logPre);
            return JsonResult.buildSuccessResult("查询成功", loanOrderDetail);
        }

        //还款计划查询
        JsonResult<OrderRepayment> orderRepaymentJsonResult = xyqbService.repaymentPlanQuery(loanId);
        if (!orderRepaymentJsonResult.isSuccess()) {
            log.error("{} 还款计划查询失败 result={}", logPre, orderRepaymentJsonResult);
            return JsonResult.buildErrorStateResult(orderRepaymentJsonResult.getMsg(), orderRepaymentJsonResult.getData());
        }

        List<RepaymentPlanItem> repaymentPlans = orderRepaymentJsonResult.getData().getRepaymentPlans();
        repaymentPlans.forEach(e -> repaymentList.add(Repayment.valueOf(e)));

        return JsonResult.buildSuccessResult("查询成功", loanOrderDetail);
    }

    @Override
    public JsonResult<OrderInfoVo> queryOrderInfo(OrderQueryParam orderQuery) {
        String logPre = "OrderServiceImpl.queryOrderInfo 查询渠道订单信息";
        if (Objects.isNull(orderQuery)
                || (StringUtils.isBlank(orderQuery.getApplyOrderNo())
                && StringUtils.isBlank(orderQuery.getApplyOrderNo())
                && Objects.isNull(orderQuery.getLoanId())
                && Objects.isNull(orderQuery.getChannelId()))) {
            log.error("{} 请求参数为空 orderQuery={}", logPre, orderQuery);
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.PARAM_ERROR.getMessage(), ErrorCodeEnum.PARAM_ERROR.getCode());
        }

        // 查询订单信息  查询ka
        JsonResult<LoanOrderMapping> orderMappingResult = kaService.findOrderMapping(orderQuery);
        LoanOrderMapping loanOrderMapping;
        if (!orderMappingResult.isSuccess()) {
            log.warn("{} 查询订单信息orderMapping失败 result={} orderQuery={}", logPre, orderMappingResult, orderQuery);
            return JsonResult.buildErrorStateResult(orderMappingResult.getMsg(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }

        loanOrderMapping = orderMappingResult.getData();
        if (Objects.nonNull(loanOrderMapping)) {
            return buildHasOrderMapping(orderQuery, loanOrderMapping);
        } else {
            // 可能两种结果 1)进件失败 此时需要根据渠道订单号和渠道号查询进件流水  2)不存在该订单
            if (StringUtils.isNotBlank(orderQuery.getChannelOrderNo()) && Objects.nonNull(orderQuery.getChannelId())) {
                return buildNoOrderMapping(orderQuery.getChannelOrderNo(), orderQuery.getChannelId());
            } else {
                // 不存在订单信息
                return JsonResult.buildSuccessResult("订单不存在", null);
            }
        }
    }

    @Override
    public JsonResult queryEarlySettleUpOrders(EarlySettleUpOrderQueryParam orderQuery) {
        List<EarlySettleUpOrder> list = new ArrayList<>();
        EarlySettleUpOrder settleUpOrder = new EarlySettleUpOrder();
        settleUpOrder.setLoanId(123L);
        settleUpOrder.setAllTerm(12);
        settleUpOrder.setChannelId(159881L);
        settleUpOrder.setChannelName("拍拍贷");
        settleUpOrder.setChannelOrderNo("PP123");
        settleUpOrder.setFundId(900L);
        settleUpOrder.setFundName("华融");
        settleUpOrder.setFundOrderNo("HR9001");
        settleUpOrder.setLoanAmount(new BigDecimal(10000));
        settleUpOrder.setPhoneNo("17700021123");
        settleUpOrder.setHideEntryFlag(Boolean.FALSE);
        settleUpOrder.setShowEntryFlag(Boolean.FALSE);
        settleUpOrder.setNoPaidTerm(12);
        list.add(settleUpOrder);
        return JsonResult.buildSuccessResult("处理成功", list);
    }

    @Override
    public JsonResult operateEntry(OperateEntryParam operateEntryParam) {
        return JsonResult.buildSuccessResult("处理成功", Boolean.TRUE);
    }

    @Override
    public JsonResult earlySettleUpTrial(Long loanId) {
        EarlySettleUpTrial settleUpTrial = new EarlySettleUpTrial();
        settleUpTrial.setPrincipal(new BigDecimal(10000));
        settleUpTrial.setInterest(new BigDecimal(10));
        settleUpTrial.setOverdueInterest(new BigDecimal(10));
        settleUpTrial.setOtherFee(new BigDecimal(10));
        settleUpTrial.setServiceFee(new BigDecimal(10));
        settleUpTrial.setLiquid(new BigDecimal(10));
        return JsonResult.buildSuccessResult("处理成功", settleUpTrial);
    }

    @Override
    public JsonResult queryOperateLog(Long loanId) {
        List<OperateLog> list = new ArrayList<>();
        OperateLog operateLog = new OperateLog();
        operateLog.setAccount("10000");
        operateLog.setCreateAt(DateUtil.getCurrentTimestamp());
        operateLog.setName("李四");
        operateLog.setOpName("修改用户密码");
        operateLog.setRemark("客户要求");
        list.add(operateLog);
        return JsonResult.buildSuccessResult("处理成功", list);
    }

    /**
     * 处理查询不到orderMapping的情况，可能是进件失败，也可能不存在该笔订单
     *
     * @param channelOrderNo
     * @param channelId
     * @return
     */
    private JsonResult<OrderInfoVo> buildNoOrderMapping(String channelOrderNo, Long channelId) {
        String logPre = "OrderServiceImpl.buildNoOrderMapping";
        // 查询进件流水信息  查询ka
        JsonResult<ApplyRequestHistory> orderMappingResult = kaService.queryLatestChannelRequestFlows(channelOrderNo, channelId);
        if (!orderMappingResult.isSuccess()) {
            log.error("{} 查询渠道进件流水信息异常 result={} channelOrderNo={},channelId={}", logPre, orderMappingResult, channelOrderNo, channelId);
            return JsonResult.buildErrorStateResult("查询渠道进件流水信息异常", ErrorCodeEnum.RETURN_ERROR.getCode());
        }

        ApplyRequestHistory applyRequestHistory = orderMappingResult.getData();
        // 不存在进件流水信息 返回查询不到该笔订单
        if (Objects.isNull(applyRequestHistory)) {
            return JsonResult.buildSuccessResult("订单不存在", null);
        }

        OrderInfo orderInfo = OrderInfo.builder()
                .channelOrderNo(channelOrderNo)
                .channelId(channelId)
                .status(applyRequestHistory.getIsSuccess() ? "进件成功" : "进件失败")
                .remark(applyRequestHistory.getDescriptionDetail())
                .build();

        OrderInfoVo orderInfoVo = OrderInfoVo.builder()
                .orderInfo(orderInfo)
                .build();
        log.info("{} 查询订单信息,返回信息 orderInfoVo={}", logPre, orderInfoVo);
        return JsonResult.buildSuccessResult("处理成功", orderInfoVo);
    }

    /**
     * 处理渠道订单表有数据的情形
     *
     * @param orderQuery
     * @param loanOrderMapping
     * @return
     */
    private JsonResult<OrderInfoVo> buildHasOrderMapping(OrderQueryParam orderQuery, LoanOrderMapping loanOrderMapping) {
        String logPre = "OrderServiceImpl.buildHasOrderMapping";

        // 查询订单状态  xyqb
        orderQuery.setApplyOrderNo(loanOrderMapping.getApplyNo());
        orderQuery.setLoanId(loanOrderMapping.getLoanId());
        orderQuery.setChannelId(loanOrderMapping.getRegisteredFrom());
        JsonResult<OrderStatus> orderStatusResult = xyqbService.orderStatusQuery(orderQuery);
        OrderStatus orderStatus;
        if (!orderStatusResult.isSuccess()) {
            log.error("{} 查询订单状态 orderStatus 失败 result={} orderQuery={}", logPre, orderStatusResult, orderQuery);
            return JsonResult.buildErrorStateResult(orderStatusResult.getMsg(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }

        orderStatus = orderStatusResult.getData();

        // 查询通知记录  查询ka
        JsonResult<CallbackRecordList> callbackListResult = kaService.querySendRecords(loanOrderMapping.getApplyNo(), loanOrderMapping.getRegisteredFrom());
        if (!callbackListResult.isSuccess()) {
            log.error("{} 查询发送记录失败  result={} orderQuery={}", logPre, callbackListResult, orderQuery);
            return JsonResult.buildErrorStateResult("查询通知记录失败", ErrorCodeEnum.RETURN_ERROR.getCode());
        }

        final String defaultStatusStr = "进件成功";
        String status = Objects.nonNull(orderStatus) ? orderStatus.getStatus().getValue() : defaultStatusStr;
        OrderInfo orderInfo = OrderInfo.builder()
                .channelOrderNo(loanOrderMapping.getChannelOrderNo())
                .applyNo(loanOrderMapping.getApplyNo())
                .channelId(loanOrderMapping.getRegisteredFrom())
                .loanId(loanOrderMapping.getLoanId())
                .status(status)
                .remark(status)
                .build();

        OrderInfoVo orderInfoVo = OrderInfoVo.builder()
                .orderInfo(orderInfo)
                .noticeList(callbackListResult.getData())
                .build();
        log.info("{} 查询订单信息,返回信息 orderInfoVo={}", logPre, orderInfoVo);
        return JsonResult.buildSuccessResult("处理成功", orderInfoVo);
    }


}
