package cn.quantgroup.customer.rest;

import cn.quantgroup.customer.entity.User;
import cn.quantgroup.customer.enums.ErrorCodeEnum;
import cn.quantgroup.customer.rest.param.LoginParam;
import cn.quantgroup.customer.rest.param.ModifyPhoneAudit;
import cn.quantgroup.customer.rest.param.ModifyPhoneFeedback;
import cn.quantgroup.customer.rest.param.ModifyPhoneQuery;
import cn.quantgroup.customer.rest.vo.AuthUserVo;
import cn.quantgroup.customer.service.IUserService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.annotation.AuthenticationPrincipal;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import javax.validation.Valid;

import static cn.quantgroup.customer.constant.Constant.*;

@Slf4j
@RestController
@RequestMapping("/user")
public class UserRest {

    private final IUserService userService;
    private final AuthenticationManager authenticationManager;

    public UserRest(AuthenticationManager authenticationManager, IUserService userService) {
        this.authenticationManager = authenticationManager;
        this.userService = userService;
    }

    @PostMapping(value = "/login")
    public JsonResult login(@Valid @ModelAttribute LoginParam loginParam, HttpServletRequest request) {
        UsernamePasswordAuthenticationToken authRequest = new UsernamePasswordAuthenticationToken(loginParam.getUsername(), loginParam.getPassword());
        Authentication authentication = authenticationManager.authenticate(authRequest);
        SecurityContextHolder.getContext().setAuthentication(authentication);
        HttpSession session = request.getSession();
        session.setAttribute(SPRING_SECURITY_CONTEXT, SecurityContextHolder.getContext());

        if (authentication != null && authentication.isAuthenticated()) {
            User principal = (User) authentication.getPrincipal();
            AuthUserVo authUserVo = AuthUserVo.parse(principal);
            return JsonResult.buildSuccessResult(LOGIN_SUCCESS, authUserVo);
        }
        return JsonResult.buildErrorStateResult(LOGIN_FAIL, null);
    }

    @PostMapping(value = "/logout")
    public JsonResult logout(@AuthenticationPrincipal User user) {
        return JsonResult.buildSuccessResult(LOGOUT_SUCCESS, null);
    }


    @GetMapping(value = "/logout-success")
    public JsonResult logoutSuccess() {
        return JsonResult.buildSuccessResult(LOGOUT_SUCCESS, null);
    }


    /**
     * 修改手机号-查询工单
     *
     * @param modifyPhoneQuery
     * @return
     */
    @GetMapping(value = "/modify/phone_no")
    public JsonResult modifyPhoneQuery(ModifyPhoneQuery modifyPhoneQuery) {
        String response = userService.modifyPhoneQuery(modifyPhoneQuery);
        if (StringUtils.isEmpty(response)) {
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.RETURN_ERROR.getMessage(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }
        return GSON.fromJson(response, JsonResult.class);
    }

    /**
     * 修改手机号-人工审核
     *
     * @param modifyPhoneAudit
     * @return
     */
    @PostMapping(value = "/modify/phone_no/audit")
    public JsonResult modifyPhoneAudit(ModifyPhoneAudit modifyPhoneAudit) {
        String response = userService.modifyPhoneAudit(modifyPhoneAudit);
        if (StringUtils.isEmpty(response)) {
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.RETURN_ERROR.getMessage(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }
        return GSON.fromJson(response, JsonResult.class);
    }

    /**
     * 修改手机号-反馈跟进
     *
     * @param modifyPhoneFeedback
     * @return
     */
    @PostMapping(value = "/modify/phone_no/feedback")
    public JsonResult modifyPhoneFeedback(ModifyPhoneFeedback modifyPhoneFeedback) {
        String response = userService.modifyPhoneFeedback(modifyPhoneFeedback);
        if (StringUtils.isEmpty(response)) {
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.RETURN_ERROR.getMessage(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }
        return GSON.fromJson(response, JsonResult.class);
    }


}
