package cn.quantgroup.customer.service.impl;

import cn.quantgroup.customer.constant.Constant;
import cn.quantgroup.customer.entity.OpLog;
import cn.quantgroup.customer.enums.ErrorCodeEnum;
import cn.quantgroup.customer.model.kaordermapping.ApplyRequestHistory;
import cn.quantgroup.customer.model.kaordermapping.LoanOrderMapping;
import cn.quantgroup.customer.model.order.*;
import cn.quantgroup.customer.model.xyqbuser.UserBasicInfo;
import cn.quantgroup.customer.rest.param.ordermapping.EarlySettleUpOrderQueryParam;
import cn.quantgroup.customer.rest.param.ordermapping.OperateEntryParam;
import cn.quantgroup.customer.rest.param.ordermapping.OrderQueryParam;
import cn.quantgroup.customer.rest.param.user.UserQueryParam;
import cn.quantgroup.customer.rest.vo.JsonResult;
import cn.quantgroup.customer.service.*;
import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;

/**
 * @author Wang Xiangwei
 * @version 2020/3/10
 */
@Slf4j
@Service
public class OrderServiceImpl implements IOrderService {
    @Autowired
    private IKaService kaService;

    @Autowired
    private IXyqbService xyqbService;

    @Autowired
    private IOperateLogService operateLogService;

    @Autowired
    private IUserService userService;


    @Override
    public JsonResult<List<FlowNode>> findFlowChart(String applyOrderNo) {
        String logPre = "OrderServiceImpl.findFlowChart";
        log.info("{} 查询流程图 applyOrderNo={}", logPre, applyOrderNo);
        OrderQueryParam param = new OrderQueryParam();
        param.setApplyOrderNo(applyOrderNo);
        JsonResult<LoanOrderMapping> orderMappingResult = kaService.findOrderMapping(param);
        LoanOrderMapping head = null;
        if (orderMappingResult.isSuccess()) {
            head = orderMappingResult.getData();
            log.info("{},申请订单号查询orderMapping applyOrderNo={},data={}", logPre, applyOrderNo, head);
        }

        JsonResult<List<FlowNode>> flowChart = xyqbService.findFlowChart(applyOrderNo);
        if (!flowChart.isSuccess()) {
            log.error("{} 流程图查询失败 result={}", logPre, flowChart);
            return flowChart;
        }

        List<FlowNode> flowNodeList = flowChart.getData();
        if (Objects.nonNull(head)) {
            FlowNode flowNode = new FlowNode();
            flowNode.setName("渠道用户信息导入");
            flowNode.setStatus(1);
            flowNode.setNode("IMPORT_INFO");
            flowNode.setRemark("进件成功");
            flowNodeList.add(0, flowNode);

        }

        return JsonResult.buildSuccessResult("", flowNodeList);
    }


    @Override
    public JsonResult<LoanOrderDetail> getLoanOrderDetail(Long loanId) {
        String logPre = "OrderServiceImpl.getLoanOrderDetail";
        log.info("{} 提现详情 loanId={}", logPre, loanId);
        JsonResult<LoanOrder> loanOrder = xyqbService.findLoanOrder(loanId);

        if (!loanOrder.isSuccess()) {
            log.error("{} 提现订单查询失败 loanId={}, result={}", logPre, loanId, loanOrder);
            return JsonResult.buildErrorStateResult(loanOrder.getMsg(), loanOrder.getData());
        }

        LoanOrderDetail loanOrderDetail = new LoanOrderDetail();
        List<Repayment> repaymentList = new ArrayList<>();
        LoanOrder loanOrderData = loanOrder.getData();
        loanOrderDetail.setLoanOrder(loanOrderData);
        loanOrderDetail.setRepaymentList(repaymentList);

        if (Objects.isNull(loanOrderData) || !loanOrderData.getShowPlans()) {
            log.info("{} 不需查询还款计划", logPre);
            return JsonResult.buildSuccessResult("查询成功", loanOrderDetail);
        }

        //还款计划查询
        JsonResult<OrderRepayment> orderRepaymentJsonResult = xyqbService.repaymentPlanQuery(loanId);
        if (!orderRepaymentJsonResult.isSuccess()) {
            log.error("{} 还款计划查询失败 result={}", logPre, orderRepaymentJsonResult);
            return JsonResult.buildErrorStateResult(orderRepaymentJsonResult.getMsg(), orderRepaymentJsonResult.getData());
        }

        List<RepaymentPlanItem> repaymentPlans = orderRepaymentJsonResult.getData().getRepaymentPlans();
        repaymentPlans.forEach(e -> repaymentList.add(Repayment.valueOf(e)));

        return JsonResult.buildSuccessResult("查询成功", loanOrderDetail);
    }

    @Override
    public JsonResult<OrderInfoVo> queryOrderInfo(OrderQueryParam orderQuery) {
        String logPre = "OrderServiceImpl.queryOrderInfo 查询渠道订单信息";
        if (Objects.isNull(orderQuery)
                || (StringUtils.isBlank(orderQuery.getApplyOrderNo())
                && StringUtils.isBlank(orderQuery.getApplyOrderNo())
                && Objects.isNull(orderQuery.getLoanId())
                && Objects.isNull(orderQuery.getChannelId()))) {
            log.error("{} 请求参数为空 orderQuery={}", logPre, orderQuery);
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.PARAM_ERROR.getMessage(), ErrorCodeEnum.PARAM_ERROR.getCode());
        }

        // 查询订单信息  查询ka
        JsonResult<LoanOrderMapping> orderMappingResult = kaService.findOrderMapping(orderQuery);
        LoanOrderMapping loanOrderMapping;
        if (!orderMappingResult.isSuccess()) {
            log.warn("{} 查询订单信息orderMapping失败 result={} orderQuery={}", logPre, orderMappingResult, orderQuery);
            return JsonResult.buildErrorStateResult(orderMappingResult.getMsg(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }

        loanOrderMapping = orderMappingResult.getData();
        if (Objects.nonNull(loanOrderMapping)) {
            return buildHasOrderMapping(orderQuery, loanOrderMapping);
        } else {
            // 可能两种结果 1)进件失败 此时需要根据渠道订单号和渠道号查询进件流水  2)不存在该订单
            if (StringUtils.isNotBlank(orderQuery.getChannelOrderNo()) && Objects.nonNull(orderQuery.getChannelId())) {
                return buildNoOrderMapping(orderQuery.getChannelOrderNo(), orderQuery.getChannelId());
            } else {
                // 不存在订单信息
                return JsonResult.buildSuccessResult("订单不存在", null);
            }
        }
    }

    @Override
    public JsonResult<List<EarlySettleUpOrder>> queryEarlySettleUpOrders(EarlySettleUpOrderQueryParam orderQuery) {
        if (!existAtLestOneParam(orderQuery)) {
            return JsonResult.buildErrorStateResult("至少输入一个请求参数", null);
        }

        // 渠道订单号和渠道号必须同时存在
        if (!existChannelOrderNoAndChannelId(orderQuery)) {
            return JsonResult.buildErrorStateResult("渠道号和渠道订单号必须同时存在", null);
        }

        // 获取用户userId
        boolean isExistUserParam = existUserParam(orderQuery);
        if (isExistUserParam) {
            JsonResult<UserBasicInfo> userInfo = this.queryAndCheckUserInfo(orderQuery);
            if (Objects.isNull(userInfo) || !userInfo.isSuccess()) {
                return JsonResult.buildErrorStateResult("查询不到用户信息", null);
            }
            UserBasicInfo userBasicInfo = userInfo.getData();
            if (Objects.nonNull(userBasicInfo)) {
                orderQuery.setPhoneNo(userBasicInfo.getPhoneNo());
                orderQuery.setUserId(userBasicInfo.getUserId());
            }

        }


        // 获取订单loanId
        boolean isExistOrderParam = existOrderParam(orderQuery);
        if (isExistOrderParam) {
            JsonResult<LoanOrderMapping> loanOrder = this.queryAndCheckOrderInfo(orderQuery);
            if (Objects.isNull(loanOrder) || !loanOrder.isSuccess()) {
                return JsonResult.buildErrorStateResult("查询不到订单信息", null);
            }

            LoanOrderMapping loanOrderMapping = loanOrder.getData();
            if (Objects.nonNull(loanOrderMapping)) {
                orderQuery.setLoanId(loanOrderMapping.getLoanId());
                orderQuery.setChannelId(loanOrderMapping.getRegisteredFrom());
                orderQuery.setChannelOrderNo(loanOrderMapping.getChannelOrderNo());
            }
        }

        // 请求xyqb
        JsonResult<List<EarlySettleUpOrder>> xyqbOrderResult = this.xyqbService.findLoanOrder4EarlySettleUp(orderQuery);
        if (Objects.isNull(xyqbOrderResult)) {
            return JsonResult.buildErrorStateResult("查询订单信息出错", null);
        }

        if(!xyqbOrderResult.isSuccess()){
            return xyqbOrderResult;
        }

        /**
         * 处理数据转换
         */
        List<EarlySettleUpOrder> settleUpOrders = xyqbOrderResult.getData();
        settleUpOrders.forEach(e -> {
            e.setChannelOrderNo(orderQuery.getChannelOrderNo());
            e.setPhoneNo(orderQuery.getPhoneNo());
            e.setShowOplog(dealWithShowOplog(e.getLoanId()));
        });

        return JsonResult.buildSuccessResult("处理成功", settleUpOrders);
    }

    /**
     * 处理是否显示  操作日志按钮
     *
     * @param loanId
     * @return
     */
    private Boolean dealWithShowOplog(Long loanId) {
        return operateLogService.findLogsCountsByLoanId(loanId) > 0;
    }


    /**
     * 查询订单loanId
     *
     * @param orderQuery
     * @return
     */
    private JsonResult<LoanOrderMapping> queryAndCheckOrderInfo(EarlySettleUpOrderQueryParam orderQuery) {
        String logPre = "OrderServiceImpl.queryAndCheckOrderInfo";

        OrderQueryParam orderQueryParam = new OrderQueryParam();
        orderQueryParam.setLoanId(orderQuery.getLoanId());
        orderQueryParam.setChannelId(orderQuery.getChannelId());
        orderQueryParam.setChannelOrderNo(orderQuery.getChannelOrderNo());
        JsonResult<LoanOrderMapping> orderMappingJsonResult = this.kaService.findOrderMapping(orderQueryParam);
        if (Objects.isNull(orderMappingJsonResult) || !orderMappingJsonResult.isSuccess()) {
            log.error("{} 查询订单失败 result={}", logPre, orderMappingJsonResult);
            return JsonResult.buildErrorStateResult("请求参数有误,查询不到订单信息", null);
        }

        return orderMappingJsonResult;
    }


    /**
     * 查询用户userId
     *
     * @param orderQuery
     * @return
     */
    private JsonResult<UserBasicInfo> queryAndCheckUserInfo(EarlySettleUpOrderQueryParam orderQuery) {
        String logPre = "OrderServiceImpl.queryAndCheckUserInfo";

        //用户参数查询
        UserQueryParam userQueryParam = new UserQueryParam();
        userQueryParam.setPhoneNo(orderQuery.getPhoneNo());
        userQueryParam.setUserId(orderQuery.getUserId());
        JsonResult<UserBasicInfo> userInfoByUserParam = this.userService.findUserInfo(userQueryParam);
        if (Objects.isNull(userInfoByUserParam) || !userInfoByUserParam.isSuccess()) {
            log.error("{} 查询用户失败 result={}", logPre, userInfoByUserParam);
            return JsonResult.buildErrorStateResult("请求参数有误,查询不到用户信息", null);
        }

        UserBasicInfo userBasicInfo = userInfoByUserParam.getData();
        if (StringUtils.isNotBlank(orderQuery.getPhoneNo())) {
            if (!orderQuery.getPhoneNo().equals(userBasicInfo.getPhoneNo())) {
                log.error("{} 用户输入手机号和查询结果不一致 orderQuery.getPhoneNo()={}, userBasicInfo={}", logPre, orderQuery.getPhoneNo(), userBasicInfo);
                return JsonResult.buildErrorStateResult("请求参数有误,查询不到用户信息", null);
            }
        }

        if (Objects.nonNull(orderQuery.getUserId())) {
            if (!orderQuery.getUserId().equals(userBasicInfo.getUserId())) {
                log.error("{} 用户输入userId和查询结果不一致 orderQuery.getUserId()={}, userBasicInfo={}", logPre, orderQuery.getUserId(), userBasicInfo);
                return JsonResult.buildErrorStateResult("请求参数有误,查询不到用户信息", null);
            }
        }

        return userInfoByUserParam;

    }

    /**
     * 判断是否存在用户基础字段
     *
     * @param orderQuery
     * @return
     */
    private boolean existUserParam(EarlySettleUpOrderQueryParam orderQuery) {
        return !(StringUtils.isBlank(orderQuery.getPhoneNo()) && Objects.isNull(orderQuery.getUserId()));
    }

    /**
     * 判断是否存在订单基础字段
     *
     * @param orderQuery
     * @return
     */
    private boolean existOrderParam(EarlySettleUpOrderQueryParam orderQuery) {
        return !(Objects.isNull(orderQuery.getLoanId()) && Objects.isNull(orderQuery.getChannelId()) && StringUtils.isBlank(orderQuery.getChannelOrderNo()));
    }

    /**
     * 渠道号和渠道订单号同时存在判断
     *
     * @param orderQuery
     * @return
     */
    private boolean existChannelOrderNoAndChannelId(EarlySettleUpOrderQueryParam orderQuery) {
        return !(Objects.isNull(orderQuery.getChannelId()) && StringUtils.isNotBlank(orderQuery.getChannelOrderNo())
                || Objects.nonNull(orderQuery.getChannelId()) && StringUtils.isBlank(orderQuery.getChannelOrderNo())
        );
    }

    private boolean existAtLestOneParam(EarlySettleUpOrderQueryParam orderQuery) {
        return existOrderParam(orderQuery) || existUserParam(orderQuery) || StringUtils.isNotBlank(orderQuery.getFundOrder());
    }

    @Override
    public JsonResult<Boolean> setOrCancelSettleUpWhiteList(OperateEntryParam operateEntryParam) {
        return this.xyqbService.setOrCancelSettleUpWhiteList(operateEntryParam);
    }

    @Override
    public JsonResult<List<Map<String, Object>>> earlySettleUpTrial(Long loanId) {
        String logPre = "OrderServiceImpl.earlySettleUpTrial";

        JsonResult<EarlySettleUpTrial> jsonResult = this.xyqbService.earlySettleUpTrial(loanId);
        if (Objects.isNull(jsonResult)) {
            log.error("{} 获取试算结果失败 loanId={}", logPre, loanId);
            return JsonResult.buildErrorStateResult("获取试算结果失败", null);
        }

        if(!jsonResult.isSuccess()){
            return JsonResult.buildErrorStateResult(jsonResult.getMsg(), null);
        }

        List<Map<String, Object>> data = dealWithFieldAndTitle(jsonResult.getData());
        log.info("{} 试算对象 data={},试算后返回 datas:{}", logPre, jsonResult.getData(), data);
        return JsonResult.buildSuccessResult("处理成功", data);
    }

    /**
     * 根据返回参数值，获取对应title
     *
     * @param data
     * @return
     */
    private List<Map<String, Object>> dealWithFieldAndTitle(EarlySettleUpTrial data) {
        JSONObject dataJson = JSONObject.parseObject(JSONObject.toJSONString(data));
        Iterator<Map.Entry<String, Object>> iterator = dataJson.entrySet().iterator();
        List<Map<String, Object>> dataList = new ArrayList<>();

        while (iterator.hasNext()) {
            Map.Entry<String, Object> me = iterator.next();
            String key = me.getKey();
            Object value = me.getValue();
            if (Objects.nonNull(value) && new BigDecimal(String.valueOf(value)).compareTo(new BigDecimal(0)) > 0) {
                Map<String, Object> fieldDataMap = new HashMap<>(4);
                if ("principal".equals(key)) {
                    fieldDataMap.put("title", "应还本金");
                } else if ("interest".equals(key)) {
                    fieldDataMap.put("title", "应还利息");
                } else if ("serviceFee".equals(key)) {
                    fieldDataMap.put("title", "应还担保费");
                } else if ("premium".equals(key)) {
                    fieldDataMap.put("title", "应还保费");
                } else if ("overDueInterest".equals(key)) {
                    fieldDataMap.put("title", "应还罚息");
                } else if ("liquidatedDamages".equals(key)) {
                    fieldDataMap.put("title", "结清违约金");
                } else if ("otherFee".equals(key)) {
                    fieldDataMap.put("title", "应还其他费用");
                } else if ("totalAmount".equals(key)) {
                    fieldDataMap.put("title", "应还总额");
                }
                fieldDataMap.put("value", Constant.DECIMAL_FORMAT2.format(value));
                dataList.add(fieldDataMap);
            }

        }

        return dataList;
    }

    @Override
    public JsonResult<List<OpLog>> queryOperateLog(Long loanId) {
        List<OpLog> list = operateLogService.findLogsByLoanId(loanId);
        return JsonResult.buildSuccessResult("处理成功", list);
    }

    /**
     * 处理查询不到orderMapping的情况，可能是进件失败，也可能不存在该笔订单
     *
     * @param channelOrderNo
     * @param channelId
     * @return
     */
    private JsonResult<OrderInfoVo> buildNoOrderMapping(String channelOrderNo, Long channelId) {
        String logPre = "OrderServiceImpl.buildNoOrderMapping";
        // 查询进件流水信息  查询ka
        JsonResult<ApplyRequestHistory> orderMappingResult = kaService.queryLatestChannelRequestFlows(channelOrderNo, channelId);
        if (!orderMappingResult.isSuccess()) {
            log.error("{} 查询渠道进件流水信息异常 result={} channelOrderNo={},channelId={}", logPre, orderMappingResult, channelOrderNo, channelId);
            return JsonResult.buildErrorStateResult("查询渠道进件流水信息异常", ErrorCodeEnum.RETURN_ERROR.getCode());
        }

        ApplyRequestHistory applyRequestHistory = orderMappingResult.getData();
        // 不存在进件流水信息 返回查询不到该笔订单
        if (Objects.isNull(applyRequestHistory)) {
            return JsonResult.buildSuccessResult("订单不存在", null);
        }

        OrderInfo orderInfo = OrderInfo.builder()
                .channelOrderNo(channelOrderNo)
                .channelId(channelId)
                .status(applyRequestHistory.getIsSuccess() ? "进件成功" : "进件失败")
                .remark(applyRequestHistory.getDescriptionDetail())
                .build();

        OrderInfoVo orderInfoVo = OrderInfoVo.builder()
                .orderInfo(orderInfo)
                .build();
        log.info("{} 查询订单信息,返回信息 orderInfoVo={}", logPre, orderInfoVo);
        return JsonResult.buildSuccessResult("处理成功", orderInfoVo);
    }

    /**
     * 处理渠道订单表有数据的情形
     *
     * @param orderQuery
     * @param loanOrderMapping
     * @return
     */
    private JsonResult<OrderInfoVo> buildHasOrderMapping(OrderQueryParam orderQuery, LoanOrderMapping loanOrderMapping) {
        String logPre = "OrderServiceImpl.buildHasOrderMapping";

        // 查询订单状态  xyqb
        orderQuery.setApplyOrderNo(loanOrderMapping.getApplyNo());
        orderQuery.setLoanId(loanOrderMapping.getLoanId());
        orderQuery.setChannelId(loanOrderMapping.getRegisteredFrom());
        JsonResult<OrderStatus> orderStatusResult = xyqbService.orderStatusQuery(orderQuery);
        OrderStatus orderStatus;
        if (!orderStatusResult.isSuccess()) {
            log.error("{} 查询订单状态 orderStatus 失败 result={} orderQuery={}", logPre, orderStatusResult, orderQuery);
            return JsonResult.buildErrorStateResult(orderStatusResult.getMsg(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }

        orderStatus = orderStatusResult.getData();

        // 查询通知记录  查询ka
        JsonResult<CallbackRecordList> callbackListResult = kaService.querySendRecords(loanOrderMapping.getApplyNo(), loanOrderMapping.getRegisteredFrom());
        if (!callbackListResult.isSuccess()) {
            log.error("{} 查询发送记录失败  result={} orderQuery={}", logPre, callbackListResult, orderQuery);
            return JsonResult.buildErrorStateResult("查询通知记录失败", ErrorCodeEnum.RETURN_ERROR.getCode());
        }

        final String defaultStatusStr = "进件成功";
        String status = Objects.nonNull(orderStatus) ? orderStatus.getStatus().getValue() : defaultStatusStr;
        OrderInfo orderInfo = OrderInfo.builder()
                .channelOrderNo(loanOrderMapping.getChannelOrderNo())
                .applyNo(loanOrderMapping.getApplyNo())
                .channelId(loanOrderMapping.getRegisteredFrom())
                .loanId(loanOrderMapping.getLoanId())
                .status(status)
                .remark(status)
                .build();

        OrderInfoVo orderInfoVo = OrderInfoVo.builder()
                .orderInfo(orderInfo)
                .noticeList(callbackListResult.getData())
                .build();
        log.info("{} 查询订单信息,返回信息 orderInfoVo={}", logPre, orderInfoVo);
        return JsonResult.buildSuccessResult("处理成功", orderInfoVo);
    }


}
