package cn.quantgroup.customer.service.impl;

import cn.quantgroup.customer.entity.Authority;
import cn.quantgroup.customer.entity.RoleAuthority;
import cn.quantgroup.customer.entity.User;
import cn.quantgroup.customer.entity.UserRole;
import cn.quantgroup.customer.repo.AuthorityRepo;
import cn.quantgroup.customer.repo.RoleAuthorityRepo;
import cn.quantgroup.customer.repo.UserRepo;
import cn.quantgroup.customer.repo.UserRoleRepo;
import cn.quantgroup.customer.service.IUserService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.stream.Collectors;

@Slf4j
@Service("userService")
public class UserServiceImpl implements IUserService {

    private final UserRepo userRepo;
    private final UserRoleRepo userRoleRepo;
    private final AuthorityRepo authorityRepo;
    private final RoleAuthorityRepo roleAuthorityRepo;

    @Autowired
    public UserServiceImpl(UserRepo userRepo, UserRoleRepo userRoleRepo,
                           AuthorityRepo authorityRepo, RoleAuthorityRepo roleAuthorityRepo) {
        this.userRepo = userRepo;
        this.userRoleRepo = userRoleRepo;
        this.authorityRepo = authorityRepo;
        this.roleAuthorityRepo = roleAuthorityRepo;
    }

    @Override
    public UserDetails loadUserByUsername(String userName) throws UsernameNotFoundException {
        User user = userRepo.findByUsername(userName);
        if (user == null) {
            throw new UsernameNotFoundException("user: " + userName + " do not exist!");
        }

        List<UserRole> userRoles = findUserRoleByUserId(user.getId());
        if (CollectionUtils.isNotEmpty(userRoles)) {
            List<RoleAuthority> roleAuthorityList = findRoleAuthorityByRoleIds(
                    userRoles.stream().map(UserRole::getRoleId).collect(Collectors.toList()));

            if (CollectionUtils.isNotEmpty(roleAuthorityList)) {
                List<Authority> authorities = findAuthorityByAuthorityIds(
                        roleAuthorityList.stream().map(RoleAuthority::getAuthorityId).collect(Collectors.toList()));

                user.setAuthorities(authorities);
            }
        }

        return user;
    }

    @Override
    public List<UserRole> findUserRoleByUserId(Long userId) {
        return userRoleRepo.findByUserId(userId);
    }

    @Override
    public List<RoleAuthority> findRoleAuthorityByRoleIds(List<Long> roleIdList) {
        return roleAuthorityRepo.findByRoleIdIn(roleIdList);
    }

    @Override
    public List<Authority> findAuthorityByAuthorityIds(List<Long> authorityIdList) {
        return authorityRepo.findByIdIn(authorityIdList);
    }
}
