package cn.quantgroup.customer.service.impl;


import cn.quantgroup.customer.service.IFastDFSService;
import cn.quantgroup.customer.util.ProtoCommon;
import com.github.tobato.fastdfs.domain.StorePath;
import com.github.tobato.fastdfs.proto.storage.DownloadByteArray;
import com.github.tobato.fastdfs.service.FastFileStorageClient;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;

/**
 * @author xing.yuan
 */
@Service
@Slf4j
public class FastDFSServiceImpl implements IFastDFSService {

    @Resource
    private FastFileStorageClient storageClient;

    @Value("${fdfs.secret_key}")
    private String secretKey;

    @Value("${fdfs.domain}")
    private String fastDfsHttp;

    /**
     * 文件上传
     *
     * @param fileInput
     * @param fileSize
     * @param fileExtName 扩展名
     * @return
     */
    @Override
    public String uploadFile(InputStream fileInput, Long fileSize, String fileExtName) throws IOException {
        StorePath storePath = null;
        try {
            storePath = storageClient.uploadFile(fileInput, fileSize, fileExtName, null);
        } finally {
            if (fileInput != null) {
                try {
                    fileInput.close();
                } catch (IOException e) {
                }
            }
        }
        if (storePath != null && StringUtils.isNotBlank(storePath.getFullPath())) {
            return storePath.getFullPath();
        } else {
            throw new IOException("文件上传失败");
        }
    }

    /**
     * 文件上传
     *
     * @param bytes
     * @param fileExtName 扩展名
     * @return
     */
    @Override
    public String uploadFile(byte[] bytes, String fileExtName) throws IOException {
        return uploadFile(new ByteArrayInputStream(bytes), (long) bytes.length, fileExtName);
    }


    /**
     * 文件上传, 下载url 再传
     *
     * @param url
     * @param fileExtName
     * @return
     * @throws IOException
     */
    @Override
    public String uploadFile(String url, String fileExtName) throws IOException {
        InputStream in = null;
        long size = 0;
        HttpURLConnection conn = null;
        try {
            URL httpUrl = new URL(url);
            conn = (HttpURLConnection) httpUrl.openConnection();
            //设置超时间为3秒
            conn.setConnectTimeout(3 * 1000);
            conn.setRequestProperty("User-Agent", "Mozilla/4.0 (compatible; MSIE 5.0; Windows NT; DigExt)");
            in = conn.getInputStream();
            size = in.available();
        } catch (IOException e) {
            log.warn("下载文件异常,url={},", url, e);
            throw e;
        }
        log.info("准备上传文件; url={}, ", url);
        try {
            String path = uploadFile(in, size, fileExtName);
            return path;
        } finally {
            if (conn != null) {
                try {
                    conn.disconnect();
                } catch (Exception e) {
                }
            }
        }
    }


    /**
     * 上传文件 base64
     *
     * @param base64String
     * @param fileExtName
     * @return
     * @throws IOException
     */
    @Override
    public String uploadFileOfBase64String(String base64String, String fileExtName) throws IOException {
        byte[] bytes = Base64.decodeBase64(base64String);
        return uploadFile(bytes, fileExtName);
    }

    /**
     * 下载文件
     *
     * @param path
     * @return
     */
    @Override
    public byte[] downloadFile(String path) throws IOException {
        if (StringUtils.isBlank(path)) {
            return null;
        }
        String[] split = StringUtils.split(path, "/", 2);
        if (split.length < 2) {
            throw new IOException("路径不对");
        }
        byte[] bytes = storageClient.downloadFile(split[0], split[1], new DownloadByteArray() {
        });
        return bytes;
    }

    /**
     * 转成一个临时可用的url
     * @param path
     * @return
     * @throws IOException
     */
    @Override
    public String toUrl(String path) throws IOException {
        if (StringUtils.isBlank(path)) {
            return null;
        }
        String fileName = path;
        if (fileName.indexOf("/M") > 0) {
            fileName = fileName.substring(fileName.indexOf("/M") + 1);
        }
        int lts = (int) (System.currentTimeMillis() / 1000);
        // 初始化secret_key
        try {
            String token = ProtoCommon.getToken(fileName, lts, secretKey);
            return fastDfsHttp + "/" + path + "?token=" + token + "&ts=" + lts;
        } catch (Exception e) {
            log.warn("生成FastDFS下载链接失败：path:{},", path, e);
            throw new IOException("生成下载链接失败");
        }
    }

}
