package cn.quantgroup.customer.service.impl;

import cn.quantgroup.customer.enums.ErrorCodeEnum;
import cn.quantgroup.customer.enums.OrderStatusEnums;
import cn.quantgroup.customer.model.Tuple;
import cn.quantgroup.customer.model.kaordermapping.LoanOrderMapping;
import cn.quantgroup.customer.model.order.*;
import cn.quantgroup.customer.rest.param.ordermapping.OrderQueryParam;
import cn.quantgroup.customer.rest.vo.JsonResult;
import cn.quantgroup.customer.service.IKaService;
import cn.quantgroup.customer.service.IOrderService;
import cn.quantgroup.customer.service.IXyqbService;
import cn.quantgroup.customer.util.DateUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author Wang Xiangwei
 * @version 2020/3/10
 */
@Slf4j
@Service
public class OrderServiceImpl implements IOrderService {
    @Autowired
    private IKaService kaService;

    @Autowired
    private IXyqbService xyqbService;

    @Override
    public JsonResult<List<FlowNode>> findFlowChart(String applyOrderNo) {
        String logPre = "OrderServiceImpl.findFlowChart";
        log.info("{} 查询流程图 applyOrderNo={}", logPre, applyOrderNo);
        OrderQueryParam param = new OrderQueryParam();
        param.setApplyOrderNo(applyOrderNo);
        JsonResult<LoanOrderMapping> orderMappingResult = kaService.findOrderMapping(param);
        LoanOrderMapping head = null;
        if (orderMappingResult.isSuccess()) {
            head = orderMappingResult.getData();
            log.info("{},申请订单号查询orderMapping applyOrderNo={},data={}", logPre, applyOrderNo, head);
        }

        JsonResult<List<FlowNode>> flowChart = xyqbService.findFlowChart(applyOrderNo);
        if (!flowChart.isSuccess()) {
            log.error("{} 流程图查询失败 result={}", logPre, flowChart);
            return flowChart;
        }

        List<FlowNode> flowNodeList = flowChart.getData();
        if (Objects.nonNull(head)) {
            FlowNode flowNode = new FlowNode();
            flowNode.setName("渠道用户信息导入");
            flowNode.setStatus(1);
            flowNode.setNode("IMPORT_INFO");
            flowNode.setRemark("进件成功");
            flowNodeList.add(0, flowNode);

        }

        return JsonResult.buildSuccessResult("", flowNodeList);
    }


    @Override
    public JsonResult<LoanOrderDetail> getLoanOrderDetail(Long loanId) {
        String logPre = "OrderServiceImpl.getLoanOrderDetail";
        log.info("{} 提现详情 loanId={}", logPre, loanId);
        JsonResult<LoanOrder> loanOrder = xyqbService.findLoanOrder(loanId);

        if (!loanOrder.isSuccess()) {
            log.error("{} 提现订单查询失败 loanId={}, result={}", logPre, loanId, loanOrder);
            return JsonResult.buildErrorStateResult(loanOrder.getMsg(), loanOrder.getData());
        }

        LoanOrderDetail loanOrderDetail = new LoanOrderDetail();
        List<Repayment> repaymentList = new ArrayList<>();
        LoanOrder loanOrderData = loanOrder.getData();
        loanOrderDetail.setLoanOrder(loanOrderData);
        loanOrderDetail.setRepaymentList(repaymentList);

        if (Objects.isNull(loanOrderData) || !loanOrderData.getShowPlans()) {
            log.info("{} 不需查询还款计划", logPre);
            return JsonResult.buildSuccessResult("查询成功", loanOrderDetail);
        }

        //还款计划查询
        JsonResult<OrderRepayment> orderRepaymentJsonResult = xyqbService.repaymentPlanQuery(loanId);
        if (!orderRepaymentJsonResult.isSuccess()) {
            log.error("{} 还款计划查询失败 result={}", logPre, orderRepaymentJsonResult);
            return JsonResult.buildErrorStateResult(orderRepaymentJsonResult.getMsg(), orderRepaymentJsonResult.getData());
        }

        List<RepaymentPlanItem> repaymentPlans = orderRepaymentJsonResult.getData().getRepaymentPlans();
        repaymentPlans.forEach(e -> repaymentList.add(Repayment.valueOf(e)));

        return JsonResult.buildSuccessResult("查询成功", loanOrderDetail);
    }

    @Override
    public JsonResult queryOrderInfo(OrderQueryParam orderQuery) {
        String logPre = "OrderServiceImpl.queryOrderInfo";
        Tuple<Boolean, String> tuple = validate(orderQuery);
        if (!tuple.getKey()) {
            log.error("{},参数验证失败，{}", logPre, tuple.getValue());
            return JsonResult.buildErrorStateResult(tuple.getValue(), null);
        }

        // 查询订单信息  查询ka
        JsonResult<LoanOrderMapping> orderMappingResult = kaService.findOrderMapping(orderQuery);
        LoanOrderMapping loanOrderMapping;
        if (!orderMappingResult.isSuccess() || Objects.isNull(loanOrderMapping = orderMappingResult.getData())) {
            log.error("{} 查询订单信息orderMapping失败 result={} orderQuery={}", logPre, orderMappingResult, orderQuery);
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.RETURN_ERROR.getMessage(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }

        // 查询订单状态  xyqb
        orderQuery.setApplyOrderNo(loanOrderMapping.getApplyNo());
        orderQuery.setLoanId(loanOrderMapping.getLoanId());
        orderQuery.setChannelId(loanOrderMapping.getRegisteredFrom());
        JsonResult<OrderStatus> orderStatusResult = xyqbService.orderStatusQuery(orderQuery);
        OrderStatus orderStatus;
        if (!orderStatusResult.isSuccess() || Objects.isNull(orderStatus = orderStatusResult.getData())) {
            log.error("{} 查询订单状态 orderStatus 失败 result={} orderQuery={}", logPre, orderStatusResult, orderQuery);
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.RETURN_ERROR.getMessage(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }

        // 查询通知记录  查询ka
        JsonResult<List<CallbackRecord>> callbackListResult = queryOrderCallback(orderQuery);
        if (!callbackListResult.isSuccess()) {
            log.error("{} 查询发送记录失败  result={} orderQuery={}", logPre, callbackListResult, orderQuery);
            return JsonResult.buildErrorStateResult(ErrorCodeEnum.RETURN_ERROR.getMessage(), ErrorCodeEnum.RETURN_ERROR.getCode());
        }

        OrderInfo orderInfo = OrderInfo.builder()
                .channelOrderNo(loanOrderMapping.getChannelOrderNo())
                .applyNo(loanOrderMapping.getApplyNo())
                .channelId(loanOrderMapping.getRegisteredFrom())
                .loanId(loanOrderMapping.getLoanId())
                .status(orderStatus.getStatus().getValue())
                .build();

        OrderInfoVo orderInfoVo = OrderInfoVo.builder()
                .orderInfo(orderInfo)
                .noticeList(callbackListResult.getData())
                .build();
        log.info("{} 查询订单信息,返回信息 orderInfoVo={}", logPre, orderInfoVo);
        return JsonResult.buildSuccessResult("处理成功", orderInfoVo);
    }

    private Tuple<Boolean, String> validate(OrderQueryParam orderQuery) {
        Tuple<Boolean, String> result = new Tuple<>();
        result.setKey(false);
        int orderParam = 0;
        if (Objects.isNull(orderQuery)) {
            result.setValue("请求参数为空");
            return result;
        }
        if (StringUtils.isNotBlank(orderQuery.getChannelOrderNo())) {
            if (Objects.isNull(orderQuery.getChannelId())) {
                result.setValue("查询参数渠道订单号和渠道号要同时存在");
                return result;
            }
            orderParam++;
        }

        if (StringUtils.isNotBlank(orderQuery.getApplyOrderNo())) {
            orderParam++;
        }

        if (Objects.nonNull(orderQuery.getLoanId())) {
            orderParam++;
        }

        if (orderParam == 0) {
            result.setValue("请求参数为空");
            return result;
        } else if (orderParam > 1) {
            result.setValue("只能选择一个参数进行查询");
            return result;
        }

        result.setKey(true);
        return result;
    }

    /**
     * 处理Ka通知记录
     *
     * @param orderQuery
     * @return
     */
    private JsonResult<List<CallbackRecord>> queryOrderCallback(OrderQueryParam orderQuery) {
        String logPre = "OrderServiceImpl.queryOrderCallback 处理Ka通知记录 ";
        log.info("{} 请求信息 orderQuery :{}", logPre, orderQuery);
        String applyNo = orderQuery.getApplyOrderNo();
        Long channelId = orderQuery.getChannelId();
        //查询发送成功记录
        JsonResult<List<CallbackRecord>> sendSuccRecordResult = kaService.querySendRecords(applyNo, channelId, Boolean.TRUE);
        if (!sendSuccRecordResult.isSuccess()) {
            log.error("{} 查询发送成功记录失败 orderQuery :{},sendSuccRecordResult：{}", logPre, orderQuery, sendSuccRecordResult);
            return sendSuccRecordResult;
        }
        //查询发送失败记录
        JsonResult<List<CallbackRecord>> sendFailRecordResult = kaService.querySendRecords(applyNo, channelId, Boolean.FALSE);
        if (!sendFailRecordResult.isSuccess()) {
            log.error("{} 查询发送失败记录失败 orderQuery :{},sendFailRecordResult：{}", logPre, orderQuery, sendFailRecordResult);
            return sendSuccRecordResult;
        }
        //聚合信息
        List<CallbackRecord> callbackRecords = doMergeLists(sendSuccRecordResult.getData(), sendFailRecordResult.getData());
        return JsonResult.buildSuccessResult("查询成功", callbackRecords);
    }

    private List<CallbackRecord> doMergeLists(List<CallbackRecord> data, List<CallbackRecord> data1) {
        data1.forEach(callbackRecord -> {
            if (!data.contains(callbackRecord)) {
                data.add(callbackRecord);
            }
        });
        List<CallbackRecord> callbackRecords1 = data.stream().sorted(Comparator.comparing(CallbackRecord::getDateTime)).collect(Collectors.toList());
        callbackRecords1.forEach(callbackRecord ->
                {
                    callbackRecord.setCreateAt(DateUtil.format(DateUtil.getByLongTimes(callbackRecord.getDateTime()), DateUtil.DATE_FORMAT_1));
                    callbackRecord.setDateTime(null);
                }
        );

        return callbackRecords1;
    }
}
