package cn.quantgroup.cashloanflowboss.core.web;


import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.RandomStringUtils;
import org.slf4j.MDC;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Objects;

/**
 * web 请求拦截器,跟踪KEY
 * @author Jie.Feng
 * @date 2017/10/30
 */
@Slf4j
public class ResponseMetricsInterceptor extends HandlerInterceptorAdapter {

    public static final String TRACING_ID = "TRACING_ID";
    private static final String RESPONSE_TIME = "RESPONSE_TIME";

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {

        String tracingId = RandomStringUtils.randomAlphanumeric(12);
        request.setAttribute(RESPONSE_TIME, System.currentTimeMillis());
        MDC.put(TRACING_ID, tracingId);

        String path = getPath(request);
        log.info("service.tracing.start.请求开始 url={}", path);
        return super.preHandle(request, response, handler);
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) throws Exception {
        try {
            log.info("service.tracing.end.请求结束 url={}, 消耗时间：{}ms", getPath(request),
                    System.currentTimeMillis()-Long.valueOf(Objects.toString(request.getAttribute(RESPONSE_TIME),"0")));
        } catch (NumberFormatException e) {
        }finally {
            MDC.remove(TRACING_ID);
        }
        super.afterCompletion(request, response, handler, ex);
    }

    private String getPath(HttpServletRequest request){
        String path = request.getServletPath();
        if(path.equalsIgnoreCase("/error")){
            path = "errorurl:" + Objects.toString(request.getAttribute("javax.servlet.error.request_uri"));
        }
        return path;
    }

}
