package cn.quantgroup.cashloanflowboss.api.user.service;

import cn.quantgroup.cashloanflowboss.api.login.model.Principal;
import cn.quantgroup.cashloanflowboss.api.role.entity.Role;
import cn.quantgroup.cashloanflowboss.api.role.repository.RoleRepository;
import cn.quantgroup.cashloanflowboss.api.user.dictionary.UserRank;
import cn.quantgroup.cashloanflowboss.api.user.dictionary.UserStatus;
import cn.quantgroup.cashloanflowboss.api.user.entity.User;
import cn.quantgroup.cashloanflowboss.api.user.model.*;
import cn.quantgroup.cashloanflowboss.api.user.repository.UserRepository;
import cn.quantgroup.cashloanflowboss.core.Application;
import cn.quantgroup.cashloanflowboss.core.asserts.Assert;
import cn.quantgroup.cashloanflowboss.core.base.Result;
import cn.quantgroup.cashloanflowboss.core.base.Tuple;
import cn.quantgroup.cashloanflowboss.core.dictionary.ApplicationStatus;
import cn.quantgroup.cashloanflowboss.spi.clf.entity.ClfChannelConfiguration;
import cn.quantgroup.cashloanflowboss.spi.clf.repository.ClfChannelConfigurationRepository;
import cn.quantgroup.cashloanflowboss.spi.user.service.XyqbUserService;
import cn.quantgroup.cashloanflowboss.spi.xyqb.service.XYQBCenterService;
import cn.quantgroup.cashloanflowboss.utils.MD5Tools;
import cn.quantgroup.user.retbean.XUser;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.criteria.Predicate;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Created by WeiWei on 2019/7/22.
 */
@Slf4j
@Service
public class UserServiceImpl implements UserService {

    @Autowired
    private UserRepository userRepository;
    @Autowired
    private XYQBCenterService xyqbCenterService;
    @Autowired
    private XyqbUserService xyqbUserService;

    @Autowired
    private RoleRepository roleRepository;

    @Autowired
    private ClfChannelConfigurationRepository clfChannelConfigurationRepository;


    /**
     * 创建用户
     *
     * @param username 用户名
     * @param password 登录密码（明文）
     * @return
     */
    @Override
    public boolean createUser(String username, String password) {

        User user = new User();
        user.setUsername(username);
        user.setPassword(MD5Tools.md5(password));
        user.setStatus(UserStatus.ENABLED);

        return Objects.nonNull(this.userRepository.save(user));

    }

    /**
     * 获取用户
     *
     * @param username 用户名
     * @return
     */
    @Override
    public User getUser(String username) {
        return this.userRepository.getUserByUsername(username);
    }

    /**
     * 获取用户列表（分页）
     *
     * @param queryUserListModel
     * @return
     */
    @Override
    public Page<User> getUsers(QueryUserListModel queryUserListModel) {

        Page<User> page = this.userRepository.findAll((root, criteriaQuery, criteriaBuilder) -> {
            List<Predicate> predicates = new ArrayList<>();

            if (StringUtils.isNotEmpty(queryUserListModel.getNickname())) {
                predicates.add(criteriaBuilder.equal(root.get("nickname"), queryUserListModel.getNickname()));
            }
            criteriaQuery.where(criteriaBuilder.and(predicates.toArray(new Predicate[predicates.size()])));
            // 指定排序
            criteriaQuery.orderBy(criteriaBuilder.desc(root.get("id")));

            return criteriaQuery.getRestriction();

        }, new PageRequest(queryUserListModel.getPageNumber(), queryUserListModel.getPageSize()));

        return page;
    }

    /**
     * 启用用户
     *
     * @param id 用户ID
     * @return
     */
    @Override
    public boolean enableUser(Long id) {

        User user = this.userRepository.findOne(id);

        // 检查用户是否存在
        Assert.isNull(user, ApplicationStatus.INVALID_USER);

        user.setStatus(UserStatus.ENABLED);

        return Objects.nonNull(this.userRepository.save(user));

    }

    /**
     * 禁用用户
     *
     * @param id 用户ID
     * @return
     */
    @Override
    public boolean disableUser(Long id) {

        User user = this.userRepository.findOne(id);

        // 检查用户是否存在
        Assert.isNull(user, ApplicationStatus.INVALID_USER);

        user.setStatus(UserStatus.DISABLED);

        return Objects.nonNull(this.userRepository.save(user));

    }

    /**
     * 更新用户
     *
     * @param user
     * @return
     */
    @Override
    public User updateUser(User user) {
        return this.userRepository.save(user);
    }

    /**
     * 移除用户
     *
     * @param id 用户ID
     * @return
     */
    @Override
    public Boolean removeUser(Long id) {

        try {
            this.userRepository.delete(id);
            return true;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }

        return false;

    }

    @Override
    public UserDetailInfo getUserDetailInfo() {
        Principal principal = Application.getPrincipal();
        Assert.isNull(principal, ApplicationStatus.INVALID_USER);

        return UserDetailInfo.valueOf(principal);
    }


    @Override
    public User saveUserInfo(UserInfoModel userInfoModel) {
        User user1 = new User();
        BeanUtils.copyProperties(userInfoModel, user1);
        if (StringUtils.isEmpty(userInfoModel.getPassword())) {
            user1.setPassword(MD5Tools.md5(userInfoModel.getUsername()));
        } else {
            user1.setPassword(MD5Tools.md5(userInfoModel.getPassword()));
        }
        user1.setRoles(userInfoModel.getRoles());
        return userRepository.save(user1);
    }

    @Async("commonAsyncExecutor")
    //@OperationAnno(channelNo = "#this[0]", opt = OptEnumName.USER_ORDER_CLEAN, succSPEL = "#this.key", optDetailSPEL = "#this.value")
    @Override
    public Tuple<Boolean, String> cleanUserActiveOrder(String mobile) {
        XUser user = xyqbUserService.findUserByPhoneNo(mobile);
        if (user == null) {
            log.info("cleanUserOrder,清除用户活跃订单失败，未找到用户 phoneNo={}", mobile);
            return new Tuple<>(false, "清除用户活跃订单失败，未找到用户");
        }
        return xyqbCenterService.cleanUserActiveOrder(user.getId());
    }


    @Override
    @Transactional
    public Result<Boolean> addUser(UserModelVo userModelVo) {
        final String LOG_PRE = "UserServiceImpl.addUser";
        log.info("{} 添加用户 rank={}", LOG_PRE, userModelVo);

        String username = userModelVo.getUsername();
        User exist = this.getUser(username);
        if (exist != null) {
            log.error("{},账户已存在 userName={}", LOG_PRE, username);
            return Result.buildFail(username + " 账户已存在");
        }

        Long roleId = userModelVo.getRoleId();
        Role role = null;
        if (roleId != null) {
            if (!roleRepository.exists(roleId)) {
                log.error("{},不存在对应的角色 roleId={}", LOG_PRE, roleId);
                return Result.buildFail("不存在对应的角色");
            }
            role = roleRepository.getOne(roleId);
        }

        User user = User.valueOf(userModelVo, role);
        //渠道检查
        Long channelId = user.getChannelId();
        if (channelId != null) {
            ClfChannelConfiguration configuration = clfChannelConfigurationRepository.findByRegisteredFrom(channelId);
            if (configuration == null) {
                log.error("{}，渠道不存在 channelId={}", LOG_PRE, channelId);
                return Result.buildFail("channelId=" + channelId + "的渠道不存在");
            }
            user.setChannelName(configuration.getChannelName());
        }
        userRepository.save(user);
        return Result.buildSuccess(true);
    }

    @Override
    @Transactional
    public Result<Boolean> modifyUser(UserModelVo userModelVo) {
        final String LOG_PRE = "UserServiceImpl.modifyUser";
        Long id = userModelVo.getId();
        if (id == null) {
            log.error("{} 缺少主键id, userModelVo={}", LOG_PRE, userModelVo);
            return Result.buildFail("缺少用户Id");
        }

        if (!userRepository.exists(id)) {
            log.error("{}不存在相应的用户，不能更新 userModelVo={}", LOG_PRE, userModelVo);
            return Result.buildFail("不存在相应的用户，不能更新");
        }
        User exist = userRepository.getOne(id);
        // 修改密码是单独的接口
        if (StringUtils.isNotEmpty(userModelVo.getPassword())) {
            exist.setPassword(userModelVo.getPassword());
        }
        if (StringUtils.isNotEmpty(userModelVo.getNickname())) {
            exist.setNickname(userModelVo.getNickname());
        }
        String existUsername = exist.getUsername();
        String username = userModelVo.getUsername();
        if (!existUsername.equals(username)) {
            log.error("{} 账号不能修改，userModelVo={}", LOG_PRE, userModelVo);
            return Result.buildFail("账号不能修改");
        }

        Long roleId = userModelVo.getRoleId();
        if (roleId != null) {
            if (!roleRepository.exists(roleId)) {
                log.error("{},不存在对应的角色 roleId={}", LOG_PRE, roleId);
                return Result.buildFail("不存在对应的角色");
            }
            Role role = roleRepository.getOne(roleId);
            List<Long> idList = exist.getRoles().stream().map(Role::getId).collect(Collectors.toList());
            if (!idList.contains(roleId)) {
                exist.getRoles().add(role);
            }
            if (roleId == 1L) {
                // 将普通用户设置成超级管理，需要级联设置用户级别
                exist.setRank(UserRank.SUPER_ADMINISTRATOR);
            } else {
                if (exist.getRank().equals(UserRank.SUPER_ADMINISTRATOR)) {
                    // 如果之前是超管 改成普通操作员，用户级别做相应调整
                    exist.setRank(UserRank.OPERATOR);
                }
            }

        }

        Long channelId = exist.getChannelId();
        // 渠道id检查
        if (channelId != null) {
            ClfChannelConfiguration configuration = clfChannelConfigurationRepository.findByRegisteredFrom(channelId);
            if (configuration == null) {
                log.error("{}，渠道不存在 channelId={}", LOG_PRE, channelId);
                return Result.buildFail("channelId=" + channelId + "的渠道不存在");
            }
            exist.setChannelId(channelId);
            exist.setChannelName(configuration.getChannelName());
        }

        exist.setUpdateTime(new Date());
        userRepository.save(exist);
        return Result.buildSuccess(true);
    }

    @Override
    public Result<Page<User>> getUserList(QueryUserListModel queryUserListModel) {
        final String LOG_PRE = "UserServiceImpl.getUserList";
        log.info("{},查询用户列表，queryUserListModel={}", LOG_PRE, queryUserListModel);
        String nickname = queryUserListModel.getNickname();
        Long roleId = queryUserListModel.getRoleId();

        PageRequest pageRequest = new PageRequest(queryUserListModel.getPageNumber(), queryUserListModel.getPageSize(), new Sort(Sort.Direction.DESC, "id"));

        Page<User> userPage = userRepository.getUserByNicknameAndRoleId(nickname == null ? "" : nickname, roleId == null ? "" : roleId.toString(),
                pageRequest);

        return Result.buildSuccess(userPage);
    }

    @Override
    public Result<Boolean> updatePassword(UpdatePasswordParam updatePasswordParam) {
        String logPre = "UserServiceImpl.updatePassword";
        log.info("{},修改密码，updatePasswordParam={}", logPre, updatePasswordParam);
        User user = userRepository.getUserByUsername(updatePasswordParam.getUsername());
        if (Objects.isNull(user)) {
            log.info("{} 不存在 {} 用户", logPre, updatePasswordParam.getUsername());
            return Result.buildFail("用户不存在，不能修改密码");
        }


        String newPassword = MD5Tools.md5(updatePasswordParam.getNewPassword());

        String oldPassword = user.getPassword();
        if(newPassword.equals(oldPassword)){
            return Result.buildFail("新密码和旧密码相同");
        }
        user.setPassword(newPassword);
        userRepository.save(user);
        return Result.buildSuccess(true);
    }
}
