package cn.quantgroup.cashloanflowboss.core.configuration.data;

import cn.quantgroup.cashloanflowboss.core.persistence.CashLoanFlowBossPubConfDataSource;
import cn.quantgroup.cashloanflowboss.spi.apollo.service.ApolloSettingService;
import com.alibaba.druid.pool.DruidDataSource;
import com.alibaba.druid.spring.boot.autoconfigure.DruidDataSourceBuilder;
import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.orm.jpa.EntityManagerFactoryBuilder;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.jpa.repository.config.EnableJpaRepositories;
import org.springframework.orm.jpa.JpaTransactionManager;
import org.springframework.orm.jpa.LocalContainerEntityManagerFactoryBean;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.annotation.EnableTransactionManagement;

import javax.persistence.EntityManagerFactory;
import javax.sql.DataSource;
import java.util.Map;

@Configuration
@EnableTransactionManagement
@EnableJpaRepositories(basePackages = {"cn.quantgroup"}, entityManagerFactoryRef = "bossPubEntityManager", transactionManagerRef = "bossPubTransactionManager", includeFilters = @ComponentScan.Filter(CashLoanFlowBossPubConfDataSource.class))
public class CashLoanFlowBossPubConfDataSourceConfiguration {

    @Autowired
    private ApolloSettingService apolloSettingService;

    private String uniquename = "bossDSPub";

    @Value("${data.boss.url}")
    private String jdbcUrl;

    @Value("${data.boss.password}")
    private String password;

    @Value("${data.boss.username}")
    private String user;

    @Value("${spring.datasource.driverClassName}")
    private String driverClassName;


    @Value("${boss.user.setting.env}")
    private String bossUserSettingEnv;

    @Value("${data.boss.pool-size}")
    private Integer poolSize;

    @Value("${debug.model}")
    private Boolean debugModel;


    @Bean(name = "bossPubDataSource")
    public DataSource createDataSource() {
        DruidDataSource source = DruidDataSourceBuilder.create().build();
        source.setDriverClassName(driverClassName);
        // 测试环境固定ka1 环境，线上环境跟去apollo配置获取
        if (debugModel) {
            Map<String, Object> paramsFromEOS = apolloSettingService.getParamsFromEOS(bossUserSettingEnv, "DB_SERVICE_HOST", "DB_SERVICE_PORT");
            if (MapUtils.isEmpty(paramsFromEOS)) {
                throw new RuntimeException("获取boss固定配置数据库失败,bossUserSettingEnv=" + bossUserSettingEnv);
            }
            String jdbcUrlFormat = "jdbc:mysql://%s:%s/cash_loan_flow_boss?useUnicode=true&characterEncoding=UTF8";
            final String testJdbcUrl = String.format(jdbcUrlFormat, paramsFromEOS.get("DB_SERVICE_HOST").toString(), paramsFromEOS.get("DB_SERVICE_PORT").toString());
            source.setUrl(testJdbcUrl);
        } else {
            source.setUrl(jdbcUrl);
        }
        source.setUsername(user);
        source.setPassword(password);

        source.setMaxActive(200);
        source.setMinIdle(10);
        source.setInitialSize(10);
        source.setMaxWait(500000);
        source.setTimeBetweenEvictionRunsMillis(60000);
        source.setMinEvictableIdleTimeMillis(300000);
        source.setValidationQuery("SELECT 'x'");
        source.setTestWhileIdle(true);
        source.setTestOnBorrow(false);
        source.setTestOnReturn(false);
        source.setPoolPreparedStatements(Boolean.FALSE);
        return source;
    }


    @Bean(name = "bossPubEntityManager")
    public LocalContainerEntityManagerFactoryBean entityManager(EntityManagerFactoryBuilder builder) {

        return builder.dataSource(createDataSource())
                .packages("cn.quantgroup.cashloanflowboss.api")
                .persistenceUnit(uniquename)
                .build();
    }


    @Bean(name = "bossPubTransactionManager")
    public PlatformTransactionManager transactionManager(@Qualifier("bossPubEntityManager") EntityManagerFactory entityManagerFactory) {
        return new JpaTransactionManager(entityManagerFactory);
    }

}
