package cn.quantgroup.cashloanflowboss.core.configuration;

import cn.quantgroup.cashloanflowboss.api.login.model.Principal;
import cn.quantgroup.cashloanflowboss.api.login.service.LoginService;
import cn.quantgroup.cashloanflowboss.api.login.service.LoginServiceImpl;
import cn.quantgroup.cashloanflowboss.api.role.entity.Role;
import cn.quantgroup.cashloanflowboss.component.security.Authority;
import cn.quantgroup.cashloanflowboss.component.security.SecurityHandler;
import cn.quantgroup.cashloanflowboss.core.Application;
import cn.quantgroup.cashloanflowboss.core.asserts.Assert;
import cn.quantgroup.cashloanflowboss.core.base.Tuple;
import cn.quantgroup.cashloanflowboss.core.dictionary.ApplicationStatus;
import cn.quantgroup.cashloanflowboss.core.exception.ApplicationException;
import org.aopalliance.intercept.MethodInvocation;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;

import javax.servlet.http.HttpServletRequest;
import java.util.ArrayList;
import java.util.List;

/**
 * Created by WeiWei on 2019/7/26.
 */
@Configuration
public class ApplicationSecurityHandler implements SecurityHandler {

    @Autowired
    private LoginService loginService;

    @Override
    public Tuple<Boolean, ApplicationStatus> doAuthentication(MethodInvocation invocation, String authorityId, Authority[] authority) {

        Principal principal = Application.getPrincipal();

        // 检查是否已登录
        if (principal == null) {
            return new Tuple<>(Boolean.FALSE, ApplicationStatus.AUTHENTICATION_LOGIN);
        }

        // 是否 被挤下线
        Boolean isLogin = Assert.isLastLogin(loginService.getConcurrentHashMapLoginInfo());
        if (BooleanUtils.isTrue(isLogin)) {
            // 退出登陆
            loginService.logout();
            // 返回 信息
            return new Tuple<>(Boolean.FALSE, ApplicationStatus.AUTHENTICATION_LOGIN_CROWD_OUT);
        }


        // 如果是超级管理员跳过权限验证
        boolean hasPrivilege = principal.isSuperAdministrator() || principal.getRoles().stream().anyMatch(role -> {
            List<Role> roleList = getRoleAndParent(role);
            if (CollectionUtils.isEmpty(roleList)) {
                return false;
            }
            return roleList.stream().anyMatch(roleItem -> this.checkAuthority(authorityId, roleItem));
        });
        return new Tuple<>(hasPrivilege, hasPrivilege ? null : ApplicationStatus.INVALID_AUTHORITY);

    }

    /**
     * 遍历出用户的父级 role
     * @param role
     * @return
     */
    private List<Role> getRoleAndParent(Role role) {
        if (role == null) {
            return null;
        }

        List<Role> list = new ArrayList<>();
        list.add(role);

        while (role.getParent() != null) {
            role = role.getParent();
            list.add(role);
        }
        return list;
    }

    /**
     * 检查权限
     *
     * @param authorityId
     * @param role
     * @return
     */
    private boolean checkAuthority(String authorityId, Role role) {
        if (role == null) {
            return false;
        }
        return CollectionUtils.isNotEmpty(role.getPermissions()) && role.getPermissions().parallelStream().anyMatch(permission -> permission.getName().equals(authorityId));
    }

}
