package cn.quantgroup.cashloanflowboss.api.customer.service.impl;

import cn.quantgroup.cashloanflowboss.api.customer.entity.CustomerInfo;
import cn.quantgroup.cashloanflowboss.api.customer.service.ICustomerService;
import cn.quantgroup.cashloanflowboss.component.route.ServiceRoute;
import cn.quantgroup.cashloanflowboss.core.base.ServiceResult;
import cn.quantgroup.cashloanflowboss.spi.clf.entity.ClfOrderMapping;
import cn.quantgroup.cashloanflowboss.spi.clf.repository.ClfOrderMappingRepository;
import cn.quantgroup.cashloanflowboss.spi.spiderCenter.SpiderCenter;
import cn.quantgroup.cashloanflowboss.spi.user.service.UserSysService;
import cn.quantgroup.user.retbean.*;
import cn.quantgroup.user.vo.UserSysResult;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @author Wang Xiangwei
 * @version 2020/7/8
 */
@Service
@Slf4j
public class CustomerService implements ICustomerService {
    @Autowired
    private UserSysService userSysService;

    @Autowired
    private ServiceRoute serviceRoute;

    @Autowired
    private ClfOrderMappingRepository clfOrderMappingRepository;

    @Autowired
    private SpiderCenter spiderCenter;

    @Override
    public CustomerInfo getCustomerInfo(String phoneNo, String channelOrderNo) {
        String logPre = "CustomerService.getCustomerInfo";
        log.info("{} 查询客户信息 phoneNo={},channelOrderNo={}", logPre, phoneNo, channelOrderNo);

        if (StringUtils.isAllBlank(phoneNo, channelOrderNo)) {
            log.error("{} 查询客户信息失败 参数为空phoneNo={},channelOrderNo={}", logPre, phoneNo, channelOrderNo);
            return null;
        }
        Long userId = getUserId(phoneNo, channelOrderNo);
        if (userId == null) {
            return null;
        }

        UserSysResult<XUser> xUserResult = userSysService.getService().findUserByUserId(userId);
        if (!xUserResult.isSuccess()) {
            log.error("{} 查询用户信息失败 userId={} userResult={}", logPre, userId, xUserResult);
            return null;
        }
        CustomerInfo info = new CustomerInfo();

        populateUser(info, xUserResult.getData());

        UserSysResult<XUserDetail> userDetailResult = userSysService.getService().findUserDetailByUserId(userId);
        if (!userDetailResult.isSuccess()) {
            log.error("{} 查询用户详细信息失败 userId={} userDetailResult={}", logPre, userId, userDetailResult);
            return null;
        }

        populateDetail(info, userDetailResult.getData());

        UserSysResult<XUserExtInfo> userExtInfoResult = userSysService.getService().findUserExtInfoByUserId(userId);
        if (!userExtInfoResult.isSuccess()) {
            log.error("{} 查询用户扩展信息失败 userId={} userExtInfoResult={}", logPre, userId, userExtInfoResult);
            return null;
        }

        populateExtInfo(info, userExtInfoResult.getData());

        UserSysResult<List<XContact>> contactsResult = userSysService.getService().findContactsByUserId(userId);

        if (!contactsResult.isSuccess()) {
            log.error("{} 查询用户紧急联系人失败 userId={} contactsResult={}", logPre, userId, contactsResult);
            return null;
        }

        populateContact(info, contactsResult.getData());

        UserSysResult<XAddress> addressResult = userSysService.getService().findAddressByUserId(userId);

        if (!addressResult.isSuccess()) {
            log.error("{}查询用户地址失败 userId={} addressResult={}", logPre, userId, addressResult);
            return null;
        }

        populateAddress(info, addressResult.getData());

//        Map<String, Object> param = getParam(channelOrderNo);
//
//        ServiceResult<List<CustomerInfo.RiskInfo>> riskInfoData = spiderCenter.getRiskInfoData(param);
//        if (!riskInfoData.isSuccess()) {
//            log.error("{}查询风控信息失败 channelOrderNo={} riskInfoData={}", logPre, channelOrderNo, riskInfoData);
//            return null;
//        }
//        info.setRiskInfoList(riskInfoData.getData());
        return info;
    }

    private Map<String, Object> getParam(String channelOrderNo) {
        String logPre = "CustomerService.getParam";
        //优先使用渠道订单号查询userId
        if (StringUtils.isBlank(channelOrderNo)) {
            log.error("{} 查询客户信息失败 参数为空 channelOrderNo={}", logPre, channelOrderNo);
            return null;
        }
        ClfOrderMapping orderMapping = clfOrderMappingRepository.findByChannelOrderNoLastOne(channelOrderNo);
        if (orderMapping == null) {
            log.error("{} 查询客户信息失败 未查到对应的order_mapping channelOrderNo={}", logPre, channelOrderNo);
            return null;
        }

        Map<String,Object> param = new HashMap<>();
        List<String> middleStageChannel = serviceRoute.getMiddleStageChannel();
        if(middleStageChannel.contains(orderMapping.getRegisteredFrom().toString())){
            param.put("type",2);
        }else {
            param.put("type",1);
        }
        Long qgUserId = orderMapping.getQgUserId();
        UserSysResult<XUser> userByUserId = userSysService.getService().findUserByUserId(qgUserId);
        if (!userByUserId.isSuccess()) {
            log.error("{} 查询用户信息失败 userId={} userResult={}", logPre, qgUserId, userByUserId);
            return null;
        }
        param.put("uuid",userByUserId.getData().getUuid());
        param.put("channelId",userByUserId.getData().getRegisteredFrom());
        param.put("applyNo",orderMapping.getApplyNo());

        return param;
    }

    private void populateAddress(CustomerInfo info, XAddress xAddress) {
        String logStr = "CustomerService.populateAddress";
        log.info("{} xAddress 填充客户信息 xAddress={}", logStr, xAddress);
        if (Objects.isNull(xAddress)) {
            log.error("{} xAddress填充客户地址失败 xAddress为空", logStr);
            return;
        }
        info.setAddress(xAddress.getAddress());
    }


    private void populateContact(CustomerInfo info, List<XContact> contacts) {
        String logStr = "CustomerService.populateContact";
        log.info("{} contacts填充客户信息 contacts={}", logStr, contacts);
        if (CollectionUtils.isEmpty(contacts)) {
            log.error("{} contacts填充客户紧急联系人失败 contacts为空", logStr);
            return;
        }
        if (contacts.size() != 2) {
            log.error("{} 紧急联系人个数错误 contacts={}", logStr, contacts);
            return;
        }
        info.setContactNameA(contacts.get(0).getName());
        info.setContactPhoneA(contacts.get(0).getPhoneNo());
        info.setContactRelationA(contacts.get(0).getRelation().name());

        info.setContactNameB(contacts.get(1).getName());
        info.setContactPhoneB(contacts.get(1).getPhoneNo());
        info.setContactRelationB(contacts.get(1).getRelation().name());

    }

    private void populateExtInfo(CustomerInfo info, XUserExtInfo xUserExtInfo) {
        String logStr = "CustomerService.populateExtInfo";
        log.info("{} xUserExtInfo填充客户信息 xUserExtInfo={}", logStr, xUserExtInfo);
        if (Objects.isNull(xUserExtInfo)) {
            log.error("{} xUserExtInfo填充客户扩展信息失败 xUserExtInfo为空", logStr);
            return;
        }

        info.setIncome(xUserExtInfo.getIncomeEnum().name());
        info.setIncomeRange(xUserExtInfo.getIncomeRangeEnum().name());
        info.setOccupation(xUserExtInfo.getOccupationEnum().name());
        info.setEducation(xUserExtInfo.getEducationEnum().name());
        info.setHasCar(xUserExtInfo.getHasCar());
        info.setHasSocialSecurity(xUserExtInfo.getHasSocialSecurity());
        info.setHasHouse(xUserExtInfo.getHasHouse());
        info.setHasCreditCard(xUserExtInfo.getHasCar());
        info.setMarryStatus(xUserExtInfo.getMarryStatus().name());
    }

    private void populateDetail(CustomerInfo info, XUserDetail xUserDetail) {
        String logStr = "CustomerService.populateDetail";
        log.info("{} xUser填充客户信息 xUser={}", logStr, xUserDetail);
        if (Objects.isNull(xUserDetail)) {
            log.error("{} xUserDetail填充客户信息失败 xUserDetail为空", logStr);
            return;
        }
        info.setName(xUserDetail.getName());
        info.setIdNo(xUserDetail.getIdNo());
        info.setEmail(xUserDetail.getEmail());

    }

    private void populateUser(CustomerInfo info, XUser xUser) {
        String logStr = "CustomerService.populateUser";
        log.info("{} xUser填充客户信息 xUser={}", logStr, xUser);
        if (Objects.isNull(xUser)) {
            log.error("{} xUser填充客户信息失败 xUser为空", logStr);
            return;
        }
        info.setUserId(xUser.getId());
        info.setPhoneNo(xUser.getPhoneNo());
        info.setUuid(xUser.getUuid());
        info.setEnable(xUser.getEnable());

    }

    private Long getUserId(String phoneNo, String channelOrderNo) {
        String logPre = "CustomerService.getUserId";
        //优先使用渠道订单号查询userId
        if (StringUtils.isNotBlank(channelOrderNo)) {
            ClfOrderMapping orderMapping = clfOrderMappingRepository.findByChannelOrderNoLastOne(channelOrderNo);
            if (orderMapping == null) {
                log.error("{} 查询客户信息失败 未查到对应的order_mapping channelOrderNo={}", logPre, channelOrderNo);
                return null;
            }
            return orderMapping.getQgUserId();
        } else {
            UserSysResult<XUser> user =
                    userSysService.getService().findUserByPhoneNo(phoneNo);
            if (!user.isSuccess()) {
                log.error("{} 查询客户信息失败 phoneNo={},result={}", logPre, phoneNo, user);
                return null;
            }
            return user.getData().getId();
        }

    }
}
