package cn.quantgroup.cashloanflowboss.api.order.controller;
import cn.quantgroup.cashloanflowboss.api.order.model.ApproveVo;
import cn.quantgroup.cashloanflowboss.api.order.model.LendingFormModel;
import cn.quantgroup.cashloanflowboss.api.order.service.OrderService;
import cn.quantgroup.cashloanflowboss.component.security.annotiation.Security;
import cn.quantgroup.cashloanflowboss.core.Application;
import cn.quantgroup.cashloanflowboss.core.annotation.channelrole.ChannelIdInit;
import cn.quantgroup.cashloanflowboss.core.annotation.channelrole.CheckChannelRole;
import cn.quantgroup.cashloanflowboss.core.annotation.channelrole.CheckChannelRoleByChannelOrderNumber;
import cn.quantgroup.cashloanflowboss.core.base.Result;
import cn.quantgroup.cashloanflowboss.core.base.Tuple;
import cn.quantgroup.cashloanflowboss.core.tuple.ThreeTuple;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;

/**
 * function:
 * date: 2019/8/8
 *
 * @author: suntao
 */

@RestController
@RequestMapping("/order")
public class OrderController {


    @Autowired
    private OrderService orderService;


    /**
     * 订单查询列表接口
     *
     * @param channelId
     * @param channelOrderNumber
     * @param pageNumber
     * @param pageSize
     * @return 返回中包含当前订单可操作的 button
     */
    @Security(authorityId = "Order.getOrderList")
    @ChannelIdInit
    @CheckChannelRole
    @GetMapping("/list")
    public Result orderList(Long channelId, String channelOrderNumber, String applyNo, Long loanId, Integer pageNumber, Integer pageSize) {
        if (pageNumber == null || pageSize == null) {
            return Result.buildFial("page信息不对");
        }
        return Result.buildSuccess(orderService.orderList(channelId, channelOrderNumber, applyNo, loanId, pageNumber, pageSize));
    }


    /**
     * 审批接口
     *
     * @param approveVo
     * @return
     */
    @Security(authorityId = "Order.approveOpt")
    @ChannelIdInit
    @PostMapping("/approve")
    public Result approveOpt(@RequestBody @Valid ApproveVo approveVo) {

        if (Application.isDebug()) {
            Tuple<Boolean, String> approveOptBooleanAndFundSettingBooleanTuple = orderService.approveOpt(approveVo);
            Boolean approveResult = approveOptBooleanAndFundSettingBooleanTuple.getKey();
            if (approveResult) {
                return Result.buildSuccess(approveResult, approveOptBooleanAndFundSettingBooleanTuple.getValue());
            } else {
                return Result.buildSuccess(false,"审批操作失败");
            }
        } else {
            return Result.buildSuccess(false,"无此操作");
        }
    }

    /**
     * 放款
     *
     * @return
     */
    @Security(authorityId = "Order.lending")
    @PostMapping("/lending")
    public Result<Boolean> lending(@RequestBody @Valid LendingFormModel lendingFormModel) {
        if (Application.isDebug()) {
            Tuple<Boolean, String> lendingResult = this.orderService.lending(lendingFormModel);
            return Result.buildSuccess(lendingResult.getKey(), lendingResult.getKey() ? "放款操作成功" : "放款操作失败");
        } else {
            return Result.buildSuccess(false,"无此操作");
        }
    }

    /**
     * 二次风控审批
     *
     * @return
     */
    @Security(authorityId = "Order.secondAudit")
    @GetMapping("/secondAudit")
    public Result<Boolean> secondAudit(String channelOrderNumber) {
        if (Application.isDebug()) {
            Tuple<Boolean, String> booleanStringTuple = this.orderService.secondAudit(channelOrderNumber);
            return Result.buildSuccess(booleanStringTuple.getKey(), booleanStringTuple.getValue());
        } else {
            return Result.buildSuccess(false,"无此操作");
        }
    }


    /**
     * 二次风控审批
     *
     * @return
     */
    //@Security(authorityId = "Order.loadSecondAuditJob")
    @GetMapping("/loadSecondAuditJob")
    public Result<Boolean> loadSecondAuditJob() {
        if (Application.isDebug()) {
            this.orderService.loadSecondAuditJob();
            return Result.buildSuccess(true, "success");
        } else {
            return Result.buildSuccess(false,"无此操作");
        }
    }




    /**
     * 贷前关单
     *
     * @return
     */
    @Security(authorityId = "Order.cancelPreLoan")
    @PostMapping("/cancel/loan")
    public Result<Boolean> cancelPreLoan(@RequestParam @Valid String channelOrderNumber) {
        if (Application.isDebug()) {
            return Result.buildSuccess(this.orderService.cancelPreLoan(channelOrderNumber));
        } else {
            return Result.buildSuccess(false,"无此操作");
        }
    }

    /**
     * 贷后关单
     *
     * @return
     */
    @Security(authorityId = "Order.cancelAfterLoan")
    @PostMapping("/cancel/after/loan")
    public Result<Boolean> cancelAfterLoan(@RequestParam @Valid String channelOrderNumber) {
        if (Application.isDebug()) {
            return Result.buildSuccess(this.orderService.cancelAfterLoan(channelOrderNumber));
        } else {
            return Result.buildSuccess(false,"无此操作");
        }
    }

    /**
     * 订单历史状态查询接口
     *
     * @param channelOrderNumber
     * @return
     */
    @Security(authorityId = "Order.getOrderHistoryStatus")
    @ChannelIdInit
    @CheckChannelRoleByChannelOrderNumber
    @GetMapping("/history/status")
    public Result getOrderHistoryStatus(String channelOrderNumber) {
        return Result.buildSuccess(orderService.getOrderHistoryStatus(channelOrderNumber));
    }

    /**
     * 订单状态推送历史记录
     *
     * @param channelOrderNumber
     * @return
     */
    @Security(authorityId = "Order.orderCallbackRecordList")
    @ChannelIdInit
    @CheckChannelRoleByChannelOrderNumber
    @GetMapping("/orderCallbackRecordList")
    public Result orderCallbackRecordList(String channelOrderNumber) {
        return Result.buildSuccess(orderService.getOrderCallbackRecordList(channelOrderNumber));
    }


    /**
     * 订单还款计划
     *
     * @param channelOrderNumber
     * @return
     */
    @Security(authorityId = "Order.repaymentPlan")
    @ChannelIdInit
    @CheckChannelRoleByChannelOrderNumber
    @GetMapping("/repaymentPlan")
    public Result repaymentPlan(String channelOrderNumber, Long loanId) {
        return Result.buildSuccess(orderService.findRepaymentPlan(channelOrderNumber, loanId));
    }





}
