package cn.quantgroup.cashloanflowboss.api.user.controller;

import cn.quantgroup.cashloanflowboss.api.user.dictionary.UserRank;
import cn.quantgroup.cashloanflowboss.api.user.dictionary.UserStatus;
import cn.quantgroup.cashloanflowboss.api.user.entity.User;
import cn.quantgroup.cashloanflowboss.api.user.model.*;
import cn.quantgroup.cashloanflowboss.api.user.service.UserService;
import cn.quantgroup.cashloanflowboss.component.security.annotiation.Security;
import cn.quantgroup.cashloanflowboss.component.validator.constraints.NotEmpty;
import cn.quantgroup.cashloanflowboss.core.base.Result;
import cn.quantgroup.cashloanflowboss.spi.user.service.XyqbUserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import java.util.ArrayList;
import java.util.List;

/**
 * Created by WeiWei on 2019/7/22.
 */
@RestController
@RequestMapping("user")
public class UserController {

    @Autowired
    private UserService userService;
    @Autowired
    private XyqbUserService xyqbUserService;


    /**
     * 注册用户
     *
     * @param registerUserFormModel
     * @return
     */
    @PostMapping()
    public Boolean createUser(@RequestBody @Valid RegisterUserFormModel registerUserFormModel) {
        return this.userService.createUser(registerUserFormModel.getUsername(), registerUserFormModel.getPassword());
    }

    /**
     * 添加用户
     *
     * @param userInfoModel
     * @return
     */
    @Security(authorityId = "User.addUser")
    @PostMapping("saveUserInfo")
    public Result<User> saveUserInfo(@RequestBody UserInfoModel userInfoModel) {
        User info = this.userService.saveUserInfo(userInfoModel);
        return Result.buildSuccess(info);
    }

    /**
     * 获取用户列表（分页）
     *
     * @return
     */
    @GetMapping("/list")
    public Page<User> queryUsers(@Valid QueryUserListModel pagination) {
        return this.userService.getUsers(pagination);
    }

    /**
     * 启用用户
     *
     * @param id 用户ID
     * @return
     */
    @PutMapping("/enable")
    @Security(authorityId = "User.enable")
    public Boolean enableUser(@RequestParam @Valid @NotNull(message = "无效的用户ID") Long id) {
        return this.userService.enableUser(id);
    }

    /**
     * 禁用用户
     *
     * @param id 用户ID
     * @return
     */
    @PutMapping("/disable")
    @Security(authorityId = "User.disable")
    public Boolean disableUser(@RequestParam @Valid @NotNull(message = "无效的用户ID") Long id) {
        return this.userService.disableUser(id);
    }

    /**
     * s删除用户
     *
     * @param id 用户ID
     * @return
     */
    @DeleteMapping("/delete")
    @Security(authorityId = "User.delete")
    public Boolean deleteUser(@RequestParam @Valid @NotNull(message = "无效的用户ID") Long id) {
        return this.userService.deleteUser(id);
    }

    /**
     * 移除用户
     *
     * @param id 用户ID
     * @return
     */
    @DeleteMapping()
    @Security(authorityId = "User.delete")
    public Boolean removeUser(@RequestParam @Valid @NotNull(message = "无效的用户ID") Long id) {
        return this.userService.removeUser(id);
    }

    /**
     * 获取用户详情
     *
     * @return
     */
    @GetMapping("/detail/info")
    @Security(authorityId = "User.userDetailInfo")
    public Result<UserDetailInfo> userDetailInfo() {
        return Result.buildSuccess(userService.getUserDetailInfo());
    }

    /**
     * 清除用户活跃订单
     *
     * @param mobile 用户ID
     * @return
     */
    @PutMapping("/order/clean")
    @Security(authorityId = "User.Order.cleanUserOrder")
    public Result<Boolean> cleanUserOrder(@RequestParam @Valid @NotEmpty(message = "无效的用户手机号") String mobile) {
        userService.cleanUserActiveOrder(mobile);
        return Result.buildSuccess(true, "异步请求成功");
    }

    /**
     * 清除用户信息
     *
     * @param mobile 用户ID
     * @return
     */
    @PutMapping("/info/clean")
    @Security(authorityId = "User.Order.cleanUserInfo")
    public Boolean deleteByUserId(@RequestParam @Valid @NotEmpty(message = "无效的用户手机号") String mobile) {
        this.xyqbUserService.deleteByUserId(mobile);
        return true;
    }


    /**
     * 添加用户
     *
     * @param userModelVo
     * @return
     */

    @Security(authorityId = "User.add")
    @PostMapping(value = "/add")
    public Result<Boolean> addUser(@RequestBody @Valid UserModelVo userModelVo) {
        return userService.addUser(userModelVo);
    }

    @PutMapping("/modify")
    @Security(authorityId = "User.update")
    public Result<Boolean> modifyUser(@RequestBody @Valid UserModelVo userModelVo) {
        return userService.modifyUser(userModelVo);
    }

    /**
     * 根据用户名、角色分页查询
     *
     * @param queryUserListModel
     * @return
     */
    @GetMapping("/list/get")
    @Security(authorityId = "User.list")
    public Result<Page<User>> UserList(@Valid QueryUserListModel queryUserListModel) {
        return userService.getUserList(queryUserListModel);
    }

    @GetMapping("/rank")
    public Result<List<String>> userRank() {
        List<String> list = new ArrayList<>();
        for (UserRank value : UserRank.values()) {
            if (value == UserRank.ADMINISTRATOR || value == UserRank.SUPER_ADMINISTRATOR) {
                continue;
            }
            list.add(value.name());
        }
        return Result.buildSuccess(list);
    }


    @GetMapping("/status")
    public Result<List<String>> userStatus() {
        List<String> list = new ArrayList<>();
        for (UserStatus value : UserStatus.values()) {
            list.add(value.name());
        }
        return Result.buildSuccess(list);
    }

    @PutMapping("/update/password")
    @Security(authorityId = "User.update")
    public Result<Boolean> modifyUser(@RequestBody @Valid UpdatePasswordParam param) {
        return userService.updatePassword(param);
    }

}
