package cn.quantgroup.cashloanflowboss.api.login.service;

import cn.quantgroup.cashloanflowboss.api.login.auth.ApiAuthService;
import cn.quantgroup.cashloanflowboss.api.login.auth.model.LoginUser;
import cn.quantgroup.cashloanflowboss.api.login.model.Principal;
import cn.quantgroup.cashloanflowboss.api.role.entity.Role;
import cn.quantgroup.cashloanflowboss.api.role.repository.RoleRepository;
import cn.quantgroup.cashloanflowboss.api.user.dictionary.UserRank;
import cn.quantgroup.cashloanflowboss.api.user.dictionary.UserStatus;
import cn.quantgroup.cashloanflowboss.api.user.entity.User;
import cn.quantgroup.cashloanflowboss.api.user.model.UserInfo;
import cn.quantgroup.cashloanflowboss.api.user.service.UserService;
import cn.quantgroup.cashloanflowboss.core.Application;
import cn.quantgroup.cashloanflowboss.core.base.Tuple;
import cn.quantgroup.cashloanflowboss.core.dictionary.ApplicationDictionary;
import cn.quantgroup.cashloanflowboss.core.dictionary.ApplicationStatus;
import cn.quantgroup.cashloanflowboss.spi.model.JsonResult;
import cn.quantgroup.cashloanflowboss.utils.IpUtil;
import cn.quantgroup.cashloanflowboss.utils.MD5Tools;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.data.redis.core.ValueOperations;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import java.util.Date;
import java.util.UUID;

/**
 * Created by WeiWei on 2019/7/22.
 */
@Slf4j
@Service
public class LoginServiceImpl implements LoginService {

    @Autowired
    private UserService userService;

    @Autowired
    private HttpServletRequest request;

    @Autowired
    private PrincipalService principalService;

    @Autowired
    private ApiAuthService apiAuthService;

    @Autowired
    private RoleRepository roleRepository;

    @Autowired
    private StringRedisTemplate redisTemplate;

    @Resource(name = "redisTemplate")
    private ValueOperations<String, Principal> loginOperations;

    /**
     * 登入
     *
     * @param username 用户名
     * @param password 密码（明文）
     * @return
     */
    @Override
    public Tuple<ApplicationStatus, String> login(String username, String password) {
        String remoteIP = IpUtil.getRemoteIP(request);
        Long increment = redisTemplate.opsForValue().increment("user:login:ip:" + remoteIP, 1);
        if (increment > 1000) {
            log.info("登陆失败，username={}, msg={}", username, "md有人在搞事情");
            return new Tuple<>(ApplicationStatus.REENTRY_LOCK_EXCEPTION, "");
        }

        User user = this.userService.getUser(username);
        if (user == null) {
            // 如果没有user 查询量星球
            JsonResult<LoginUser> result = apiAuthService.autoLogin(username + "@quantgroup.cn", password, "KA_MA");
            if (!result.isSuccess() || result.getData() == null) {
                log.info("登陆失败，username={}, msg={}", username, result.getMessage());
                return new Tuple<>(ApplicationStatus.INVALID_USER, "");
            } else {
                LoginUser data = result.getData();
                Role role = roleRepository.getByName("量化派操作员");
                user = new User();
                user.setId(-1L);
                user.setUsername(username);
                user.setNickname(data.getName());
                user.setPassword(MD5Tools.md5(password));
                user.setRank(UserRank.OPERATOR);
                user.setRole(role);
                user.setStatus(UserStatus.ENABLED);
            }
        }
        // 检查用户是否被禁用
        if (UserStatus.DISABLED.equals(user.getStatus())) {
            log.info("用户已禁用username=" + username);
            return new Tuple<>(ApplicationStatus.DISABLED_USER, "");
        }
        // 检查密码是否正确
        final String passwordMd5 = MD5Tools.md5(password);
        if (!user.getPassword().equalsIgnoreCase(passwordMd5)) {
            log.info("用户名或密码错误username=" + username);
            log.info("user.name:{},user.pwd:{},passwordMd5:{}", user.getUsername(), user.getPassword(), passwordMd5);
            return new Tuple<>(ApplicationStatus.USERNAME_OR_PASSWORD_ERROR, "");
        }

        // 创建Session
        HttpSession session = this.request.getSession(true);

        // 设置用户主要信息
        Principal principal = new Principal();
        UserInfo userInfo = new UserInfo();
        userInfo.setUserId(user.getId());
        userInfo.setUsername(user.getUsername());
        userInfo.setNickname(user.getNickname());

        principal.setUserInfo(userInfo);
        principal.setChannelId(user.getChannelId());
        principal.setRank(user.getRank());
        principal.setRoles(user.getRole());
        long currentTimeMillis = System.currentTimeMillis();
        principal.setLoginTimeMillis(currentTimeMillis);

        String token = principalService.setUserBossTokenInRedis(username);
        principalService.pushPrincipal(token, principal);

        // session登陆时间，毫秒值
        session.setAttribute(ApplicationDictionary.USER_SESSION_LOGIN_TIME, currentTimeMillis);

        // 保存用户最后登陆时间
        user.setLastLoginTime(new Date());
        userService.updateUser(user);
        return new Tuple<>(ApplicationStatus.SUCCESS, token);
    }

    /**
     * 登出
     *
     * @return
     */
    @Override
    public boolean logout() {
        principalService.removePrincipal();

        this.request.getSession().removeAttribute(ApplicationDictionary.PRINCIPAL);

        return true;

    }

}
