package cn.quantgroup.cashloanflowboss.api.order.controller;
import java.util.Date;

import cn.quantgroup.cashloanflowboss.api.optlog.entity.OptHistoryLog;
import cn.quantgroup.cashloanflowboss.api.optlog.service.OptHistoryLogService;
import cn.quantgroup.cashloanflowboss.api.order.model.ApproveVo;
import cn.quantgroup.cashloanflowboss.api.order.model.LendingFormModel;
import cn.quantgroup.cashloanflowboss.api.order.service.OrderService;
import cn.quantgroup.cashloanflowboss.component.security.annotiation.Security;
import cn.quantgroup.cashloanflowboss.core.annotation.ChannelIdInit;
import cn.quantgroup.cashloanflowboss.core.annotation.CheckChannelRole;
import cn.quantgroup.cashloanflowboss.core.annotation.CheckChannelRoleByChannelOrderNumber;
import cn.quantgroup.cashloanflowboss.core.base.Result;
import cn.quantgroup.cashloanflowboss.core.base.Tuple;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;

/**
 * function:
 * date: 2019/8/8
 *
 * @author: suntao
 */

@RestController
@RequestMapping("/order")
public class OrderController {


    @Autowired
    private OrderService orderService;


    @Value("${debug.model}")
    private Boolean debugModel;

    /**
     * 订单查询列表接口
     *
     * @param channelId
     * @param channelOrderNumber
     * @param pageNumber
     * @param pageSize
     * @return 返回中包含当前订单可操作的 button
     */
    @Security(authorityId = "Order.getOrderList")
    @ChannelIdInit
    @CheckChannelRole
    @GetMapping("/list")
    public Result orderList(Long channelId, String channelOrderNumber, String applyNo, Long loanId, Integer pageNumber, Integer pageSize) {
        if (pageNumber == null || pageSize == null) {
            return Result.buildFial("page信息不对");
        }
        return Result.buildSuccess(orderService.orderList(channelId, channelOrderNumber, applyNo, loanId, pageNumber, pageSize));
    }


    /**
     * 审批接口
     *
     * @param approveVo
     * @return
     */
    @Security(authorityId = "Order.approveOpt")
    @ChannelIdInit
    @PostMapping("/approve")
    public Result approveOpt(@RequestBody @Valid ApproveVo approveVo) {
        if (debugModel) {
            Tuple<Boolean, Boolean> approveOptBooleanAndFundSettingBooleanTuple = orderService.approveOpt(approveVo);
            Boolean approveResult = approveOptBooleanAndFundSettingBooleanTuple.getKey();
            if (approveResult) {
                return Result.buildSuccess(approveResult, approveOptBooleanAndFundSettingBooleanTuple.getValue() ? "审批操作成功" : "请检查资产计划配置，审批操作成功");
            } else {
                return Result.buildSuccess(false,"审批操作失败");
            }
        } else {
            return Result.buildSuccess(false,"无此操作");
        }
    }

    /**
     * 放款
     *
     * @return
     */
    @Security(authorityId = "Order.lending")
    @PostMapping("/lending")
    public Result<Boolean> lending(@RequestBody @Valid LendingFormModel lendingFormModel) {
        if (debugModel) {
            return Result.buildSuccess(this.orderService.lending(lendingFormModel));
        } else {
            return Result.buildSuccess(false,"无此操作");
        }
    }

    /**
     * 贷前关单
     *
     * @return
     */
    @Security(authorityId = "Order.cancelPreLoan")
    @PostMapping("/cancel/loan")
    public Result<Boolean> cancelPreLoan(@RequestBody @Valid String channelOrderNumber) {
        if (debugModel) {
            return Result.buildSuccess(this.orderService.cancelPreLoan(channelOrderNumber));
        } else {
            return Result.buildSuccess(false,"无此操作");
        }
    }

    /**
     * 贷后关单
     *
     * @return
     */
    @Security(authorityId = "Order.cancelAfterLoan")
    @PostMapping("/cancel/after/loan")
    public Result<Boolean> cancelAfterLoan(@RequestParam @Valid String channelOrderNumber) {
        if (debugModel) {
            return Result.buildSuccess(this.orderService.cancelAfterLoan(channelOrderNumber));
        } else {
            return Result.buildSuccess(false,"无此操作");
        }
    }

    /**
     * 订单历史状态查询接口
     *
     * @param channelOrderNumber
     * @return
     */
    @Security(authorityId = "Order.getOrderHistoryStatus")
    @ChannelIdInit
    @CheckChannelRoleByChannelOrderNumber
    @GetMapping("/history/status")
    public Result getOrderHistoryStatus(String channelOrderNumber) {
        return Result.buildSuccess(orderService.getOrderHistoryStatus(channelOrderNumber));
    }

    /**
     * 订单状态推送历史记录
     *
     * @param channelOrderNumber
     * @return
     */
    @Security(authorityId = "Order.orderCallbackRecordList")
    @ChannelIdInit
    @CheckChannelRoleByChannelOrderNumber
    @GetMapping("/orderCallbackRecordList")
    public Result orderCallbackRecordList(String channelOrderNumber) {
        return Result.buildSuccess(orderService.getOrderCallbackRecordList(channelOrderNumber));
    }


    /**
     * 订单还款计划
     *
     * @param channelOrderNumber
     * @return
     */
    @Security(authorityId = "Order.repaymentPlan")
    @ChannelIdInit
    @CheckChannelRoleByChannelOrderNumber
    @GetMapping("/repaymentPlan")
    public Result repaymentPlan(String channelOrderNumber, Long loanId) {
        return Result.buildSuccess(orderService.findRepaymentPlan(channelOrderNumber, loanId));
    }





}
