package cn.quantgroup.cashloanflowboss.core.persistence;

import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

/**
 * Created by WeiWei on 2017/5/17.
 */
@Component
public class SnowflakeGenerator {

    /**
     * 开始时间截 (2015-01-01)
     */
    private final long START_STMP = 1420041600000L;

    /**
     * 每一部分占用的位数
     */
    private final static long SEQUENCE_BIT = 12; //序列号占用的位数
    private final static long MACHINE_BIT = 5;   //机器标识占用的位数
    private final static long DATA_CENTER_BIT = 5;//数据中心占用的位数

    /**
     * 每一部分的最大值
     */
    private final static long MAX_SEQUENCE_NUMBER = -1L ^ (-1L << SEQUENCE_BIT);
    private final static long MAX_MACHINE_NUMBER = -1L ^ (-1L << MACHINE_BIT);
    private final static long MAX_DATA_CENTER_NUMBER = -1L ^ (-1L << DATA_CENTER_BIT);

    /**
     * 每一部分向左的位移
     */
    private final static long MACHINE_LEFT = SEQUENCE_BIT;
    private final static long DATA_CENTER_LEFT = SEQUENCE_BIT + MACHINE_BIT;
    private final static long TIMESTAMP_LEFT = DATA_CENTER_LEFT + DATA_CENTER_BIT;

    /**
     * 数据中心ID
     */
    @Value("${application.server.dataCenterId:0}")
    private long dataCenterId;

    /**
     * 机器标识ID
     */
    @Value("${application.server.machineId:0}")
    private long machineId;

    /**
     * 序列号
     */
    private long sequence;

    /**
     * 上次记录时间
     */
    private long lastTimestamp = -1L;

    public SnowflakeGenerator() {
        if (dataCenterId > MAX_DATA_CENTER_NUMBER || dataCenterId < 0) {
            throw new IllegalArgumentException("dataCenterId can't be greater than MAX_DATA_CENTER_NUMBER or less than 0");
        }
        if (machineId > MAX_MACHINE_NUMBER || machineId < 0) {
            throw new IllegalArgumentException("machineId can't be greater than MAX_MACHINE_NUMBER or less than 0");
        }
    }

    /**
     * 生成下一个ID
     *
     * @return
     */
    public synchronized long nextId() {

        long currentTimestamp = System.currentTimeMillis();

        if (currentTimestamp < lastTimestamp) {
            throw new IllegalStateException("Clock moved backwards. Refusing to generate id");
        }

        if (currentTimestamp == lastTimestamp) {

            //相同毫秒内，序列号自增
            sequence = (sequence + 1) & MAX_SEQUENCE_NUMBER;

            //同一毫秒的序列数已经达到最大
            if (sequence == 0L) {
                while (currentTimestamp <= lastTimestamp) {
                    currentTimestamp = System.currentTimeMillis();
                }
            }

        } else {
            //不同毫秒内，序列号置为0
            sequence = 0L;
        }

        lastTimestamp = currentTimestamp;

        return (currentTimestamp - START_STMP) << TIMESTAMP_LEFT | dataCenterId << DATA_CENTER_LEFT | machineId << MACHINE_LEFT | sequence;

    }

}