package cn.quantgroup.cashloanflowboss.api.role.service;

import cn.quantgroup.cashloanflowboss.api.mapping.UserRoleMapping;
import cn.quantgroup.cashloanflowboss.api.permissionmodule.entity.Permission;
import cn.quantgroup.cashloanflowboss.api.permissionmodule.repository.PermissionRepository;
import cn.quantgroup.cashloanflowboss.api.role.entity.Role;
import cn.quantgroup.cashloanflowboss.api.role.model.RoleModel;
import cn.quantgroup.cashloanflowboss.api.role.model.RoleModelVo;
import cn.quantgroup.cashloanflowboss.api.role.model.RoleQueryModel;
import cn.quantgroup.cashloanflowboss.api.role.repository.RoleRepository;
import cn.quantgroup.cashloanflowboss.core.base.Result;
import cn.quantgroup.cashloanflowboss.api.mapping.UserRoleMappingRepository;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.criteria.Predicate;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

@Slf4j
@Service
public class RoleServiceImpl implements RoleService {

    @Autowired
    private RoleRepository roleRepository;

    @Autowired
    private PermissionRepository permissionRepository;

    @Autowired
    private UserRoleMappingRepository userRoleMappingRepository;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Result<Boolean> addRole(RoleModelVo roleModelVo) {
        final String LOG_PRE = "RoleServiceImpl.addRole";
        log.info("{} 添加角色 roleModelVo={}", LOG_PRE, roleModelVo);

        String name = roleModelVo.getName();
        Role exist = roleRepository.getByName(name);
        if (exist != null) {
            log.error("{} 已存在角色 name={}", LOG_PRE, name);
            return Result.buildFail(name + "已存在，请更换角色名");
        }

        Role role = new Role();
        BeanUtils.copyProperties(roleModelVo, role);

        Long parentId = roleModelVo.getParentId();

        if (parentId != null) {
            log.info("{},父角色查询 parentId={}", LOG_PRE, parentId);
            if (!roleRepository.exists(parentId)) {
                log.error("id为 {} 的父角色不存在");
                return Result.buildFail("父角色不存在");
            }
            Role parent = roleRepository.getOne(parentId);
            role.setParent(parent);
        }

        List<Long> permissions = roleModelVo.getPermissions();
        if (CollectionUtils.isNotEmpty(permissions)) {
            List<Permission> all = permissionRepository.findAll(permissions);
            log.info("{} 权限ids={} 对应的权限size={}", LOG_PRE, permissions, all.size());
            role.setPermissions(all);

        }

        roleRepository.save(role);
        return Result.buildSuccess(true);
    }

    @Override
    public Result<Page<Role>> getRoleByName(RoleQueryModel roleQueryModel) {
        final String LOG_PRE = "RoleServiceImpl.getRoleByName";
        log.info("{} 查询角色 roleQueryModel={}", LOG_PRE, roleQueryModel);
        Page<Role> rolePage = roleRepository.findAll((root, criteriaQuery, criteriaBuilder) -> {
            List<Predicate> predicates = new ArrayList<>();

            if (StringUtils.isNotEmpty(roleQueryModel.getName())) {
                predicates.add(criteriaBuilder.equal(root.get("name"), roleQueryModel.getName()));
            }
            criteriaQuery.where(criteriaBuilder.and(predicates.toArray(new Predicate[predicates.size()])));
            // 指定排序
            criteriaQuery.orderBy(criteriaBuilder.desc(root.get("id")));

            return criteriaQuery.getRestriction();

        }, new PageRequest(roleQueryModel.getPageNumber(), roleQueryModel.getPageSize()));

        return Result.buildSuccess(rolePage);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Result<Boolean> modifyRole(RoleModelVo roleModelVo) {
        final String LOG_PRE = "RoleServiceImpl.modifyRole";
        log.info("{} 更新角色 roleQueryModel={}", LOG_PRE, roleModelVo);

        Long id = roleModelVo.getId();
        if (id == null) {
            log.error("{} 更新的角色没有Id,无法更新 roleModelVo={}", LOG_PRE, roleModelVo);
            return Result.buildFail("角色没有id，无法更新");

        }

        if (!roleRepository.exists(id)) {
            log.error("{},没有对应的角色 id={}", LOG_PRE, id);
            return Result.buildFail("没有对应的角色");
        }
        Role exist = roleRepository.getOne(id);

        BeanUtils.copyProperties(roleModelVo, exist);

        Long parentId = roleModelVo.getParentId();
        if (parentId != null) {
            if(!roleRepository.exists(parentId)){
                log.error("{} 不存在父角色 id={}",LOG_PRE,parentId);
                return Result.buildFail("父角色不存在");
            }
            Role parent = roleRepository.getOne(parentId);
            log.info("{},id={},parentId={},父角色为{}", LOG_PRE, id, parentId, parent);
            exist.setParent(parent);

        }
        List<Long> permissions = roleModelVo.getPermissions();
        if (CollectionUtils.isEmpty(permissions)) {
            exist.setPermissions(null);
        } else {
            List<Permission> permissionList = permissionRepository.findAll(permissions);
            exist.setPermissions(permissionList);
        }
        exist.setUpdateTime(new Date());
        roleRepository.save(exist);

        return Result.buildSuccess(true);
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public Result<Boolean> deleteRole(Long roleId) {
        final String LOG_PRE = "RoleServiceImpl.deleteRole";
        log.info("{} 删除角色 roleId={}", LOG_PRE, roleId);
        if(roleId == null){
            log.error("{} 删除角色失败 id为空");
            return Result.buildFail("角色id不能为空");
        }
        if (roleRepository.exists(roleId)) {
            roleRepository.delete(roleId);
            userRoleMappingRepository.deleteAllByRoleId(roleId);
        }

        return Result.buildSuccess(true);
    }

    @Override
    public Result<List<RoleModel>> getAll() {
        List<Role> all = roleRepository.findAll();
        List<RoleModel> list = new ArrayList<>();
        if(CollectionUtils.isNotEmpty(all)){
            all.forEach(e->{
                RoleModel model = new RoleModel();
                model.setId(e.getId());
                model.setName(e.getName());
                list.add(model);
            });
        }
        return Result.buildSuccess(list);
    }
}
