package cn.quantgroup.cashloanflowboss.api.login.auth;

import cn.quantgroup.cashloanflowboss.api.login.auth.model.LoginUser;
import cn.quantgroup.cashloanflowboss.core.base.ServiceResult;
import cn.quantgroup.cashloanflowboss.spi.model.JsonResult;
import cn.quantgroup.cashloanflowboss.spi.util.HttpService;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.ValueOperations;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.web.client.HttpClientErrorException;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * @author Jie.Feng
 * @date 2017/12/22
 */
@Slf4j
@Service
public class ApiAuthServiceImpl implements ApiAuthService {

    private final static String REDIS_KEY_PREFIX = "login:user:";
    /**
     * 系统标识;
     * 新运营系统;
     */
    private final static String YY_SYSTEM_NAME = "XYYXT";

    /**
     * 成功返回的code
     */
    private final static String RSP_SUCCESS = "0000";

    @Resource(name = "redisTemplate")
    private ValueOperations<String, LoginUser> operations;

    @Autowired
    private HttpService httpService;

    @Value("${system.auth.host}")
    private String authHost;

    static Map<String, String> headers = new HashMap<>(1);

    static {
        headers.put("Content-Type", "application/x-www-form-urlencoded");
    }

    /**
     * 查询系统用户; 权限是 XYYXT  + ZJYYXT
     *
     * @param token
     * @return
     */
    @Override
    public JsonResult<LoginUser> findUser(String token) {
        return findUser(token, YY_SYSTEM_NAME);
    }

    @Override
    public JsonResult<LoginUser> findUser(String token, String... systemName) {
        Assert.notNull(token, "token is null!!");
        Assert.notEmpty(systemName, "systemName is empty!!");
        LoginUser user = operations.get(REDIS_KEY_PREFIX.concat(token));
        if (user != null) {
            return JsonResult.buildSuccess("登陆成功", user);
        } else {
            Map<String, String> param = new HashMap<>(4);
            param.put("token", token);
            param.put("x-requested-with", "XMLHttpRequest");
            String rsp;
            try {
                rsp = httpService.post(authHost.concat("/user/info"), headers, param);
            } catch (HttpClientErrorException e) {
                log.error("鉴权失败,请重新登录: token={}; ", token, e);
                if (e.getStatusCode() == HttpStatus.UNAUTHORIZED) {
                    return JsonResult.buildError("登录失效,请重新登录");
                }
                return JsonResult.buildError("内部错误，请求异常");
            }
            return getLoginUser(token, rsp, systemName);
        }
    }

    private JsonResult<LoginUser> getLoginUser(String token, String rsp, String... systemName) {
        LoginUser user = operations.get(REDIS_KEY_PREFIX.concat(token));
        if (user != null) {
            return JsonResult.buildSuccess("success", user);
        } else {
            JSONObject json = JSONObject.parseObject(rsp);
            String code = json.getString("code");
            if (StringUtils.equals(code, RSP_SUCCESS)) {
                //成功获取到用户
                JSONObject data = json.getJSONObject("data");
                if (data == null) {
                    return JsonResult.buildError("用户不存在");
                }
                user = new LoginUser(data.getString("name"), data.getString("phone"),
                        data.getString("user"), token);
                JSONObject privilege = data.getJSONObject("privilege");
                if (privilege == null) {
                    log.info("无效的用户,用户不存在权限,token: {},username={}", token, user.getName());
                    return JsonResult.buildError("无效的用户");
                }
                //获取系统权限
                for (String n : systemName) {
                    JSONObject systemPrivilege = privilege.getJSONObject(n);
                    if (systemPrivilege == null) {
                        continue;
                    }
                    JSONArray roles = systemPrivilege.getJSONArray("roles");
                    if (roles == null || roles.isEmpty()) {
                        continue;
                    }
                    for (Object role : roles) {
                        //role
                        String r = Objects.toString(role, null);
                        user.addRole(n, r);
                    }
                }
                operations.set(REDIS_KEY_PREFIX.concat(token), user, 1, TimeUnit.HOURS);
            } else {
                log.info("无效token: {}", token);
                return JsonResult.buildError(json.getString("msg"));
            }
            return JsonResult.buildSuccess("success", user);
        }
    }

    @Override
    public JsonResult<LoginUser> autoLogin(String user, String password, String sessionKey) {
        Map<String, String> params = new HashMap<>(4);
        params.put("user", user);
        params.put("password", password);
        String rsp = httpService.get(authHost.concat("/user/login"), headers, params);
        JSONObject jsonObject = JSONObject.parseObject(rsp);
        String code = jsonObject.getString("code");
        if ("0000".equals(code)) {
            JSONObject data = JSONObject.parseObject(jsonObject.getString("data"));
            if (data == null) {
                return JsonResult.buildError("用户不存在");
            }
            String token = data.getString("token");
            JsonResult<LoginUser> result = findUser(token, YY_SYSTEM_NAME);
            if (result.isSuccess() && result.getData() != null)
                operations.set(REDIS_KEY_PREFIX.concat(sessionKey), result.getData());
            log.info("成功初始化用户信息 token {}", token);
            return JsonResult.buildSuccess("success", result.getData());
        }
        return JsonResult.buildError(jsonObject.getString("msg"));
    }
}

