package cn.quantgroup.cashloanflowboss.utils;

import com.alibaba.fastjson.JSONObject;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.google.gson.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.lang.reflect.Type;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Optional;

/**
 * JSON定制化转换工具
 *
 * @author dengcong
 */
@Slf4j
public class JsonUtil {

    public static final Gson GSON = new Gson();

    /**
     * 处理转换数组带来的多余小数点问题
     *
     * @author dengcong
     */
    public static final Gson gsonDoubleAdapter = new GsonBuilder().registerTypeAdapter(Double.class,
            new JsonSerializer<Double>() {
                @Override
                public JsonElement serialize(Double d, Type type, JsonSerializationContext context) {
                    DecimalFormat format = new DecimalFormat("##0");  // 不要小数点
                    String temp = format.format(d);
                    JsonPrimitive pri = new JsonPrimitive(temp);
                    return pri;
                }
            }
    ).create();

    /**
     * 转换List数组
     *
     * @param json
     * @param cls
     * @return
     * @author dengcong
     */
    public static <T> ArrayList<T> fromJsonList(String json, Class<T> cls) {
        ArrayList<T> mList = new ArrayList<T>();
        JsonArray array = new JsonParser().parse(json).getAsJsonArray();
        for (final JsonElement elem : array) {
            mList.add(GSON.fromJson(elem, cls));
        }
        return mList;
    }

    /**
     * 转换List数组
     *
     * @param json
     * @param typeToken Type type = new TypeToken<JsonResultGenericData<List<XyqbLoanApplicationHistory>>>() {}.getType();
     * @return
     */
    public static <T> Optional<T> fromGsonByParamType(String json, Type typeToken) {
        try {
            return Optional.ofNullable(GSON.fromJson(json, typeToken));
        } catch (Exception e) {
            log.error("[exception][fromJson_exception]json={}", json, e);
        }
        return Optional.empty();
    }

    /**
     * 转换List数组
     *
     * @param gson
     * @param json
     * @param cls
     * @return
     * @author suntao
     */
    public static <T> ArrayList<T> fromJsonList(Gson gson, String json, Class<T> cls) {
        ArrayList<T> mList = new ArrayList();
        JsonArray array = new JsonParser().parse(json).getAsJsonArray();
        for (final JsonElement elem : array) {
            mList.add(gson.fromJson(elem, cls));
        }
        return mList;
    }

    /**
     * 避免html转义
     */
    public static final Gson disableHtmlEscapeGson = new GsonBuilder().disableHtmlEscaping().create();

    /**
     * 转换json为对象
     *
     * @param jsonStr
     * @param clazz
     * @param <T>
     * @return
     */
    public static <T> Optional<T> fromJson(String jsonStr, Class<T> clazz) {
        try {
            return Optional.ofNullable(GSON.fromJson(jsonStr, clazz));
        } catch (Exception e) {
            log.error("[exception][fromJson_exception]jsonStr={}", jsonStr, e);
        }
        return Optional.empty();
    }

    /**
     * jackson工具
     * 转换json为对象
     *
     * @param jsonStr
     * @param clazz
     * @param <T>
     * @return
     */
    public static <T> Optional<T> fromJsonWithOrdinalEnum(String jsonStr, Class<T> clazz) {
        try {
            return Optional.ofNullable(getObjectMapper().readValue(jsonStr, clazz));
        } catch (Exception e) {
            log.error("[exception][fromJson_exception]error={}", e);
        }
        return Optional.empty();
    }

    /**
     * jackson工具
     * 转换json为对象
     *
     * @param jsonStr
     * @param typeReference TypeReference<JsonResultGenericData<XyqbLoanApplicationHistory>> typeReference = new TypeReference<JsonResultGenericData<XyqbLoanApplicationHistory>>() {};
     * @param <T>
     * @return
     */
    public static <T> Optional<T> fromJsonWithOrdinalEnum(String jsonStr, TypeReference typeReference) {
        try {
            return Optional.ofNullable(getObjectMapper().readValue(jsonStr, typeReference));
        } catch (Exception e) {
            log.error("[exception][fromJson_exception]jsonStr={}", jsonStr, e);
        }
        return Optional.empty();
    }

    private static ObjectMapper getObjectMapper() {
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.configure(SerializationFeature.WRITE_ENUMS_USING_INDEX, true);
        objectMapper.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
//        objectMapper.setDateFormat(new SimpleDateFormat(DateUtil.DATE_FORMAT_1));
        return objectMapper;
    }

    /**
     * 将对象转成json字符串
     *
     * @param object
     * @return
     */
    public static String toJson(Object object) {
        return GSON.toJson(object);
    }

    public static String toJSONString(Object obj) {
        return JSONObject.toJSONString(obj);
    }

    public static Object toSubString(Object addInfo, int beginIndex, int endIndex) {
        return StringUtils.substring(toJSONString(addInfo), beginIndex, endIndex);
    }
}
