package cn.quantgroup.cashloanflowboss.api.order.controller;

import cn.quantgroup.cashloanflowboss.api.order.model.ApproveVo;
import cn.quantgroup.cashloanflowboss.api.order.model.LendingFormModel;
import cn.quantgroup.cashloanflowboss.api.order.model.RepayStatusEnum;
import cn.quantgroup.cashloanflowboss.api.order.model.RepaymentPlansResultModel;
import cn.quantgroup.cashloanflowboss.api.order.service.MoOrderServiceImpl;
import cn.quantgroup.cashloanflowboss.api.order.service.OrderService;
import cn.quantgroup.cashloanflowboss.component.route.ServiceRoute;
import cn.quantgroup.cashloanflowboss.component.security.annotiation.Security;
import cn.quantgroup.cashloanflowboss.core.Application;
import cn.quantgroup.cashloanflowboss.core.annotation.channelrole.ChannelIdInit;
import cn.quantgroup.cashloanflowboss.core.annotation.channelrole.CheckChannelRole;
import cn.quantgroup.cashloanflowboss.core.base.Result;
import cn.quantgroup.cashloanflowboss.core.base.Tuple;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.Objects;

/**
 * function:
 * date: 2019/8/8
 *
 * @author: suntao
 */

@RestController
@RequestMapping("/order")
@Slf4j
public class OrderController {


    @Autowired
    private ServiceRoute serviceRoute;


    /**
     * 订单查询列表接口
     * <p>
     * 渠道用户只能查自己渠道订单 如果是别的渠道订单 不能查询（如果channelId 赋值 channelOrderNumber 是别的渠道订单 应该查询不到数据，所以不需要CheckChannelRole）
     *
     * @param channelId
     * @param channelOrderNumber
     * @param pageNumber
     * @param pageSize
     * @return 返回中包含当前订单可操作的 button
     */
    @Security(authorityId = "Order.getOrderList")
    @ChannelIdInit(channelIdSPEL = "#this[0]")
    @GetMapping("/list")
    public Result orderList(Long channelId, String channelOrderNumber, String applyNo, Long loanId, Integer pageNumber, Integer pageSize) {
        if (pageNumber == null || pageSize == null) {
            return Result.buildFail("page信息不对");
        }
        OrderService orderService = Application.getBean(MoOrderServiceImpl.class);
        return Result.buildSuccess(orderService.orderList(channelId, channelOrderNumber, applyNo, loanId, pageNumber, pageSize));
    }


    /**
     * 审批接口
     * <p>
     * 渠道只能审批自己渠道的订单
     *
     * @param approveVo
     * @return
     */
    @Security(authorityId = "Order.approveOpt")
    @CheckChannelRole(channelOrderNumberSPEL = "#this[0].channelOrderNumber")
    @PostMapping("/approve")
    public Result approveOpt(@RequestBody @Valid ApproveVo approveVo) {
        if (Application.isDebug()) {
            OrderService orderService = serviceRoute.routeOrderService(null,approveVo.getChannelOrderNumber(),null);
            final Tuple<Boolean, String> approveOptTuple = orderService.approveOpt(approveVo);
            return Result.buildSuccess(approveOptTuple.getKey(), approveOptTuple.getValue());
        } else {
            return Result.buildSuccess(false, "无此操作");
        }
    }

    /**
     * 放款
     *
     * @return
     */
    @Security(authorityId = "Order.lending")
    @CheckChannelRole(channelOrderNumberSPEL = "#this[0].channelOrderNumber")
    @PostMapping("/lending")
    public Result<Boolean> lending(@RequestBody @Valid LendingFormModel lendingFormModel) {
        if (Application.isDebug()) {
            OrderService orderService = serviceRoute.routeOrderService(lendingFormModel.getChannelId(),lendingFormModel.getChannelOrderNumber(),null);
            final Tuple<Boolean, String> lending = orderService.lending(lendingFormModel);
            return Result.buildSuccess(lending.getKey(), lending.getValue());
        } else {
            return Result.buildSuccess(false, "无此操作");
        }
    }

    /**
     * 二次风控审批
     *
     * @return
     */
    @Security(authorityId = "Order.secondAudit")
    @CheckChannelRole(channelOrderNumberSPEL = "#this[0]")
    @GetMapping("/secondAudit")
    public Result<Boolean> secondAudit(String channelOrderNumber) {
        if (Application.isDebug()) {
            OrderService orderService = serviceRoute.routeOrderService(null,channelOrderNumber,null);
            final Tuple<Boolean, String> booleanStringTuple = orderService.secondAudit(channelOrderNumber);
            return Result.buildSuccess(booleanStringTuple.getKey(), booleanStringTuple.getValue());
        } else {
            return Result.buildSuccess(false, "无此操作");
        }
    }


    /**
     * 二次风控审批
     *
     * @return
     */
    //@Security(authorityId = "Order.loadSecondAuditJob")
    @GetMapping("/loadSecondAuditJob")
    public Result<Boolean> loadSecondAuditJob() {
        if (Application.isDebug()) {
            OrderService orderService = serviceRoute.routeOrderService(null,null,null);
            orderService.loadSecondAuditJob();
            return Result.buildSuccess(true, "success");
        } else {
            return Result.buildSuccess(false, "无此操作");
        }
    }


    /**
     * 贷前关单
     *
     * @return
     */
    @Security(authorityId = "Order.cancelPreLoan")
    @CheckChannelRole(channelOrderNumberSPEL = "#this[0]")
    @PostMapping("/cancel/loan")
    public Result<Boolean> cancelPreLoan(@RequestParam @Valid String channelOrderNumber) {
        if (Application.isDebug()) {
            OrderService orderService = serviceRoute.routeOrderService(null,channelOrderNumber,null);
            final Tuple<Boolean, String> preLoan = orderService.cancelPreLoan(channelOrderNumber);
            return Result.buildSuccess(preLoan.getKey(), preLoan.getValue());
        } else {
            return Result.buildSuccess(false, "无此操作");
        }
    }

    /**
     * 贷后关单
     *
     * @return
     */
    @Security(authorityId = "Order.cancelAfterLoan")
    @CheckChannelRole(channelOrderNumberSPEL = "#this[0]")
    @PostMapping("/cancel/after/loan")
    public Result<Boolean> cancelAfterLoan(@RequestParam @Valid String channelOrderNumber) {
        if (Application.isDebug()) {
            OrderService orderService = serviceRoute.routeOrderService(null,channelOrderNumber,null);
            final Tuple<Boolean, String> afterLoan = orderService.cancelAfterLoan(channelOrderNumber);
            return Result.buildSuccess(afterLoan.getKey(), afterLoan.getValue());
        } else {
            return Result.buildSuccess(false, "无此操作");
        }
    }

    /**
     * 订单历史状态查询接口
     *
     * @param channelOrderNumber
     * @return
     */
    @Security(authorityId = "Order.getOrderHistoryStatus")
    @CheckChannelRole(channelOrderNumberSPEL = "#this[0]")
    @GetMapping("/history/status")
    public Result getOrderHistoryStatus(String channelOrderNumber) {
        OrderService orderService = serviceRoute.routeOrderService(null,channelOrderNumber,null);
        return Result.buildSuccess(orderService.getOrderHistoryStatus(channelOrderNumber));
    }

    /**
     * 订单状态推送历史记录
     *
     * @param channelOrderNumber
     * @return
     */
    @Security(authorityId = "Order.orderCallbackRecordList")
    @CheckChannelRole(channelOrderNumberSPEL = "#this[0]")
    @GetMapping("/orderCallbackRecordList")
    public Result orderCallbackRecordList(String channelOrderNumber) {
        OrderService orderService = serviceRoute.routeOrderService(null,channelOrderNumber,null);
        return Result.buildSuccess(orderService.getOrderCallbackRecordList(channelOrderNumber));
    }


    /**
     * 订单还款计划
     *
     * @param channelOrderNumber
     * @return
     */
    @Security(authorityId = "Order.repaymentPlan")
    @CheckChannelRole(channelOrderNumberSPEL = "#this[0]")
    @GetMapping("/repaymentPlan")
    public Result repaymentPlan(String channelOrderNumber, Long loanId) {
        if (StringUtils.isEmpty(channelOrderNumber) && Objects.isNull(loanId)) {
            log.error("还款计划查询 channelOrderNumber 和 loanId 不能同时为空 ");
            return Result.buildFail("channelOrderNumber 和 loanId 不能同时为空");
        }
        OrderService orderService = serviceRoute.routeOrderService(null,channelOrderNumber,loanId);
        RepaymentPlansResultModel result = orderService.findRepaymentPlan(channelOrderNumber, loanId);
        if (result != null) {
            result.getRepaymentPlans().forEach(r -> {
                RepayStatusEnum statusEnum = RepayStatusEnum.getByValue(r.getRepaymentStatus());
                r.setRepaymentStatus(statusEnum == null ? null : statusEnum.getDesc());
            });
        }
        return Result.buildSuccess(result);
    }

    /**
     * 发起结清
     *
     * @param channelOrderNumber
     * @return
     */
    @Security(authorityId = "Order.settle")
    @CheckChannelRole(channelOrderNumberSPEL = "#this[0]")
    @RequestMapping("/settle")
    public Result settle(String channelOrderNumber) {
        OrderService orderService = serviceRoute.routeOrderService(null,channelOrderNumber,null);
        final Tuple<Boolean, String> settle = orderService.settle(channelOrderNumber);
        return Result.buildSuccess(settle.getKey(), settle.getValue());
    }


}
