package cn.quantgroup.cashloanflowboss.api.order.service;

import cn.quantgroup.cashloanflowboss.api.channel.entity.ChannelConf;
import cn.quantgroup.cashloanflowboss.api.channel.repository.ChannelConfRepository;
import cn.quantgroup.cashloanflowboss.api.channel.util.ChannelConfUtil;
import cn.quantgroup.cashloanflowboss.api.optlog.entity.OptHistoryLog;
import cn.quantgroup.cashloanflowboss.api.optlog.model.OptEnumName;
import cn.quantgroup.cashloanflowboss.api.optlog.service.OptHistoryLogServiceImpl;
import cn.quantgroup.cashloanflowboss.api.order.entity.OrderApprove;
import cn.quantgroup.cashloanflowboss.api.order.model.*;
import cn.quantgroup.cashloanflowboss.api.order.repository.OrderApproveRepository;
import cn.quantgroup.cashloanflowboss.api.order.util.OrderUtil;
import cn.quantgroup.cashloanflowboss.core.Application;
import cn.quantgroup.cashloanflowboss.core.annotation.opt.OperationAnno;
import cn.quantgroup.cashloanflowboss.core.base.ServiceResult;
import cn.quantgroup.cashloanflowboss.core.base.Tuple;
import cn.quantgroup.cashloanflowboss.core.constants.Constants;
import cn.quantgroup.cashloanflowboss.spi.clf.entity.CallbackFailRecord;
import cn.quantgroup.cashloanflowboss.spi.clf.entity.CallbackRecord;
import cn.quantgroup.cashloanflowboss.spi.clf.entity.ClfChannelConfiguration;
import cn.quantgroup.cashloanflowboss.spi.clf.entity.ClfOrderMapping;
import cn.quantgroup.cashloanflowboss.spi.clf.model.CallbackRecordVoModel;
import cn.quantgroup.cashloanflowboss.spi.clf.model.KANoticeType;
import cn.quantgroup.cashloanflowboss.spi.clf.model.LoanProgress;
import cn.quantgroup.cashloanflowboss.spi.clf.repository.ClfOrderMappingRepository;
import cn.quantgroup.cashloanflowboss.spi.clf.service.CLFCenterService;
import cn.quantgroup.cashloanflowboss.spi.clotho.service.ClothoCenterService;
import cn.quantgroup.cashloanflowboss.spi.jolyne.JolyneService;
import cn.quantgroup.cashloanflowboss.spi.opapi.OPCenter;
import cn.quantgroup.cashloanflowboss.spi.user.service.XyqbUserService;
import cn.quantgroup.cashloanflowboss.spi.xyqb.entity.*;
import cn.quantgroup.cashloanflowboss.spi.xyqb.repository.CancelPreLoanRepository;
import cn.quantgroup.cashloanflowboss.spi.xyqb.service.XYQBCenterService;
import cn.quantgroup.cashloanflowboss.utils.JSONTools;
import cn.quantgroup.user.retbean.XUser;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.PostConstruct;
import javax.persistence.criteria.Predicate;
import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.*;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * function:
 * date: 2019/8/8
 *
 * @author: suntao
 */

@Slf4j
@Service
public class OrderServiceImpl implements OrderService {

    //多订单渠道配置
    @Value("${multiple.order.channels}")
    private String multipleOrderChannel;

    @Autowired
    private OptHistoryLogServiceImpl optHistoryLogService;

    @Autowired
    private ChannelConfRepository channelConfRepository;
    @Autowired
    private OrderApproveRepository orderApproveRepository;


    @Autowired
    private XyqbUserService xyqbUserService;
    @Autowired
    private XYQBCenterService xyqbCenterService;
    @Autowired
    private ClothoCenterService clothoCenterService;
    @Autowired
    private CLFCenterService clfCenterService;
    @Autowired
    private JolyneService jolyneService;


    @Autowired
    private ClfOrderMappingRepository clfOrderMappingRepository;

    @Autowired
    private CancelPreLoanRepository cancelPreLoanRepository;

    @Autowired
    private OPCenter opCenter;

    @Value("${cloth.auth}")
    private String auth;

    private static final int CONSCONT_STATUS = 2;

    private List<Long> channels;

    @PostConstruct
    public void init() {
        channels = Stream.of(multipleOrderChannel.split(",")).map(Long::valueOf).collect(Collectors.toList());
    }

    @Deprecated
    @Override
    public Page<OrderVo> orderList(Long channelId, String channelOrderNo, String applyNo, Long loanId, Integer pageNumber, Integer pageSize) {

        Page<ClfOrderMapping> page = this.clfOrderMappingRepository.findAll((root, criteriaQuery, criteriaBuilder) -> {

            List<Predicate> predicates = new ArrayList<>();

            // 指定渠道号
            if (Objects.nonNull(channelId)) {
                predicates.add(criteriaBuilder.equal(root.get("registeredFrom"), channelId.longValue()));
            }
            if (StringUtils.isNotEmpty(channelOrderNo)) {
                predicates.add(criteriaBuilder.equal(root.get("channelOrderNo"), channelOrderNo));
            }
            if (StringUtils.isNotEmpty(applyNo)) {
                predicates.add(criteriaBuilder.equal(root.get("applyNo"), applyNo));
            }
            if (loanId != null && loanId > 0) {
                predicates.add(criteriaBuilder.equal(root.get("loanId"), loanId));
            }

            // 设置查询条件
            criteriaQuery.where(criteriaBuilder.and(predicates.toArray(new Predicate[predicates.size()])));
            // 指定排序
            criteriaQuery.orderBy(criteriaBuilder.desc(root.get("id")));

            return criteriaQuery.getRestriction();

        }, new PageRequest(pageNumber, pageSize));


        return page.map(it -> {
            OrderVo orderVo = new OrderVo();
            orderVo.setId(it.getId());
            orderVo.setChannelId(it.getRegisteredFrom());
            orderVo.setChannelOrderNumber(it.getChannelOrderNo());
            if (!Application.getPrincipal().isChannel()) {
                orderVo.setLoanId(it.getLoanId());
                // 授信申请号，非渠道权限登陆显示admin及以上权限用户
                orderVo.setApplyNo(it.getApplyNo());
            }
            orderVo.setCreatedAt(it.getCreatedAt());
            if (StringUtils.isNotEmpty(it.getApplyNo())) {

                ServiceResult<XyqbCurrentOrderStatusServiceResultModel> xyqbCenterServiceXyqbOrderStatus = xyqbCenterService.getXyqbOrderStatus(it.getApplyNo(), it.getLoanId());
                log.info("[xyqbCenterService.getXyqbOrderStatus]xyqbOrderStatus={}", JSONTools.serialize(xyqbCenterServiceXyqbOrderStatus));
                if (xyqbCenterServiceXyqbOrderStatus.isSuccess()) {
                    QueryXyqbOrderStatus currentStatus = xyqbCenterServiceXyqbOrderStatus.getData().getCurrentStatus();

                    Tuple<String, List<OrderVo.OptButton>> currentStatusAndButtons = this.getCurrentStatusAndButtons(currentStatus, it);
                    orderVo.setStatus(currentStatusAndButtons.getKey());
                    if (Application.isDebug()) {
                        // 只有测试环境才需要订单操作
                        orderVo.setOpt(currentStatusAndButtons.getValue());
                    }
                    orderVo.setMessage("");
                } else {
                    orderVo.setStatus("未知，稍后查询");
                    orderVo.setMessage("订单查询错误");
                }
            } else {
                orderVo.setStatus("授信中");
                orderVo.setOpt(new ArrayList<>());
                orderVo.setMessage("");
            }

            return orderVo;
        });
    }

    /**
     * @param approveVo
     * @return Tuple<Boolean 审批结果 true 审批完成 false 审批操作失败, String msg 信息>
     */
    @OperationAnno(channelNo = "#this[0].channelOrderNumber", opt = OptEnumName.AUDIT, succSPEL = "#this.key", optDetailSPEL = "#this.value")
    @Override
    public Tuple<Boolean, String> approveOpt(ApproveVo approveVo) {
        log.info("approveOpt,审批操作入参，approveVo={}", JSONTools.serialize(approveVo));
        if (StringUtils.isEmpty(approveVo.getAmount())) {
            approveVo.setAmount("4000");
        }
        if (approveVo.getPeriod() == null) {
            approveVo.setPeriod(3);
        }
        ClfOrderMapping orderMapping = clfOrderMappingRepository.findByChannelOrderNoLastOne(approveVo.getChannelOrderNumber());
        if (orderMapping == null) {
            log.info("approveOpt,审批失败，无订单 channelOrderNumber={}", approveVo.getChannelOrderNumber());
            return new Tuple(false, "审批失败，无订单");
        }
        XUser xUser = xyqbUserService.findXUserById(orderMapping.getQgUserId());
        if (xUser == null) {
            log.info("approveOpt,审批失败，未找到用户 channelOrderNumber={}", approveVo.getChannelOrderNumber());
            return new Tuple(false, "审批失败，未找到用户");
        }
        ChannelConf channelConf = channelConfRepository.getByChannelId(orderMapping.getRegisteredFrom());
        if (channelConf == null) {
            log.info("approveOpt,审批失败，boss渠道配置为空 channelOrderNumber={}", approveVo.getChannelOrderNumber());
            return new Tuple(false, "审批失败，boss渠道配置为空");
        }
        Integer fundId = ChannelConfUtil.getFundIdByType(approveVo.getFundType(), channelConf);
        FinanceProduct financeProduct = xyqbCenterService.getByFundCorpId(Long.valueOf(fundId));
        log.info("approveOpt审批，fundProductId={}", financeProduct.getId());
        // 资方 及 期数额度规则

        BigDecimal fundRate = getFundRate(orderMapping.getApplyNo());
        String fundFormat = null;
        if(Objects.isNull(fundRate)){
            fundFormat = String.format(OrderUtil.financeProductsFormat, approveVo.getAmount(), approveVo.getAmount(),
                    approveVo.getPeriod(), fundId, financeProduct.getId());
        }else {
            fundFormat = String.format(OrderUtil.financeProductsWithFundRateFormat, approveVo.getAmount(), approveVo.getAmount(),
                    approveVo.getPeriod(), fundId, fundRate.toString(),financeProduct.getId());
        }
        saveOrUpdateOrderApprove(approveVo, orderMapping, fundId);
        ClfChannelConfiguration clfChannelConfiguration = clfCenterService.findChannelConfigurationByChannelId(orderMapping.getRegisteredFrom());
        if (clfChannelConfiguration == null) {
            return new Tuple(false, "审批失败，clfChannelConfiguration渠道配置为空");
        }
        QuotaProduct quotaProduct = null;
        Map<String, Object> data = null;
        if (channels.contains(orderMapping.getRegisteredFrom())) {
            data = getApproveObjectMap(approveVo, orderMapping, xUser, fundFormat, 9);

        } else {
            quotaProduct = xyqbCenterService.getXyqbProduct(clfChannelConfiguration.getXyqbProductId());
            if (quotaProduct == null) {
                return new Tuple(false, "审批失败，xyqb.quota_product配置为空(".concat(clfChannelConfiguration.getXyqbProductId()).concat(")"));
            }
            data = getApproveObjectMap(approveVo, orderMapping, xUser, fundFormat, quotaProduct.getProductType());
        }


        // 发起审批
        log.info("[approveOpt] 向clotho发起审批，data={}", JSONTools.serialize(data));
        String approveStringResult = clothoCenterService.approve(data);
        log.info("[approveOpt] 返回结果 result={}", approveStringResult);
        Boolean corpPolicyValidate = false;
        try {
            corpPolicyValidate = xyqbCenterService.validateCorpPolicy(Long.valueOf(fundId), new Date());
        } catch (Exception e) {
            log.error("[approveOpt]审批,资方校验异常，channelOrderNumber={}，fundId={}, e={}", orderMapping.getChannelOrderNo(), fundId, ExceptionUtils.getStackTrace(e));
        }
        // 操作记录
        String optLogDetail = approveVo.getIsPass() ? "审批通过:(".concat(fundId + "").concat(")") : "审批拒绝:";

        Boolean optResult;
        if (Constants.SUCCESS.equals(approveStringResult)) {
            optLogDetail = optLogDetail.concat(corpPolicyValidate ? "审批操作成功" : "请检查资产配置，请联系管理员配置，审批操作成功");
            optResult = true;
            log.info("[approveOpt]审批申请成功，channelOrderNumber={}", orderMapping.getChannelOrderNo());
        } else {
            optLogDetail = optLogDetail.concat("clotho审批失败");
            log.info("[approveOpt]审批申请失败，channelOrderNumber={}，requestParam={}", orderMapping.getChannelOrderNo(), JSONTools.serialize(data));
            optResult = false;
        }
        return new Tuple(optResult, optLogDetail);
    }

    /**
     * 二次风控
     *
     * @param channelOrderNumber
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    @OperationAnno(channelNo = "#this[0]", opt = OptEnumName.SECORD_AUDIT, succSPEL = "#this.key", optDetailSPEL = "#this.value")
    public Tuple<Boolean, String> secondAudit(String channelOrderNumber) {

        boolean result = true;


        ClfOrderMapping orderMapping = this.clfOrderMappingRepository.findByChannelOrderNoLastOne(channelOrderNumber);
        if (orderMapping == null || orderMapping.getLoanId() == null) {
            log.info("secondAudit,orderMapping为空或者loanId为空，channelOrderNumber=".concat(channelOrderNumber));
            return new Tuple<>(false, "orderMapping为空或者loanId为空");
        }

        // 更新合同状态
        Contract conscont = xyqbCenterService.findContractByUserId(orderMapping.getQgUserId());
        if (conscont != null) {
            if (conscont.getGenerateStatus() != CONSCONT_STATUS) {
                log.info("secondAudit,合同状态不对，channelOrderNumber=".concat(channelOrderNumber));
                // 修改合同状态
                xyqbCenterService.updateContractStatus(orderMapping.getQgUserId(), CONSCONT_STATUS);
            }
        } else {
            // 合同为空 log表记录问题，UI用户查询
            log.info("secondAudit,合同为空，channelOrderNumber=".concat(channelOrderNumber));
            // 不需要return
        }

        // 二次风控回调
        XUser xUser = xyqbUserService.findXUserById(orderMapping.getQgUserId());
        if (xUser == null) {
            log.info("[secendAudit]二次风控失败，未找到user channelOrderNumber={}", channelOrderNumber);
            return new Tuple<>(false, "二次风控失败，未找到user");
        }

        LoanApplicationHistory loanApplicationHistory = xyqbCenterService.findLoanApplicationHistoryById(orderMapping.getLoanId());

        if (LoanProgress.START_ORDER.equals(loanApplicationHistory.getProgress())) {

            ClfChannelConfiguration clfChannelConfiguration = clfCenterService.findChannelConfigurationByChannelId(orderMapping.getRegisteredFrom());
            if (clfChannelConfiguration == null) {
                return new Tuple(false, "二次风控失败，clfChannelConfiguration渠道配置为空");
            }


            if (channels.contains(clfChannelConfiguration.getRegisteredFrom())) {//乐信渠道 bizType直接为9
                result = clothoCenterService.orderAuditNotify(xUser.getUuid(), orderMapping.getLoanId(), true, 9);
            } else {

                QuotaProduct quotaProduct = xyqbCenterService.getXyqbProduct(clfChannelConfiguration.getXyqbProductId());
                if (quotaProduct == null) {
                    return new Tuple(false, "二次风控失败，xyqb.quota_product配置为空(".concat(clfChannelConfiguration.getXyqbProductId()).concat(")"));
                }
                // 50
                log.info("[secendAudit]二次风控发起,channelOrderNumber={}, bizType={}", channelOrderNumber, quotaProduct.getProductType());
                result = clothoCenterService.orderAuditNotify(xUser.getUuid(), orderMapping.getLoanId(), true, quotaProduct.getProductType());
            }
            if (result) {
                log.info("[secendAudit]二次风控成功,channelOrderNumber={}, result={}", channelOrderNumber, result);
            } else {
                log.info("[secendAudit]二次风控失败，clotho返回失败 channelOrderNumber={}", channelOrderNumber);
                return new Tuple<>(false, "二次风控失败，clotho返回失败");
            }
        } else if (LoanProgress.WAITING_FUND.equals(loanApplicationHistory.getProgress())) {
            // 20 已经回调过
            result = true;
        }

        jolyneService.delayUpdateWaitingXyqbSql(orderMapping.getLoanId(), 200L);

        log.info("[secendAudit]二次风控操作 最后结果,channelOrderNumber={}, result={}", channelOrderNumber, result);

        return new Tuple<>(result, "二次风控审批完成,请等待,有部分异步任务");

    }

    /**
     * 放款
     *
     * @param lendingFormModel
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    @OperationAnno(channelNo = "#this[0].channelOrderNumber", opt = OptEnumName.PAY, succSPEL = "#this.key", optDetailSPEL = "#this.value")
    public Tuple<Boolean, String> lending(LendingFormModel lendingFormModel) {

        boolean result = false;

        OrderApprove orderApprove = this.orderApproveRepository.findByChannelOrderNumber(lendingFormModel.getChannelOrderNumber());

        if (orderApprove == null) {
            return new Tuple<>(false, "需要使用KA联调平台先审批操作，才能使用放款操作");
        }

        ClfOrderMapping orderMapping = this.clfOrderMappingRepository.findByChannelOrderNoAndRegisteredFromLastOne(lendingFormModel.getChannelOrderNumber(), lendingFormModel.getChannelId());

        String optDetail = BooleanUtils.isTrue(lendingFormModel.getResult()) ? "放款成功:" : "放款失败:";


        // 更新合同状态
        Contract conscont = xyqbCenterService.findContractByUserId(orderMapping.getQgUserId());
        if (conscont == null) {
            // 合同为空 log表记录问题，UI用户查询
            optDetail = optDetail.concat("合同未生成，请注意，需要不用身份证才能自动生成合同,");
        }


        if (orderApprove.getFundType() == 0) {
            // 如果是非存管
            result = this.xyqbCenterService.payResultNotify(orderMapping.getLoanId(), lendingFormModel.getResult());
            log.info("[order_lending]非存管，放款结果通知,channelOrderNumber={}, result={}", lendingFormModel.getChannelOrderNumber(), result);
        } else {
            // p2p

            Map data = null;
            try {
                data = this.xyqbCenterService.queryLendingRecordCount(Long.valueOf(orderApprove.getFundId()));
            } catch (Exception e) {
                log.error("[order_lending][queryLendingRecordCount]查询待打款数量 失败,channelOrderNumber={}, data={}", lendingFormModel.getChannelOrderNumber(), data);
            }
            log.info("[order_lending][queryLendingRecordCount]查询待打款数量,channelOrderNumber={}, data={}", lendingFormModel.getChannelOrderNumber(), data);
            if (data != null && data.size() > 0) {
                if (Objects.nonNull(data.get(Constants.TOTAL_AMOUNT)) && Objects.nonNull(data.get(Constants.TOTAL_COUNT))) {
                    result = this.clothoCenterService.lending(orderApprove.getFundId(), new BigDecimal(String.valueOf(data.get(Constants.TOTAL_AMOUNT))), Integer.valueOf(String.valueOf(data.get(Constants.TOTAL_COUNT))));
                    log.info("[order_lending]直投打款,channelOrderNumber={}, result={}", lendingFormModel.getChannelOrderNumber(), result);
                }
            }

            result = true;
        }

        if (result) {
            optDetail = optDetail.concat("放款操作成功");
        } else {
            optDetail = optDetail.concat("放款操作失败");
        }
        log.info("[order_lending]放款操作 最后结果,channelOrderNumber={}, result={}", lendingFormModel.getChannelOrderNumber(), result);

        return new Tuple<>(result, optDetail);

    }

    @Override
    public HistoryOrderStatusVoModel getOrderHistoryStatus(String channelOrderNumber) {
        ClfOrderMapping orderMapping = clfOrderMappingRepository.findByChannelOrderNoLastOne(channelOrderNumber);
        HistoryOrderStatusVoModel historyOrderStatusVoModel = new HistoryOrderStatusVoModel();
        if (orderMapping == null) {
            return historyOrderStatusVoModel;
        }

        ServiceResult<XyqbHistoryOrderStatusServiceResultModel> xyqbOrderHistoryStatusResult = xyqbCenterService.getXyqbOrderHistoryStatus(orderMapping.getApplyNo(), orderMapping.getLoanId());
        if (xyqbOrderHistoryStatusResult.isSuccess()) {
            log.info("[getOrderHistoryStatus]查询成功channelOrderNumber={}, xyqbOrderHistoryStatusResult={}", channelOrderNumber, JSONTools.serialize(xyqbOrderHistoryStatusResult));
            XyqbHistoryOrderStatusServiceResultModel data = xyqbOrderHistoryStatusResult.getData();
            List<VoOrderStatus> historyStatusList = new ArrayList<>();
            for (QueryXyqbOrderStatus queryXyqbOrderStatus : data.getHistoryStatus()) {
                VoOrderStatus voOrderStatus = new VoOrderStatus();
                Tuple<String, String> stautsName = OrderUtil.KaNoticeTypeNameConvertChnName(queryXyqbOrderStatus.getStatus());
                voOrderStatus.setStatus(stautsName.getValue());
                voOrderStatus.setUpdateTime(new Date(queryXyqbOrderStatus.getUpdateTime()));
                historyStatusList.add(voOrderStatus);
            }
            historyOrderStatusVoModel.setHistoryStatus(historyStatusList);
            return historyOrderStatusVoModel;
        } else {
            log.info("[getOrderHistoryStatus]查询失败channelOrderNumber={}, xyqbOrderHistoryStatusResult={}", channelOrderNumber, JSONTools.serialize(xyqbOrderHistoryStatusResult));
            return historyOrderStatusVoModel;
        }
    }

    /**
     * 贷前关单
     *
     * @param
     * @return
     */
    @OperationAnno(channelNo = "#this[0]", opt = OptEnumName.PRE_CANCEL, succSPEL = "#this.key", optDetailSPEL = "#this.value")
    @Override
    public Tuple<Boolean, String> cancelPreLoan(String channelOrderNumber) {
        boolean cancel = true;
        String token = null;
        String result = "";
        ClfOrderMapping orderMapping = clfOrderMappingRepository.findByChannelOrderNoLastOne(channelOrderNumber);
        if (orderMapping == null) {
            log.info("cancelPreLoan,贷前关单失败，无订单 channelOrderNumber={}", channelOrderNumber);
            return new Tuple<>(false, "贷前关单失败，无订单");
        }
        try {
            Long id = cancelPreLoanRepository.cancelPreLoan(orderMapping.getApplyNo());
            String response = opCenter.getToken(auth);
            JSONObject responseJ = JSONObject.parseObject(response);
            if (responseJ != null && responseJ.containsKey(Constants.DATA)) {
                JSONObject data = (JSONObject) responseJ.get(Constants.DATA);
                if (data != null && data.containsKey(Constants.TOKEN)) {
                    token = String.valueOf(data.get(Constants.TOKEN));
                }
            }

            HashMap<Object, Object> data = Maps.newHashMap();
            Long loanId = orderMapping.getLoanId();
            if (Objects.nonNull(loanId)) {
                data.put("loanIds", loanId);
                result = clothoCenterService.cancelPreLoanHasLoanId(data, token);

            } else {
                data.put("data", id);
                data.put("remark", "贷前关单");
                result = clothoCenterService.cancelPreLoan(data, token);
            }
            JSONObject resp = JSONObject.parseObject(result);
            if (Objects.isNull(resp) || Objects.isNull(resp.get(Constants.SUCCESS)) || !Constants.TRUE.equals(String.valueOf(resp.get(Constants.SUCCESS)))) {
                cancel = false;
            }
        } catch (Exception e) {
            cancel = false;
            log.error(e.getMessage(), e);
        }
        return new Tuple<>(cancel, cancel ? "贷前关单成功" : "贷前关单失败");
    }

    /**
     * 贷后提现超时关单
     *
     * @param
     * @return
     */
    @OperationAnno(channelNo = "#this[0]", opt = OptEnumName.LOAN_CANCEL, succSPEL = "#this.key", optDetailSPEL = "#this.value")
    @Override
    public Tuple<Boolean, String> cancelAfterLoan(String channelOrderNumber) {
        return huijinsuoCloseOrder72(channelOrderNumber);
    }

    /**
     * 惠金所 72 小时提现超时
     *
     * @param channelOrderNumber
     * @return
     */
    private Tuple<Boolean, String> huijinsuoCloseOrder72(String channelOrderNumber) {
        ClfOrderMapping orderMapping = this.clfOrderMappingRepository.findByChannelOrderNoLastOne(channelOrderNumber);

        if (orderMapping == null) {
            log.info("cancelAfterLoan,贷后关单失败，无订单  ={}", channelOrderNumber);
            return new Tuple<>(false, "贷后关单失败，无订单");
        }
        if (orderMapping.getLoanId() == null) {
            log.info("cancelAfterLoan,贷后关单失败，loanId为空，channelOrderNumber={}", channelOrderNumber);
            return new Tuple<>(false, "贷后关单失败，loanId为空");
        }
        ConcurrentMap<Object, Object> data = Maps.newConcurrentMap();
        data.put("loanId", orderMapping.getLoanId());
        data.put("userId", orderMapping.getQgUserId());
        data.put("flowNo", UUID.randomUUID());
        String id = clothoCenterService.cancelAfterLoan(data);

        return new Tuple<>(true, "贷后关单成功");
    }

    /**
     * 查询订单历史推送记录
     *
     * @param channelOrderNumber
     * @return
     */
    @Override
    public List<CallbackRecordVoModel> getOrderCallbackRecordList(String channelOrderNumber) {

        List<CallbackRecordVoModel> callbackRecordVoModelList = new ArrayList<>();

        ClfOrderMapping orderMapping = clfOrderMappingRepository.findByChannelOrderNoLastOne(channelOrderNumber);
        if (orderMapping == null) {
            return null;
        }

        List<CallbackRecord> callbackRecordList = clfCenterService.findCallbackRecordByApplyNo(orderMapping.getApplyNo());
        if (CollectionUtils.isNotEmpty(callbackRecordList)) {
            for (CallbackRecord callbackRecord : callbackRecordList) {
                CallbackRecordVoModel callbackRecordVoModel = new CallbackRecordVoModel();
                callbackRecordVoModel.setChannelOrderNo(orderMapping.getChannelOrderNo());
                callbackRecordVoModel.setApplyNo(orderMapping.getApplyNo());
                callbackRecordVoModel.setLoanId(orderMapping.getLoanId());
                callbackRecordVoModel.setTermNo(callbackRecord.getTermNo());

                Tuple<String, String> stringStringTuple = OrderUtil.KaNoticeTypeNameConvertChnName(callbackRecord.getNoticeType());
                callbackRecordVoModel.setNoticeType(!"未知状态".equals(stringStringTuple.getValue()) ? stringStringTuple.getValue() : callbackRecord.getNoticeType().getDesc());

                callbackRecordVoModel.setCallbackStatus(callbackRecord.getCallbackStatus());
                callbackRecordVoModel.setChannelId(callbackRecord.getChannelId());
                callbackRecordVoModel.setCreatedAt(callbackRecord.getCreatedAt());
                callbackRecordVoModel.setUpdatedAt(callbackRecord.getUpdatedAt());
                callbackRecordVoModel.setTableName(Constants.SUCCESS);
                callbackRecordVoModelList.add(callbackRecordVoModel);
            }
        }

        List<CallbackFailRecord> callbackFailRecordList = clfCenterService.findCallbackFailRecordByApplyNo(orderMapping.getApplyNo());
        if (CollectionUtils.isNotEmpty(callbackFailRecordList)) {
            for (CallbackFailRecord callbackFailRecord : callbackFailRecordList) {
                CallbackRecordVoModel callbackFailRecordVoModel = new CallbackRecordVoModel();
                callbackFailRecordVoModel.setChannelOrderNo(orderMapping.getChannelOrderNo());
                callbackFailRecordVoModel.setApplyNo(orderMapping.getApplyNo());
                callbackFailRecordVoModel.setLoanId(orderMapping.getLoanId());
                callbackFailRecordVoModel.setTermNo(callbackFailRecord.getTermNo());

                Tuple<String, String> stringStringTuple = OrderUtil.KaNoticeTypeNameConvertChnName(callbackFailRecord.getNoticeType());
                callbackFailRecordVoModel.setNoticeType(!"未知状态".equals(stringStringTuple.getValue()) ? stringStringTuple.getValue() : callbackFailRecord.getNoticeType().getDesc());

                callbackFailRecordVoModel.setCallbackStatus(callbackFailRecord.getCallbackStatus());
                callbackFailRecordVoModel.setChannelId(callbackFailRecord.getChannelId());
                callbackFailRecordVoModel.setRetryTimes(callbackFailRecord.getRetryTimes());
                callbackFailRecordVoModel.setFailCode(callbackFailRecord.getFailCode());
                callbackFailRecordVoModel.setCallbackRouter(callbackFailRecord.getCallbackRouter());
                callbackFailRecordVoModel.setIsActive(callbackFailRecord.getIsActive());
                callbackFailRecordVoModel.setExdata(callbackFailRecord.getExdata());
                callbackFailRecordVoModel.setCreatedAt(callbackFailRecord.getCreatedAt());
                callbackFailRecordVoModel.setUpdatedAt(callbackFailRecord.getUpdatedAt());
                callbackFailRecordVoModel.setTableName("Fail");
                callbackRecordVoModelList.add(callbackFailRecordVoModel);
            }
        }
        return callbackRecordVoModelList;
    }

    /**
     * 查询还款计划
     *
     * @param channelOrderNumber
     * @param loanId
     * @return
     */
    @Override
    public RepaymentPlansResultModel findRepaymentPlan(String channelOrderNumber, Long loanId) {
        log.info("[findRepaymentPlan] 查询还款计划 channelOrderNumber={},loanId={}", channelOrderNumber, loanId);
        if (loanId == null || loanId < 1) {
            ClfOrderMapping orderMapping = clfCenterService.findOrderMappingByChannelOrderNo(channelOrderNumber);
            if (orderMapping == null || orderMapping.getLoanId() == null) {
                log.warn("[findRepaymentPlan] 订单信息不存在或没有loanId");
                return null;
            }
            loanId = orderMapping.getLoanId();
        }

        ServiceResult<RepaymentPlansResultModel> serviceResult = xyqbCenterService.findRepaymentPlan(loanId);
        log.info("[findRepaymentPlan] 查询还款计划，返回信息 channelOrderNumber={},loanId={},serviceResult={}", channelOrderNumber, loanId, serviceResult);
        if (Objects.isNull(serviceResult) || !serviceResult.isSuccess()) {
            log.error("[findRepaymentPlan] 查询还款计划异常，返回页面空list");
            return null;
        }

        RepaymentPlansResultModel repaymentPlansVo = serviceResult.getData();
        if(Objects.nonNull(repaymentPlansVo)){
            repaymentPlansVo.setLoanId(loanId);
        }
        log.info("[findRepaymentPlan] 查询还款计划，返回信息 channelOrderNumber={},loanId={},repaymentPlansVo={}", channelOrderNumber, loanId, repaymentPlansVo);
        return repaymentPlansVo;
    }

    @Override
    public void loadSecondAuditJob() {
        jolyneService.reloadJob("cn.qg.clotho.job.FetchDataLoanJob");
        jolyneService.reloadJob("cn.qg.clotho.job.LoanDataJob");
    }

    @OperationAnno(channelNo = "#this[0]", opt = OptEnumName.ALL_REPAID, succSPEL = "#this.key", optDetailSPEL = "#this.value")
    @Override
    public Tuple<Boolean, String> settle(String channelOrderNumber) {

        ClfOrderMapping orderMapping = clfCenterService.findOrderMappingByChannelOrderNo(channelOrderNumber);
        if (orderMapping != null && StringUtils.isNotEmpty(orderMapping.getApplyNo())) {
            ServiceResult<XyqbCurrentOrderStatusServiceResultModel> xyqbOrderStatus = xyqbCenterService.getXyqbOrderStatus(orderMapping.getApplyNo(), orderMapping.getLoanId());
            if (xyqbOrderStatus.isSuccess() && xyqbOrderStatus.getData() != null) {
                if (KANoticeType.FUND_WAITING_WITHDRAW.equals(xyqbOrderStatus.getData().getCurrentStatus().getKaNoticeType())) {
                    // 如果是 存管待提现状态
                    return huijinsuoCloseOrder72(channelOrderNumber);
                }

            }
            // 普通 订单 关单(清楚用户活跃订单)
            return xyqbCenterService.cleanUserActiveOrder(orderMapping.getQgUserId());
        } else {
            return new Tuple<>(false, "订单不存在");
        }
    }


    /**
     * 查询订单状态
     *
     * @param currentStatus
     * @param orderMapping
     * @return
     */
    private Tuple<String, List<OrderVo.OptButton>> getCurrentStatusAndButtons(QueryXyqbOrderStatus currentStatus, ClfOrderMapping orderMapping) {
        Tuple tuple = new Tuple();
        List<OrderVo.OptButton> buttonList = new ArrayList<>();
        if (currentStatus != null) {
            if (StringUtils.isNotEmpty(currentStatus.getStatus())) {
                OrderVo.OptButton buttonBeforeCancel = new OrderVo.OptButton();
                buttonBeforeCancel.setAction(OrderVo.OptButtonAction.cancel.name());
                buttonBeforeCancel.setName(OrderVo.OptButtonAction.cancel.getDesc());
                switch (currentStatus.getStatus()) {
                    case "CREDIT_SUCC":
                        return getCreditSuccessTuple(tuple, buttonList);
                    case "APPROVE_ING":
                        return getApprovingTuple(orderMapping, tuple, buttonList);
                    case "REJECT":
                        tuple.setKey("审批拒绝");
                        tuple.setValue(buttonList);
                        return tuple;
                    case "FUAD_ASSIFN_SUCC":
                        tuple.setKey("审批通过，待提现");
                        buttonList.add(buttonBeforeCancel);
                        tuple.setValue(buttonList);
                        return tuple;
                    case "WITHDRAW_APPROVING":
                        return getWithdrawApprovingTuple(tuple, buttonList, buttonBeforeCancel);
                    case "WITHDRAW":
                    case "PAY_ING":
                        return getWithdrawTuple(orderMapping, tuple, buttonList, buttonBeforeCancel);
                    case "FUND_WAITING_WITHDRAW":
                        tuple.setKey("放款到存管账户，待存管提现");
                        OrderVo.OptButton cancelAfterButton0 = new OrderVo.OptButton();
                        cancelAfterButton0.setAction(OrderVo.OptButtonAction.cancel_after.name());
                        cancelAfterButton0.setName(OrderVo.OptButtonAction.cancel_after.getDesc());
                        buttonList.add(cancelAfterButton0);
                        tuple.setValue(buttonList);
                        return tuple;
                    case "FUND_SUCC":
                        tuple.setKey("放款成功");
                        return tuple;
                    case "FUND_WITHDRAW_SUCC":
                        tuple.setKey("存管提现成功");
                        tuple.setValue(buttonList);
                        return tuple;
                    case "FUND_FAIL":
                        tuple.setKey("放款失败");
                        tuple.setValue(buttonList);
                        return tuple;
                    case "ALL_REPAID":
                        tuple.setKey("已结清");
                        tuple.setValue(buttonList);
                        return tuple;
                    case "CANCEL_LOAN":
                        tuple.setKey("已关单");
                        tuple.setValue(buttonList);
                        return tuple;
                    case "CANCEL_AFTER_LOAN":
                        tuple.setKey("存管提现超时或贷后关单");
                        tuple.setValue(buttonList);
                        return tuple;
                    default:
                        tuple.setKey("未知状态");
                        tuple.setValue(buttonList);
                        return tuple;
                }
            }
        } else {
            tuple.setKey("未知状态");
            tuple.setValue(buttonList);
            return tuple;
        }
        tuple.setKey("未知状态");
        tuple.setValue(buttonList);
        return tuple;
    }

    private Tuple<String, List<OrderVo.OptButton>> getWithdrawTuple(ClfOrderMapping orderMapping, Tuple tuple, List<OrderVo.OptButton> buttonList, OrderVo.OptButton buttonBeforeCancel) {
        LoanApplicationHistory loanApplicationHistory = xyqbCenterService.findLoanApplicationHistoryById(orderMapping.getLoanId());
        if (loanApplicationHistory != null) {
            if (LoanProgress.WAITING_FUND.equals(loanApplicationHistory.getProgress())) {
                // 更改放款时间,二次风控回调
                // 提现申请成功 需要放款
                tuple.setKey("二次风控审批中");
            } else if (LoanProgress.FINAL_SENDED_TO_FUNDING_CORP.equals(loanApplicationHistory.getProgress())) {
                tuple.setKey("待放款");
                OrderVo.OptButton buttonPaySucc = new OrderVo.OptButton();
                buttonPaySucc.setAction(OrderVo.OptButtonAction.pay_succ.name());
                buttonPaySucc.setName(OrderVo.OptButtonAction.pay_succ.getDesc());
                buttonList.add(buttonPaySucc);
                OrderApprove orderApprove = orderApproveRepository.findByCreditNumber(orderMapping.getApplyNo());
                if (orderApprove != null && orderApprove.getFundType() == 0) {
                    // 非存管 才有放款失败按钮
                    OrderVo.OptButton buttonPayFail = new OrderVo.OptButton();
                    buttonPayFail.setAction(OrderVo.OptButtonAction.pay_fail.name());
                    buttonPayFail.setName(OrderVo.OptButtonAction.pay_fail.getDesc());
                    buttonList.add(buttonPayFail);
                }
                buttonList.add(buttonBeforeCancel);
            }
            tuple.setValue(buttonList);
        }
        return tuple;
    }

    private Tuple<String, List<OrderVo.OptButton>> getWithdrawApprovingTuple(Tuple tuple, List<OrderVo.OptButton> buttonList, OrderVo.OptButton buttonBeforeCancel) {
        tuple.setKey("提现成功，待二次风控审批");
        // 更改放款时间,二次风控回调
        OrderVo.OptButton secondAudit2 = new OrderVo.OptButton();
        secondAudit2.setAction(OrderVo.OptButtonAction.second_audit.name());
        secondAudit2.setName(OrderVo.OptButtonAction.second_audit.getDesc());
        buttonList.add(secondAudit2);
        buttonList.add(buttonBeforeCancel);
        tuple.setValue(buttonList);
        return tuple;
    }

    private Tuple<String, List<OrderVo.OptButton>> getApprovingTuple(ClfOrderMapping orderMapping, Tuple tuple, List<OrderVo.OptButton> buttonList) {
        OptHistoryLog optHistoryLog = optHistoryLogService.findByChannelOrderNumberAndOptNameAndSuccess(orderMapping.getChannelOrderNo(), OptEnumName.AUDIT);
        if (optHistoryLog != null) {
            tuple.setKey("审批中");
            tuple.setValue(buttonList);
        } else {
            OrderVo.OptButton buttonApprove2 = new OrderVo.OptButton();
            buttonApprove2.setAction(OrderVo.OptButtonAction.audit.name());
            buttonApprove2.setName(OrderVo.OptButtonAction.audit.getDesc());
            buttonList.add(buttonApprove2);
            tuple.setKey("授信成功,待审批");
            tuple.setValue(buttonList);
        }
        return tuple;
    }

    private Tuple<String, List<OrderVo.OptButton>> getCreditSuccessTuple(Tuple tuple, List<OrderVo.OptButton> buttonList) {
        // 授信成功 需要 审批
        OrderVo.OptButton buttonApprove = new OrderVo.OptButton();
        buttonApprove.setAction(OrderVo.OptButtonAction.audit.name());
        buttonApprove.setName(OrderVo.OptButtonAction.audit.getDesc());
        buttonList.add(buttonApprove);
        tuple.setKey("授信成功");
        tuple.setValue(buttonList);
        return tuple;
    }


    private Calendar getCalendar() {
        // 额度有效期
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());
        calendar.add(Calendar.DAY_OF_YEAR, 7);
        return calendar;
    }

    private Map<String, Object> getApproveObjectMap(ApproveVo approveVo, ClfOrderMapping orderMapping, XUser xUser, String fundFormat, Integer bizType) {
        Map<String, Object> data = new HashMap<>(16);
        data.put("code", 0);
        data.put("msg", "success");
        data.put("bizChannel", orderMapping.getRegisteredFrom());
        data.put("uuid", xUser.getUuid());
        data.put("bizNo", orderMapping.getApplyNo());

        data.put("bizType", bizType);

        data.put("auditResult", approveVo.getIsPass());
        data.put("amount", approveVo.getAmount());
        data.put("deadLine", getCalendar().getTime().getTime());
        data.put("financeProducts", fundFormat);
        return data;
    }

    private void saveOrUpdateOrderApprove(ApproveVo approveVo, ClfOrderMapping orderMapping, Integer fundId) {
        OrderApprove orderApproveExsit = orderApproveRepository.findByChannelOrderNumber(approveVo.getChannelOrderNumber());
        if (orderApproveExsit != null) {
            orderApproveExsit.setFundId(fundId);
            orderApproveExsit.setFundType(approveVo.getFundType());
            orderApproveExsit.setUpdateTime(new Date());
            orderApproveRepository.save(orderApproveExsit);
        } else {
            OrderApprove orderApprove = new OrderApprove();
            orderApprove.setChannelOrderNumber(approveVo.getChannelOrderNumber());
            orderApprove.setCreditNumber(orderMapping.getApplyNo());
            orderApprove.setChannelId(orderMapping.getRegisteredFrom());
            orderApprove.setFundId(fundId);
            orderApprove.setFundType(approveVo.getFundType());
            orderApprove.setCreateTime(new Date());
            orderApprove.setUpdateTime(new Date());
            orderApproveRepository.save(orderApprove);
        }
    }


    /**
     *
     * @param applyNo
     * @return
     */
    private BigDecimal getFundRate(String applyNo) {
        log.info("getFundRate 通过授信订单号获取动态年利率 applyNo={}", applyNo);
        ApplyScenarioInfoEntity applyScenarioInfoEntity = xyqbCenterService.findFirstByApplyNo(applyNo);
        if (Objects.isNull(applyScenarioInfoEntity)) {
            log.info("getFundRate applyNo={} 没有动态年利率", applyNo);
            return null;
        }

        String scenarioData = applyScenarioInfoEntity.getScenarioData();
        if (StringUtils.isBlank(scenarioData)) {
            log.error("getFundRate  场景数据为空 applyScenarioInfo={}", applyScenarioInfoEntity);
            return null;
        }
        JSONObject jsonObject = JSONObject.parseObject(scenarioData);
        BigDecimal rate = jsonObject.getBigDecimal("rate");
        if (Objects.isNull(rate)) {
            log.error("getFundRate  获取年利率为空 applyScenarioInfo={}", applyScenarioInfoEntity);
            return null;
        }


        BigDecimal result = rate.divide(new BigDecimal("100"));

        DecimalFormat format6 = new DecimalFormat("#####0.000000");
        String format = format6.format(result);
        return new BigDecimal(format);


    }

}
