package api_server

import (
	"fmt"
	"git.quantgroup.cn/DevOps/enoch/pkg/global"
	"git.quantgroup.cn/DevOps/enoch/pkg/glog"
	"github.com/buaazp/fasthttprouter"
	"github.com/valyala/fasthttp"
	"github.com/vrg0/go-common/logger"
	"github.com/vrg0/go-common/util"
	"runtime"
	"strconv"
	"strings"
	"sync"
	"time"
)

var (
	router = fasthttprouter.New()
	once   = new(sync.Once)
)

const (
	GET     = "GET"
	PUT     = "PUT"
	DELETE  = "DELETE"
	POST    = "POST"
	PATCH   = "PATCH"
	HEAD    = "HEAD"
	OPTIONS = "OPTIONS"
)

//跨域
func wrap(handler fasthttp.RequestHandler) fasthttp.RequestHandler {
	return func(ctx *fasthttp.RequestCtx) {
		handler(ctx)
		ctx.Response.Header.Add("Access-Control-Allow-Origin", "*")
	}
}

//注册函数
//HandlerFunc只能在init()中执行
func HandlerFunc(method string, path string, handle fasthttp.RequestHandler) {
	if pc, _, _, ok := runtime.Caller(1); ok && strings.Contains(runtime.FuncForPC(pc).Name(), ".init") {
		router.Handle(method, path, handle)
	} else {
		glog.Error("HandlerFunc must be execute in init()")
	}
}

//程序对外API的http-server
//启动服务
func ListenAndServe() {
	once.Do(listenAndServe)
}

func listenAndServe() {
	//handlerFunc 发生 panic 后返回的内容
	router.PanicHandler = func(ctx *fasthttp.RequestCtx, i interface{}) {
		rtn := "panic: " + ctx.String() + " "
		switch x := i.(type) {
		case error:
			rtn += x.Error()
		case string:
			rtn += x
		default:
			rtn += "未识别的panic类型"
		}
		rtn += "\n"
		rtn += ctx.Request.String()

		glog.Error(rtn)
		ctx.Response.SetStatusCode(500)
		ctx.Response.SetBodyString(rtn)
	}

	//启动http server
	go func() {
		//server panic后自动重启
		defer func() {
			if err := recover(); err != nil {
				errStr := fmt.Sprint("api fasthttp panic:", err)
				glog.Error(errStr)
				time.Sleep(time.Second * 1)
				listenAndServe()
			}
		}()

		//创建http-server对象
		l := global.Logger.GetStandardLogger()
		l.SetPrefix("_API_SERVER_ ")
		hook := logger.NewHookWriter(l.Writer())
		hook.AddHookFunc(func(data []byte) bool {
			body := util.BytesString(data)
			if strings.Contains(body, "error") {
			}
			return true
		})
		l.SetOutput(hook)
		server := fasthttp.Server{
			Logger:  l,
			Handler: wrap(router.Handler),
		}

		//启动http-server
		firstRunFlag := true
		for {
			if firstRunFlag {
				firstRunFlag = false
			} else {
				time.Sleep(time.Second * 1) //二次重启延时1秒
			}

			if err := server.ListenAndServe("0.0.0.0:" + strconv.Itoa(global.HttpPort)); err != nil {
				glog.Error("api fasthttp err", err)
			}
		}
	}()
	//等待http-server启动成功
	time.Sleep(time.Millisecond * 100)
}
