package points

import (
	"fmt"
	"sync"
)

//服务A 的 X 接口 被 那些服务访问？ 那些节点访问
type PathNode struct {
	ServiceName      string
	Path             string
	RemoteServiceMap map[string]struct{}
}

func NewPathNode(path string, serviceName string) PathNode {
	return PathNode{
		Path:             path,
		ServiceName:      serviceName,
		RemoteServiceMap: make(map[string]struct{}),
	}
}

type ServiceNode struct {
	ServiceName string
	PathMap     map[string]PathNode
	IpMap       map[string]struct{}
}

func NewServiceNode(serviceName string) ServiceNode {
	return ServiceNode{
		ServiceName: serviceName,
		PathMap:     make(map[string]PathNode),
		IpMap:       make(map[string]struct{}),
	}
}

type Statistics struct {
	ServiceMap map[string]ServiceNode
	lock       *sync.Mutex
}

func (s *Statistics) String() string {
	s.lock.Lock()
	defer s.lock.Unlock()
	rtn := ""

	for _, service := range s.ServiceMap {
		rtn += fmt.Sprintf("%s: ", service.ServiceName)
		for ip := range service.IpMap {
			rtn += ip + "  "
		}
		rtn += "\n"

		for _, path := range service.PathMap {
			rtn += path.Path + "  "
			for remoteService := range path.RemoteServiceMap {
				rtn += remoteService + "  "
			}
			rtn += "\n"
		}

		rtn += "\n"
	}

	return rtn
}

func NewStatistics() *Statistics {
	return &Statistics{
		ServiceMap: make(map[string]ServiceNode),
		lock:       new(sync.Mutex),
	}
}

func (s *Statistics) addServer(local localEndpoint, remote remoteEndpoint, tag tags) {
	if _, ok := s.ServiceMap[local.ServiceName]; !ok {
		s.ServiceMap[local.ServiceName] = NewServiceNode(local.ServiceName)
	}

	if _, ok := s.ServiceMap[local.ServiceName].IpMap[local.Ipv4]; !ok {
		s.ServiceMap[local.ServiceName].IpMap[local.Ipv4] = struct{}{}
	}

	path := tag.HttpMethod + "-" + tag.HttpPath
	_, ok := s.ServiceMap[local.ServiceName].PathMap[path]
	if !ok {
		s.ServiceMap[local.ServiceName].PathMap[path] = NewPathNode(path, local.ServiceName)
	}
}

func (s *Statistics) addClient(local localEndpoint, remote remoteEndpoint, tag tags) {
	for _, service := range s.ServiceMap {
		for ip := range service.IpMap {
			if ip == remote.Ipv4 {
				//添加服务
				path := tag.HttpMethod + "-" + tag.HttpPath
				if _, ok := service.PathMap[path]; !ok {
					service.PathMap[path] = NewPathNode(path, service.ServiceName)
				}

				//service.PathMap[path].
				service.PathMap[path].RemoteServiceMap[local.ServiceName] = struct{}{}
			}
		}
	}
}

func (s *Statistics) AddNode(point TraceMsg) {
	s.lock.Lock()
	defer s.lock.Unlock()

	switch point.Kind {
	case "SERVER":
		s.addServer(point.LocalEndpoint, point.RemoteEndpoint, point.Tags)
	case "CLIENT":
		s.addClient(point.LocalEndpoint, point.RemoteEndpoint, point.Tags)
	default:
		//位置类型，不处理
	}
}

/*
type PathNode struct {
	ServiceName    string
	Path           string
	SrcServiceList map[string]struct{}
	SrcNodeIp      map[string]struct{}
}

type Statistics struct {
	PathMap map[string]PathNode
	lock    *sync.Mutex
}

func NewStatistics() *Statistics {
	return &Statistics{
		PathMap: make(map[string]PathNode),
		lock:    new(sync.Mutex),
	}
}

*/
