package report_form

import (
	"fmt"
	"git.quantgroup.cn/DevOps/enoch/pkg/glog"
	"sort"
	"strings"
	"time"
)

const (
	MinCount = 64
)

type Service struct {
	name      string    //服务名称
	startTime time.Time //开始时间
	endTime   time.Time //结束时间
	//path相关
	pathMap               map[string]*Path //path列表
	sumDuration           time.Duration    //总响应时间
	count                 int              //总访问量
	averageDuration       time.Duration    //平均响应时间
	maxMedianPath         *Path            //最大中位响应时间path
	maxAveragePath        *Path            //最大平均响应时间path
	maxDurationTracePoint *TracePoint      //最大响应时间的tracePoint
	qps                   int
	//node相关
	nodeMap map[string]*Node //节点列表
}

var IgnorePathMap = map[string]struct{}{
	"get":                {},
	"post":               {},
	"put":                {},
	"delete":             {},
	"patch":              {},
	"head":               {},
	"option":             {},
	"/tech/health/check": {},
}

func NewService(name string, startTime time.Time, endTime time.Time) *Service {
	rtn := Service{
		name:      name,
		startTime: startTime,
		endTime:   endTime,
		pathMap:   make(map[string]*Path),
		nodeMap:   make(map[string]*Node),
		qps:       0,
	}

	//初始化path
	pathList := rtn.getPathList()
	for _, path := range pathList {
		//屏蔽掉健康检查url，屏蔽掉命名不合法的url
		pathSplit := strings.Split(path, " ")
		if len(pathSplit) != 2 {
			continue
		}
		if _, ok := IgnorePathMap[strings.ToLower(pathSplit[1])]; ok {
			continue
		}
		glog.Info("init path: ", rtn.name, " ", path)
		pathObj, ok := NewPath(rtn.name, path, rtn.startTime, rtn.endTime)
		if !ok {
			continue
		}
		rtn.pathMap[path] = pathObj
	}

	//初始化访问时间相关的参数
	rtn.initDuration()
	//初始化qps
	rtn.initQps()

	//初始化node
	nodeList := rtn.getNodeList()
	for _, node := range nodeList {
		glog.Info("init node: ", rtn.name, " ", node)
		rtn.nodeMap[node] = NewNode(rtn.name, node, rtn.startTime, rtn.endTime)
	}

	return &rtn
}

//初始化qps
func (s *Service) initQps() {
	for _, p := range s.pathMap {
		s.qps += p.GetQps()
	}
}

//获取服务名称
func (s *Service) Name() string {
	return s.name
}

//获取节点列表
func (s *Service) getNodeList() []string {
	const sqlGetNodeList = `SHOW TAG VALUES FROM machine_info WITH key = "host" WHERE sys_name = '%s';`
	sql := fmt.Sprintf(sqlGetNodeList, s.name)
	resp, err := query(sql)
	if err != nil {
		glog.Error("query sql:", sql, err)
		return []string{}
	}

	if len(resp.Results) != 1 {
		return []string{}
	}
	res := resp.Results[0]
	if len(res.Series) != 1 {
		return []string{}
	}
	row := res.Series[0]
	idx, ok := getModRowKeyIdx("value", row)
	if !ok {
		return []string{}
	}

	rtn := make([]string, 0)
	for _, v := range row.Values {
		pathInterface := v[idx]
		path, ok := pathInterface.(string)
		if !ok {
			continue
		}
		rtn = append(rtn, path)
	}

	return rtn
}

func (s *Service) getPathList() []string {
	const sqlGetPathList = `SHOW TAG VALUES FROM trace_info WITH key = "path" WHERE sys_name = '%s';`
	sql := fmt.Sprintf(sqlGetPathList, s.name)
	resp, err := query(sql)
	if err != nil {
		glog.Error("query sql:", sql, err)
		return []string{}
	}

	if len(resp.Results) != 1 {
		return []string{}
	}
	res := resp.Results[0]
	if len(res.Series) != 1 {
		return []string{}
	}
	row := res.Series[0]
	idx, ok := getModRowKeyIdx("value", row)
	if !ok {
		return []string{}
	}

	rtn := make([]string, 0)
	for _, v := range row.Values {
		pathInterface := v[idx]
		path, ok := pathInterface.(string)
		if !ok {
			continue
		}
		rtn = append(rtn, path)
	}

	return rtn
}

//初始化访问时间
func (s *Service) initDuration() {
	sum := time.Duration(0)
	count := 0
	for _, pathObj := range s.pathMap {
		sum += pathObj.GetSumDuration()
		count += pathObj.GetCount()
	}
	s.count = count
	s.sumDuration = sum

	//平均
	if s.count == 0 {
		return
	}
	s.averageDuration = s.sumDuration / time.Duration(s.count)

	medianMax := time.Duration(-99999)
	averageMax := time.Duration(-99999)
	maxDuration := time.Duration(-99999)

	for _, pathObj := range s.pathMap {
		if pathObj.GetCount() > MinCount && pathObj.GetMedianDuration() > medianMax {
			s.maxMedianPath = pathObj
			medianMax = pathObj.GetMedianDuration()
		}
		if pathObj.GetCount() > MinCount && pathObj.GetAverageDuration() > averageMax {
			s.maxAveragePath = pathObj
			averageMax = pathObj.GetAverageDuration()
		}

		point := pathObj.GetMaxDurationTracePoint()
		if point.Duration > maxDuration {
			s.maxDurationTracePoint = &point
			maxDuration = point.Duration
		}
	}
}

//获取 MaxDuration Path 排行
func (s *Service) GetAverageDurationPathList() []*Path {
	rtn := make([]*Path, 0)

	for _, path := range s.pathMap {
		rtn = append(rtn, path)
	}

	sort.Slice(rtn, func(i, j int) bool {
		if rtn[i].GetAverageDuration() > rtn[j].GetAverageDuration() {
			return true
		}
		return false
	})

	return rtn
}

//获取qps
func (s *Service) GetQps() int {
	return s.qps
}

//获取 MedianDuration Path 排行
func (s *Service) GetMedianDurationPathList() []*Path {
	rtn := make([]*Path, 0)

	for _, path := range s.pathMap {
		rtn = append(rtn, path)
	}

	sort.Slice(rtn, func(i, j int) bool {
		if rtn[i].GetMedianDuration() > rtn[j].GetMedianDuration() {
			return true
		}
		return false
	})

	return rtn
}

//获取 MaxDuration Path 排行
func (s *Service) GetMaxDurationPathList() []*Path {
	rtn := make([]*Path, 0)

	for _, path := range s.pathMap {
		rtn = append(rtn, path)
	}

	sort.Slice(rtn, func(i, j int) bool {
		if rtn[i].GetMaxDurationTracePoint().Duration > rtn[j].GetMaxDurationTracePoint().Duration {
			return true
		}
		return false
	})

	return rtn
}

//获取总访问时间
func (s *Service) GetSumDuration() time.Duration {
	return s.sumDuration
}

//获取总访问量
func (s *Service) GetCount() int {
	return s.count
}

//获取平均访问时间
func (s *Service) GetAverageDuration() time.Duration {
	return s.averageDuration
}

//获取中位访问时间最大的path
func (s *Service) GetMaxMedianDurationPath() (*Path, bool) {
	return s.maxMedianPath, s.maxMedianPath != nil
}

//获取平均访问时间最大的path
func (s *Service) GetMaxAverageDurationPath() (*Path, bool) {
	return s.maxAveragePath, s.maxAveragePath != nil
}

//获取最大访问时间最大的Point
func (s *Service) GetMaxDurationTracePoint() (*TracePoint, bool) {
	return s.maxDurationTracePoint, s.maxDurationTracePoint != nil
}

//求平均值
func (s *Service) getNodeMapValue(f func(node *Node) int) int {
	if len(s.nodeMap) == 0 {
		return 0
	}

	sum := 0
	for _, node := range s.nodeMap {
		sum += f(node)
	}

	return sum / len(s.nodeMap)
}

func (s *Service) GetAverageCpu() int {
	return s.getNodeMapValue(func(node *Node) int {
		return node.GetAverageCpu()
	})
}

func (s *Service) GetRemoveN100MaxCpu() int {
	return s.getNodeMapValue(func(node *Node) int {
		return node.GetRemoveN100MaxCpu()
	})
}

func (s *Service) GetMaxCpu() int {
	return s.getNodeMapValue(func(node *Node) int {
		return node.GetMaxCpu()
	})
}

func (s *Service) GetAverageMem() int {
	return s.getNodeMapValue(func(node *Node) int {
		return node.GetAverageMem()
	})
}

func (s *Service) GetMaxMem() int {
	return s.getNodeMapValue(func(node *Node) int {
		return node.GetMaxMem()
	})
}

func (s *Service) GetAverageDisk() int {
	return s.getNodeMapValue(func(node *Node) int {
		return node.GetAverageDisk()
	})
}

func (s *Service) GetMaxDisk() int {
	return s.getNodeMapValue(func(node *Node) int {
		return node.GetMaxDisk()
	})
}

func (s *Service) GetAverageThread() int {
	return s.getNodeMapValue(func(node *Node) int {
		return node.GetAverageThread()
	})
}

func (s *Service) GetMaxThread() int {
	return s.getNodeMapValue(func(node *Node) int {
		return node.GetMaxThread()
	})
}
