package report_form

import (
	"fmt"
	"git.quantgroup.cn/DevOps/enoch/pkg/glog"
	"time"
)

type Node struct {
	serviceName string
	address     string
	startTime   time.Time
	endTime     time.Time
	//cpu
	averageCpu int
	maxCpu     int
	//mem
	averageMem int
	maxMem     int
	//disk
	averageDisk int
	maxDisk     int
	//thread
	averageThread int
	maxThread     int
}

func NewNode(serviceName string, address string, startTime time.Time, endTime time.Time) *Node {
	rtn := &Node{
		serviceName: serviceName,
		address:     address,
		startTime:   startTime,
		endTime:     endTime,
	}

	rtn.initCpu()
	rtn.initMem()
	rtn.initDisk()
	rtn.initThread()

	return rtn
}

//依次返回：平均、最大
//如果获取失败返回0，0，0
func (n *Node) queryKey(key string) (int, int) {
	//注：这里使用了子查询，InfluxDB目前不支持函数参数为表达式，如：sum("key"*10)
	sqlFormat := `SELECT mean("x"), max("x") ` +
		`FROM (` +
		`SELECT %s AS "x" FROM machine_info ` +
		`WHERE sys_name = '%s' AND host = '%s'AND time >= %d AND time < %d` +
		`);`

	sql := fmt.Sprintf(sqlFormat, key, n.serviceName, n.address, n.startTime.UnixNano(), n.endTime.UnixNano())
	resp, err := query(sql)
	if err != nil {
		glog.Error("node query key:", err)
	}
	if len(resp.Results) != 1 {
		return 0, 0
	}
	res := resp.Results[0]
	if len(res.Series) != 1 {
		return 0, 0
	}
	row := res.Series[0]
	if len(row.Values) != 1 {
		return 0, 0
	}
	value := row.Values[0]

	meanIdx, ok2 := getModRowKeyIdx("mean", row)
	maxIdx, ok3 := getModRowKeyIdx("max", row)
	if !ok2 || !ok3 {
		return 0, 0
	}

	mean, ok2 := jsonNumberToInt(value[meanIdx])
	max, ok3 := jsonNumberToInt(value[maxIdx])
	if !ok2 || !ok3 {
		return 0, 0
	}

	return mean, max
}

func (n *Node) initCpu() {
	n.averageCpu, n.maxCpu = n.queryKey(`"processors"`)
}

func (n *Node) initMem() {
	n.averageMem, n.maxMem = n.queryKey(`(mem_tol-mem_free)/mem_tol*100`)
}

func (n *Node) initDisk() {
	n.averageDisk, n.maxDisk = n.queryKey(`(disk_tol-disk_free)/disk_tol*100`)
}

func (n *Node) initThread() {
	n.averageThread, n.maxThread = n.queryKey(`"thread_peak"`)
}

func (n *Node) GetServiceName() string {
	return n.serviceName
}

func (n *Node) GetAddress() string {
	return n.address
}

func (n *Node) GetStartTime() time.Time {
	return n.startTime
}

func (n *Node) GetEndTime() time.Time {
	return n.endTime
}

func (n *Node) GetAverageCpu() int {
	return n.averageCpu
}

func (n *Node) GetMaxCpu() int {
	return n.maxCpu
}

func (n *Node) GetAverageMem() int {
	return n.averageMem
}

func (n *Node) GetMaxMem() int {
	return n.maxMem
}

func (n *Node) GetAverageDisk() int {
	return n.averageDisk
}

func (n *Node) GetMaxDisk() int {
	return n.maxDisk
}

func (n *Node) GetAverageThread() int {
	return n.averageThread
}

func (n *Node) GetMaxThread() int {
	return n.maxThread
}
