package service_check

import (
	"encoding/json"
	"git.quantgroup.cn/DevOps/enoch/pkg/global"
	"git.quantgroup.cn/DevOps/enoch/pkg/glog"
	"github.com/influxdata/influxdb/client/v2"
)

//项
type Item struct {
	Sql       string
	ResultMap map[string]float64
}

func NewItem(sql string) *Item {
	return &Item{
		Sql:       sql,
		ResultMap: make(map[string]float64),
	}
}

func query(sql string) (*client.Response, error) {
	config := client.HTTPConfig{
		Addr: global.InfluxDbAddress,
	}

	connect, err := client.NewHTTPClient(config)
	defer func() { _ = connect.Close() }()
	if err != nil {
		return nil, err
	}

	rtn, err := connect.Query(client.NewQuery(sql, global.InfluxDbName, ""))
	if err != nil {
		return nil, err
	}

	return rtn, nil
}

func (i *Item) Run() {
	//清理
	i.Clean()

	//重新拉数据
	resp, err := query(i.Sql)
	if err != nil {
		glog.Error("查询失败：", i.Sql)
		return
	}
	for _, r := range resp.Results {
		for _, row := range r.Series {
			for _, v := range row.Values {
				id := i.tagsToId(row.Tags)
				if len(v) != 2 {
					i.ResultMap[id] = 0
					continue
				}
				j, ok := v[1].(json.Number)
				if !ok {
					continue
				}
				vFloat64, err := j.Float64()
				if err != nil {
					continue
				}
				i.ResultMap[id] = vFloat64
			}
		}
	}
}

//目前支持三种tag：sys_name path host
func (i *Item) tagsToId(tags map[string]string) string {
	tagList := []string{"sys_name", "path", "host"}
	rtn := ""
	for _, tag := range tagList {
		if v, ok := tags[tag]; ok {
			rtn += "-" + v
		}
	}
	rtn += "-"
	return rtn
}

func (i *Item) Clean() {
	i.ResultMap = make(map[string]float64)
}

//解释器
type Interpreter struct {
	ItemMap map[string]*Item
}

func NewInterpreter() *Interpreter {
	return &Interpreter{ItemMap: map[string]*Item{
		"count":                NewItem("select count(traceId) as count from trace_info where time>now()-7m and time<now()-2m group by sys_name fill(0);"),
		"yesterday_count":      NewItem("select count(traceId) as yesterday_count from trace_info where time>now()-1d7m and time<now()-1d2m group by sys_name fill(0);"),
		"path_count":           NewItem("select count(traceId) as path_count from trace_info where time>now()-7m and time<now()-2m group by sys_name, path fill(0);"),
		"yesterday_path_count": NewItem("select count(traceId) as yesterday_path_count from trace_info where time>now()-1d7m and time<now()-1d2m group by sys_name, path fill(0);"),
		"apdex":                NewItem("select (sum(sat)+sum(tol))/sum(ct_all) as apdex from apdex where time > now()-7m and time < now()-2m group by sys_name fill(0);"),
		"cpu":                  NewItem("select last(system_load_average)/last(processors) from machine_info where time > now() - 5m group by sys_name, host fill(0);"),
		"mem":                  NewItem("select last(mem_free)/last(mem_tol) from machine_info where time > now() - 5m group by sys_name,host fill(0);"),
		"disk":                 NewItem("select last(disk_free)/last(disk_tol) from machine_info where time > now() - 5m group by sys_name,host fill(0);"),
	}}
}

//刷新item中的数据
func (i *Interpreter) FlashData() {
	//加载数据
	for _, v := range i.ItemMap {
		v.Run()
	}
}

//解析一条表达式
// (((x1>=10)&&(x2<=20))||(x3=0))
// ()                   //调整优先级
// > < = >= <= && ||    //运算符
// abc                  //变量，对应itemMap中sql的执行结果
// 1.11                 //数值，统一为float64类型

//
//true
func (i *Interpreter) Explain(e string) {
	//i.ItemMap["123"]
}
