package report_form

import (
	"fmt"
	"git.quantgroup.cn/DevOps/enoch/pkg/glog"
	"sort"
	"time"
)

type ServiceMap struct {
	startTime  time.Time
	endTime    time.Time
	serviceMap map[string]*Service
}

func NewServiceMap(startTime time.Time, endTime time.Time, serviceList []string) *ServiceMap {
	rtn := &ServiceMap{
		startTime:  startTime,
		endTime:    endTime,
		serviceMap: make(map[string]*Service),
	}

	//初始化service
	//	serviceList := rtn.getServiceList()
	for _, serviceName := range serviceList {
		glog.Info("init service :", serviceName, " start")
		rtn.serviceMap[serviceName] = NewService(serviceName, startTime, endTime)
	}
	return rtn
}

func (sm *ServiceMap) getServiceList() []string {
	sql := `show tag values from trace_info with key = "sys_name";`
	resp, err := query(sql)
	if err != nil {
		return []string{}
	}
	if len(resp.Results) != 1 {
		return []string{}
	}
	res := resp.Results[0]
	if len(res.Series) != 1 {
		return []string{}
	}
	row := res.Series[0]
	idx, ok := getModRowKeyIdx("value", row)
	if !ok {
		return []string{}
	}
	rtn := make([]string, 0)
	for _, v := range row.Values {
		serviceName, ok := v[idx].(string)
		if !ok {
			continue
		}
		rtn = append(rtn, serviceName)
	}

	return rtn
}

//获取平均cpu使用率列表
func (sm *ServiceMap) GetAverageCpuServiceList() []*Service {
	rtn := make([]*Service, 0)
	for _, s := range sm.serviceMap {
		rtn = append(rtn, s)
	}

	sort.Slice(rtn, func(i, j int) bool {
		if rtn[i].GetAverageCpu() > rtn[j].GetAverageCpu() {
			return true
		}
		return false
	})

	return rtn
}

//去N100后的峰值cpu使用率列表
func (sm *ServiceMap) GetRemoveN100MaxCpuServiceList() []*Service {
	rtn := make([]*Service, 0)
	for _, s := range sm.serviceMap {
		rtn = append(rtn, s)
	}

	sort.Slice(rtn, func(i, j int) bool {
		if rtn[i].GetRemoveN100MaxCpu() > rtn[j].GetRemoveN100MaxCpu() {
			return true
		}
		return false
	})

	return rtn
}

//峰值cpu使用率列表
func (sm *ServiceMap) GetMaxCpuServiceList() []*Service {
	rtn := make([]*Service, 0)
	for _, s := range sm.serviceMap {
		rtn = append(rtn, s)
	}

	sort.Slice(rtn, func(i, j int) bool {
		if rtn[i].GetMaxCpu() > rtn[j].GetMaxCpu() {
			return true
		}
		return false
	})

	return rtn
}

//获取峰值内存使用率列表
func (sm *ServiceMap) GetMaxMemServiceList() []*Service {
	rtn := make([]*Service, 0)
	for _, s := range sm.serviceMap {
		rtn = append(rtn, s)
	}

	sort.Slice(rtn, func(i, j int) bool {
		if rtn[i].GetMaxMem() > rtn[j].GetMaxMem() {
			return true
		}
		return false
	})

	return rtn
}

//获取平均内存使用率列表
func (sm *ServiceMap) GetAverageMemServiceList() []*Service {
	rtn := make([]*Service, 0)
	for _, s := range sm.serviceMap {
		rtn = append(rtn, s)
	}

	sort.Slice(rtn, func(i, j int) bool {
		if rtn[i].GetAverageMem() > rtn[j].GetAverageMem() {
			return true
		}
		return false
	})

	return rtn
}

//获取峰值磁盘使用率列表
func (sm *ServiceMap) GetMaxDiskServiceList() []*Service {
	rtn := make([]*Service, 0)
	for _, s := range sm.serviceMap {
		rtn = append(rtn, s)
	}

	sort.Slice(rtn, func(i, j int) bool {
		if rtn[i].GetMaxDisk() > rtn[j].GetMaxDisk() {
			return true
		}
		return false
	})

	return rtn
}

//获取qps
func (sm *ServiceMap) GetQps() float64 {
	t := sm.endTime.Unix() - sm.startTime.Unix()
	if t == 0 {
		return 0
	}
	return float64(sm.GetCount()) / float64(t)
}

//获取最大qps
func (sm *ServiceMap) GetMaxQps() int {
	//单位秒
	const lang = 3600 * 12
	t := sm.endTime.Unix() - sm.startTime.Unix()
	if t <= 0 {
		return 0
	}

	//分段获取MAXQPS
	start := sm.startTime
	end := sm.endTime
	maxList := make([]int, 0)
	for start.Unix() < end.Unix() {
		glog.Info("分段统计MaxQps: ", start.Format(timeFormat), end.Format(timeFormat))
		if start.Unix()+lang < end.Unix() {
			maxList = append(maxList, sm.getMaxQps(start, end))
		} else {
			maxList = append(maxList, sm.getMaxQps(start, sm.endTime))
		}
		start = start.Add(time.Second * lang)
	}

	//取最大值
	max := 0
	for _, x := range maxList {
		if x > max {
			max = x
		}
	}
	return max
}

func (sm *ServiceMap) getMaxQps(start time.Time, end time.Time) int {
	const sqlMaxQps = "SELECT max(qps) as maxqps FROM " +
		"(SELECT count(\"traceId\") AS qps FROM trace_info " +
		"WHERE time >= %d AND time < %d " +
		"GROUP BY time(1s));"
	sql := fmt.Sprintf(sqlMaxQps, start.UnixNano(), end.UnixNano())
	maxqps, ok := queryOneAndToInt(sql)
	if !ok {
		return 0
	}
	return maxqps
}

//服务平均响应时间排行
func (sm *ServiceMap) GetAverageDurationServiceList() []*Service {
	rtn := make([]*Service, 0)

	for _, s := range sm.serviceMap {
		//屏蔽掉访问量为0的服务
		if s.GetCount() == 0 {
			continue
		}
		rtn = append(rtn, s)
	}

	sort.Slice(rtn, func(i, j int) bool {
		if rtn[i].GetAverageDuration() > rtn[j].GetAverageDuration() {
			return true
		}
		return false
	})

	return rtn
}

//获取最大中位响应时间列表
func (sm *ServiceMap) GetMedianDurationPathList() []*Path {
	rtn := make([]*Path, 0)

	for _, s := range sm.serviceMap {
		pathObj, ok := s.GetMaxMedianDurationPath()
		if ok {
			rtn = append(rtn, pathObj)
		}
	}

	sort.Slice(rtn, func(i, j int) bool {
		if rtn[i].GetMedianDuration() > rtn[j].GetMedianDuration() {
			return true
		}
		return false
	})

	return rtn
}

//获取最大平均响应时间列表
func (sm *ServiceMap) GetAverageDurationPathList() []*Path {
	rtn := make([]*Path, 0)

	for _, s := range sm.serviceMap {
		pathObj, ok := s.GetMaxAverageDurationPath()
		if ok {
			rtn = append(rtn, pathObj)
		}
	}

	sort.Slice(rtn, func(i, j int) bool {
		if rtn[i].GetAverageDuration() > rtn[j].GetAverageDuration() {
			return true
		}
		return false
	})

	return rtn
}

//获取最大响应时间列表
func (sm *ServiceMap) GetMaxDurationTracePointList() []*TracePoint {
	rtn := make([]*TracePoint, 0)
	for _, s := range sm.serviceMap {
		point, ok := s.GetMaxDurationTracePoint()
		if ok {
			rtn = append(rtn, point)
		}
	}

	sort.Slice(rtn, func(i, j int) bool {
		if rtn[i].Duration > rtn[j].Duration {
			return true
		}
		return false
	})

	return rtn
}

//获取访问量
func (sm *ServiceMap) GetCount() int {
	sum := 0
	for _, s := range sm.serviceMap {
		sum += s.GetCount()
	}
	return sum
}

//获取service map
func (sm *ServiceMap) GetServiceMap() map[string]*Service {
	return sm.serviceMap
}

//获取总响应时间
func (sm *ServiceMap) GetSumDuration() time.Duration {
	sum := time.Duration(0)
	for _, s := range sm.serviceMap {
		sum += s.GetSumDuration()
	}
	return sum
}

//获取总平均响应时间
func (sm *ServiceMap) GetAverageDuration() time.Duration {
	count := sm.GetCount()
	if count == 0 {
		return 0
	}
	return sm.GetSumDuration() / time.Duration(count)
}
