package report_form

import (
	"fmt"
	"git.quantgroup.cn/DevOps/enoch/pkg/glog"
	"strings"
	"time"
)

type Service struct {
	name                  string           //服务名称
	startTime             time.Time        //开始时间
	endTime               time.Time        //结束时间
	pathMap               map[string]*Path //path列表
	sumDuration           time.Duration    //总响应时间
	count                 int              //总访问量
	averageDuration       time.Duration    //平均响应时间
	maxMedianPath         *Path            //最大中位响应时间path
	maxAveragePath        *Path            //最大平均响应时间path
	maxDurationTracePoint *TracePoint      //最大响应时间的tracePoint
}

var IgnorePathMap = map[string]struct{}{
	"get":                {},
	"post":               {},
	"put":                {},
	"delete":             {},
	"patch":              {},
	"head":               {},
	"option":             {},
	"/tech/health/check": {},
}

func NewService(name string, startTime time.Time, endTime time.Time) *Service {
	rtn := Service{
		name:      name,
		startTime: startTime,
		endTime:   endTime,
		pathMap:   make(map[string]*Path),
	}

	//初始化path
	pathList := rtn.getPathList()
	for _, path := range pathList {
		//屏蔽掉健康检查url，屏蔽掉命名不合法的url
		pathSplit := strings.Split(path, " ")
		if len(pathSplit) != 2 {
			continue
		}
		if _, ok := IgnorePathMap[strings.ToLower(pathSplit[1])]; ok {
			continue
		}
		pathObj, ok := NewPath(rtn.name, path, rtn.startTime, rtn.endTime)
		if !ok {
			continue
		}
		rtn.pathMap[path] = pathObj
	}

	//初始化访问时间相关的参数
	rtn.initDuration()

	return &rtn
}

func (s *Service) getPathList() []string {
	const sqlGetPathList = `SHOW TAG VALUES FROM trace_info WITH key = "path" WHERE sys_name = '%s';`
	sql := fmt.Sprintf(sqlGetPathList, s.name)
	resp, err := query(sql)
	if err != nil {
		glog.Error("query sql:", err)
		return []string{}
	}

	if len(resp.Results) != 1 {
		return []string{}
	}
	res := resp.Results[0]
	if len(res.Series) != 1 {
		return []string{}
	}
	row := res.Series[0]
	idx, ok := getModRowKeyIdx("value", row)
	if !ok {
		return []string{}
	}

	rtn := make([]string, 0)
	for _, v := range row.Values {
		pathInterface := v[idx]
		path, ok := pathInterface.(string)
		if !ok {
			continue
		}
		rtn = append(rtn, path)
	}

	return rtn
}

//初始化访问时间
func (s *Service) initDuration() {
	sum := time.Duration(0)
	count := 0
	for _, pathObj := range s.pathMap {
		sum += pathObj.GetSumDuration()
		count += pathObj.GetCount()
	}
	s.count = count
	s.sumDuration = sum

	//平均
	if s.count == 0 {
		return
	}
	s.averageDuration = s.sumDuration / time.Duration(s.count)

	if len(s.pathMap) != 0 {
		medianMax := time.Duration(-99999)
		averageMax := time.Duration(-99999)
		maxDuration := time.Duration(-99999)
		for _, pathObj := range s.pathMap {
			if pathObj.GetMedianDuration() > medianMax {
				s.maxMedianPath = pathObj
				medianMax = pathObj.GetMedianDuration()
			}
			if pathObj.GetAverageDuration() > averageMax {
				s.maxAveragePath = pathObj
				averageMax = pathObj.GetAverageDuration()
			}

			point := pathObj.GetMaxDurationTracePoint()
			if point.Duration > maxDuration {
				s.maxDurationTracePoint = &point
				maxDuration = point.Duration
			}
		}
	}
}

//获取总访问时间
func (s *Service) GetSumDuration() time.Duration {
	return s.sumDuration
}

//获取总访问量
func (s *Service) GetCount() int {
	return s.count
}

//获取平均访问时间
func (s *Service) GetAverageDuration() time.Duration {
	return s.averageDuration
}

//获取中位访问时间最大的path
func (s *Service) GetMaxMedianDurationPath() (*Path, bool) {
	return s.maxMedianPath, s.maxMedianPath != nil
}

//获取平均访问时间最大的path
func (s *Service) GetMaxAverageDurationPath() (*Path, bool) {
	return s.maxAveragePath, s.maxAveragePath != nil
}

//获取最大访问时间最大的Point
func (s *Service) GetMaxDurationTracePoint() (*TracePoint, bool) {
	return s.maxDurationTracePoint, s.maxDurationTracePoint != nil
}
