package alarm

import (
	"encoding/json"
	"git.quantgroup.cn/DevOps/enoch/service/log"
	"git.quantgroup.cn/DevOps/enoch/service/util"
	"net/http"
	"strings"
	"sync"
	"time"
)

var SenderWatcher sync.Map

func Sender(target []string, title string, info string, notice Notice) {
	uniqueKey := buildUniqueKey(target)
	if !isExpired(uniqueKey, notice.Interval) { //小于报警周期
		return
	}
	info = uniqueKey + info
	switch notice.Sender {
	case DING_DING:
		senderDingDing(title, info, notice.Receiver)
		break
	case MAIL:
		senderMail(title, info, notice.Receiver)
	default:
		logger.Error.Println("策略配置错误，未匹配到对应的Sender")

	}
}

func senderMail(title string, info string, receiver []string) {
	for _, r := range receiver {
		util.SendEmail(title, info, r)
	}
}

func senderDingDing(title string, info string, receiver [] string) {
	bodyStr := buildDingDingMsg(title, info)

	for _, r := range receiver {
		data := strings.NewReader(string(bodyStr))
		_, err := http.Post(r, "application/json;charset=utf-8", data)
		if err != nil {
			logger.Error.Println(err)
		}
	}
}

type dinDingMsg struct {
	MsgType string `json:"msgtype"`
	Link    link   `json:"link"`
}
type link struct {
	Title      string `json:"title"`
	Text       string `json:"text"`
	MessageUrl string `json:"messageUrl"`
}

func buildDingDingMsg(title string, info string) []byte {
	msg := dinDingMsg{
		MsgType: "link",
		Link: link{
			Title:      title,
			Text:       info,
			MessageUrl: "https://open-doc.dingtalk.com/docs/doc.htm?spm=a219a.7629140.0.0.Rqyvqo&treeId=257&articleId=105735&docType=1",
		},
	}

	msgStr, err := json.Marshal(msg)

	if nil != err {
		logger.Error.Println("无法序列化ding ding msg", err)
	}
	logger.Info.Println("buildDingDingMsg", msgStr)
	return msgStr

}

/**
判断是否符合报警的时间间隔
 */
func isExpired(key string, interval int64) bool {
	now := time.Now().Unix()
	lastTime, hasValue := SenderWatcher.LoadOrStore(key, now)
	logger.Info.Println("---------时间间隔：old:", lastTime, ",new:", now)
	if hasValue { // 存在旧值，判断是否过期
		if now-lastTime.(int64) >= interval { //过期
			return true
		}
		//不过期，重新赋周期内最小值
		SenderWatcher.Store(key, lastTime)
	}
	return true
}

func buildUniqueKey(target []string) string {
	length := len(target)
	str := strings.Builder{}
	for i := length - 1; i >= 0; i-- {
		str.WriteString(target[i])
		str.WriteString(":")
	}
	return str.String()
}

/**
	持久化报警
 */
func persistence() {

}
