package node_check

import (
	"encoding/json"
	"fmt"
	"git.quantgroup.cn/DevOps/enoch/pkg/global"
	"git.quantgroup.cn/DevOps/enoch/pkg/glog"
	"github.com/vrg0/go-common/notify"
	"github.com/vrg0/go-common/registry"
	"strings"
	"sync"
)

var (
	notifyDingDing = newNotifyDingDing()
)

func newNotifyDingDing() *notify.Notify {
	cfgStr := global.Config.GetOrDefault(global.NamespaceApplication, "notify.dingding", `[]`)
	dstList := make([]string, 0)
	_ = json.Unmarshal([]byte(cfgStr), &dstList)
	return notify.New(dstList)
}

func NodeHealthCheckAndNotify() {
	//设置观察者节点
	if err := registry.SetObserver(newWatch()); err != nil {
		glog.Info("设置观察者节点失败", err)
		return
	}
}

//registry观察者，观察服务的状态，当节点挂掉时告警，当服务挂掉时告警
type watch struct {
	serviceMap     map[string]*registry.Service
	serviceMapLock *sync.Mutex
}

func newWatch() *watch {
	return &watch{
		serviceMap:     make(map[string]*registry.Service),
		serviceMapLock: new(sync.Mutex),
	}
}

func (w watch) DeleteService(serviceName string) {
	w.serviceMapLock.Lock()
	defer w.serviceMapLock.Unlock()

	//服务下线
	glog.Info("服务下线：", serviceName)

	//删除服务
	delete(w.serviceMap, serviceName)
}

func (w watch) UpdateNodes(service *registry.Service) {
	w.serviceMapLock.Lock()
	defer w.serviceMapLock.Unlock()

	//状态检查&告警
	w.statusCheckAndNotify(service)

	//更新本地缓存的节点状态
	w.serviceMap[service.Name] = service
}

func (w watch) statusCheckAndNotify(service *registry.Service) {
	//服务信息初始化
	//服务不存在
	if _, ok := w.serviceMap[service.Name]; !ok {
		glog.Info("服务信息初始化：", service.Name)
		return
	}
	//服务存在，但所有节点都是critical 或 没有节点
	if service, ok := w.serviceMap[service.Name]; ok {
		allNodeCritical := true
		for _, node := range service.NodeMap {
			if node.Status == registry.Passing {
				allNodeCritical = false
				break
			}
		}
		if allNodeCritical {
			glog.Info("服务信息初始化：", service.Name)
			return
		}
	}

	//服务挂掉告警（服务的全部节点都挂掉）
	if len(service.NodeMap) != 0 {
		serviceUp := false
		for _, node := range service.NodeMap {
			if node.Status == registry.Passing {
				serviceUp = true
				break
			}
		}

		if !serviceUp {
			sb := new(strings.Builder)
			sb.WriteString(fmt.Sprintf("服务健康状态异常：%s", service.Name))
			for _, node := range service.NodeMap {
				sb.WriteString(fmt.Sprintf(" %s:%s", node.Id, node.Status))
			}
			sb.WriteString("\n")
			glog.Warn(sb.String())
			notifyDingDing.SendText(sb.String())
			return
		}
	}

	//节点挂掉告警（服务的部分节点挂掉）
	for _, node := range service.NodeMap {
		//如果节点状态从passing变为其他，则告警
		if oldService, ok := w.serviceMap[node.ServiceName]; ok {
			if oldNode, ok := oldService.NodeMap[node.Id]; ok {
				if oldNode.Status == registry.Passing && node.Status != registry.Passing {
					s := fmt.Sprintf("服务节点健康状态异常：%s %s:%s", service.Name, node.Id, node.Status)
					glog.Warn(s)
					notifyDingDing.SendText(s)
				}
			}
		}
	}
}
